# Python обвязка для алгоритма Kyber

Данная реализация предоставляет Python-интерфейс для постквантового алгоритма обмена ключами Kyber с использованием CFFI для взаимодействия с оригинальной C-реализацией.

## Требования

- Python 3.12 или 3.13
- Компилятор C (для Windows: MSVC)
- Библиотека CFFI (`pip install cffi`)

## Структура проекта

- `kyber-build.py` - скрипт для сборки C-библиотеки с использованием CFFI
- `kyber.py` - Python-обертка для взаимодействия с C-функциями Kyber
- `example.py` - пример использования библиотеки
- `benchmark.py` - скрипт для замера производительности библиотеки
- `ref/` - оригинальная C-реализация Kyber

## Установка

1. Клонируйте репозиторий
2. Установите зависимости:
   ```
   pip install cffi
   ```
3. Скомпилируйте библиотеку:
   ```
   python kyber-build.py
   ```

По умолчанию компилируется Kyber768 (KYBER_K=3). Для компиляции другого уровня безопасности измените вызов `build_kyber()` в конце файла `kyber-build.py`.

## Использование

### Пример использования

```python
from kyber import Kyber768

# Генерация ключей
pk, sk = Kyber768.keygen()

# Инкапсуляция (создание общего секрета)
ct, ss_alice = Kyber768.encapsulate(pk)

# Декапсуляция (восстановление общего секрета)
ss_bob = Kyber768.decapsulate(ct, sk)

# Проверка совпадения секретов
assert ss_alice == ss_bob
```

### Запуск примера

```bash
python example.py
```

### Выполнение бенчмарка

```bash
python benchmark.py
```

## Уровни безопасности

- **Kyber512** (KYBER_K=2): соответствует уровню безопасности AES-128
- **Kyber768** (KYBER_K=3): соответствует уровню безопасности AES-192
- **Kyber1024** (KYBER_K=4): соответствует уровню безопасности AES-256

## Особенности реализации

Данная обвязка использует CFFI для прямого взаимодействия с C-кодом без изменения оригинальной реализации. Это обеспечивает максимальную совместимость и производительность. Реализация включает все необходимые функции для работы с Kyber:

- Генерация ключей (keygen)
- Инкапсуляция (encapsulate)
- Декапсуляция (decapsulate)

