"""Module for handling AutoPkg recipe processing in cloud-autopkg-runner.

This module defines classes and functions for representing, parsing,
and processing AutoPkg recipes. It provides tools for extracting
information from recipes, generating lists of recipes, and performing
other recipe-related operations.

Key classes:
- `Recipe`: Represents an AutoPkg recipe and provides methods for accessing
  recipe metadata, parsing the recipe contents, running the recipe, and
  managing trust information.
"""

import asyncio
import plistlib
from datetime import datetime, timezone
from enum import Enum, auto
from pathlib import Path
from typing import Any, TypedDict

import yaml

from cloud_autopkg_runner import (
    AutoPkgPrefs,
    Settings,
    file_utils,
    logging_config,
    metadata_cache,
    recipe_report,
    shell,
)
from cloud_autopkg_runner.exceptions import (
    InvalidPlistContents,
    InvalidYamlContents,
    RecipeFormatException,
    RecipeInputException,
)
from cloud_autopkg_runner.metadata_cache import DownloadMetadata, RecipeCache
from cloud_autopkg_runner.recipe_report import ConsolidatedReport


class RecipeContents(TypedDict):
    """Represents the structure of a recipe's contents.

    This `TypedDict` defines the expected structure of the parsed contents of an
    AutoPkg recipe file, whether it's in PLIST or YAML format. It includes all
    standard top-level keys found in AutoPkg recipes.

    Attributes:
        Description: A brief description of what the recipe does. Optional.
        Identifier: A unique string identifier for the recipe (e.g.,
            "com.github.autopkg.pkg.GoogleChrome").
        Input: A `dict` of input variables that can be customized for the recipe.
        MinimumVersion: The minimum AutoPkg version required to run this recipe.
            Optional.
        ParentRecipe: The identifier of the parent recipe if this is an override
            or a child recipe. Optional.
        Process: A `list` of `dict`s, where each `dict` defines a single
            processor step in the recipe's execution workflow.
    """

    Description: str | None
    Identifier: str
    Input: dict[str, Any]
    MinimumVersion: str | None
    ParentRecipe: str | None
    Process: list[dict[str, Any]]


class RecipeFormat(Enum):
    """Enumerates the supported recipe file formats.

    This `Enum` defines the possible file formats for AutoPkg recipes,
    which are typically either YAML or Apple Property List (PLIST) format.
    These values are used to determine how a recipe file's contents should be
    parsed.

    Values:
        YAML: Represents a recipe stored in YAML format (e.g., `.recipe.yaml`).
        PLIST: Represents a recipe stored in Apple Property List format (e.g.,
            `.recipe` or `.recipe.plist`).
    """

    YAML = "yaml"
    PLIST = "plist"


class Recipe:
    """Represents an AutoPkg recipe.

    This class encapsulates an AutoPkg recipe, providing functionality to
    read its contents, access metadata, run the recipe via AutoPkg CLI,
    and manage its trust information and reporting. It acts as an abstraction
    layer over the raw recipe file.

    Attributes:
        _logger: An instance of `logging.Logger` for logging events within the class.
        _autopkg_prefs: An `AutoPkgPrefs` object containing AutoPkg's global
            preferences.
        _settings: A `Settings` object containing application-wide settings.
        _name: The base name of the recipe file (e.g., "GoogleChrome.pkg.recipe").
        _path: The `Path` object pointing to the recipe file.
        _format: A `RecipeFormat` enum value indicating the file format (YAML/PLIST).
        _contents: A `RecipeContents` TypedDict holding the parsed content of the
            recipe.
        _trusted: A `TrustInfoVerificationState` enum value representing the current
            trust status of the recipe, initialized to `UNTESTED`.
        _result: A `RecipeReport` object used to store and manage the output
            report generated by AutoPkg.
    """

    def __init__(
        self,
        recipe_path: Path,
        report_dir: Path | None = None,
        autopkg_prefs: AutoPkgPrefs | None = None,
    ) -> None:
        """Initialize a Recipe object.

        Args:
            recipe_path: The `Path` object to the recipe file. This path must
                point to an existing and valid AutoPkg recipe file.
            report_dir: An optional `Path` object to the directory where AutoPkg
                reports should be stored. If `None`, the `report_dir` from
                `Settings` will be used. A unique filename for the report will
                be generated within this directory.
            autopkg_prefs: An optional `AutoPkgPrefs` instance containing AutoPkg's
                preferences. If `None`, a default `AutoPkgPrefs` instance will
                be created.

        Raises:
            RecipeFormatException: If the recipe file's extension is not
                recognized as a valid AutoPkg recipe format.
            InvalidPlistContents: If the recipe is a PLIST and its contents are
                malformed.
            InvalidYamlContents: If the recipe is a YAML and its contents are
                malformed.
        """
        self._logger = logging_config.get_logger(__name__)
        self._autopkg_prefs = autopkg_prefs if autopkg_prefs else AutoPkgPrefs()
        self._settings = Settings()

        self._name: str = recipe_path.name
        self._path: Path = recipe_path
        self._format: RecipeFormat = self.format()
        self._contents: RecipeContents = self._get_contents()
        self._trusted: TrustInfoVerificationState = TrustInfoVerificationState.UNTESTED

        now_str: str = datetime.now(tz=timezone.utc).strftime("%y%m%d_%H%M")
        if report_dir is None:
            report_dir = self._settings.report_dir
        report_path: Path = report_dir / f"report_{now_str}_{self.name}.plist"

        counter: int = 1
        original_report_path: Path = report_path
        while report_path.exists():
            # If a report already exists, append a counter to ensure uniqueness
            report_path = original_report_path.with_stem(
                f"{original_report_path.stem}_{counter}"
            )
            counter += 1

        # Ensure the report directory exists
        report_path.parent.mkdir(parents=True, exist_ok=True)
        self._result: recipe_report.RecipeReport = recipe_report.RecipeReport(
            report_path
        )

    @property
    def contents(self) -> RecipeContents:
        """Returns the recipe's contents as a dictionary.

        This property provides read-only access to the parsed contents of the
        AutoPkg recipe file, represented as a `RecipeContents` TypedDict.

        Returns:
            The recipe's contents as a `RecipeContents` TypedDict.
        """
        return self._contents

    @property
    def description(self) -> str:
        """Returns the recipe's description.

        This property provides the value of the "Description" key from the
        recipe's parsed contents. If the key is not present or its value is
        `None`, an empty string is returned.

        Returns:
            The recipe's description as a `str`. Returns an empty string
            if the recipe does not have a description.
        """
        return self._contents.get("Description") or ""

    @property
    def identifier(self) -> str:
        """Returns the recipe's identifier.

        This property provides the unique identifier of the recipe as specified
        in its contents. This is a mandatory field in AutoPkg recipes.

        Returns:
            The recipe's identifier as a `str`.
        """
        return self._contents["Identifier"]

    @property
    def input(self) -> dict[str, Any]:
        """Returns the recipe's input dictionary.

        This property provides the `dict` of input variables defined within
        the recipe's contents. These variables can be used to customize the
        recipe's behavior during execution.

        Returns:
            The recipe's input dictionary, containing the input variables
            used by the recipe.
        """
        return self._contents["Input"]

    @property
    def input_name(self) -> str:
        """Returns the recipe's `NAME` input variable.

        This property attempts to retrieve the value associated with the
        `"NAME"` key within the recipe's "Input" dictionary. This is a common
        input variable used in AutoPkg recipes for naming generated packages
        or items.

        Returns:
            The recipe's `NAME` input variable as a `str`.

        Raises:
            RecipeInputException: If the recipe's "Input" dictionary does not
                contain a "NAME" key, indicating a missing required input.
        """
        try:
            return str(self._contents["Input"]["NAME"])
        except KeyError as exc:
            raise RecipeInputException(self._path) from exc

    @property
    def minimum_version(self) -> str:
        """Returns the recipe's minimum AutoPkg version.

        This property provides the value of the "MinimumVersion" key from the
        recipe's parsed contents. If the key is not present or its value is
        `None`, an empty string is returned. This indicates the earliest AutoPkg
        version compatible with the recipe.

        Returns:
            The recipe's minimum version as a `str`. Returns an empty string
            if the recipe does not have a minimum version specified.
        """
        return self._contents.get("MinimumVersion") or ""

    @property
    def name(self) -> str:
        """Returns the recipe's filename.

        This property provides the full filename of the recipe, including its
        extension (e.g., "GoogleChrome.pkg.recipe").

        Returns:
            The recipe's filename as a `str`.
        """
        return self._path.name

    @property
    def parent_recipe(self) -> str:
        """Returns the recipe's parent recipe identifier.

        This property provides the identifier of the parent recipe if this
        recipe is an override or inherits from another recipe. If the "ParentRecipe"
        key is not present or its value is `None`, an empty string is returned.

        Returns:
            The recipe's parent recipe identifier as a `str`. Returns an empty
            string if the recipe does not have a parent recipe.
        """
        return self._contents.get("ParentRecipe") or ""

    @property
    def process(self) -> list[dict[str, Any]]:
        """Returns the recipe's process array.

        This property provides the `list` of `dict`s, where each `dict` represents
        a single processor step that AutoPkg will execute when running this recipe.
        If the "Process" key is not present or its value is `None`, an empty
        list is returned.

        Returns:
            The recipe's process array, which is a `list` of `dict`s
            defining the steps in the recipe's processing workflow.
        """
        return self._contents.get("Process") or []

    async def _autopkg_run_cmd(self, *, check: bool = False) -> list[str]:
        """Constructs the command-line arguments for running AutoPkg.

        This private asynchronous method generates the complete `autopkg run`
        command, including arguments for the specific recipe, report output,
        AutoPkg preferences file, and optional pre/post-processors. It also
        conditionally adds the `--check` flag and verbosity settings.

        Args:
            check: A boolean flag. If `True`, the `--check` argument will be
                added to the `autopkg run` command, performing a dry run to
                check for updates. Defaults to `False`.

        Returns:
            A `list` of strings representing the complete command-line arguments
            suitable for execution (e.g., by `subprocess.run`).
        """
        # Ensure preferences are written to a temporary file for AutoPkg to use
        prefs_file_path: Path = await self._autopkg_prefs.to_json_file(indent=2)

        cmd: list[str] = [
            "/usr/local/bin/autopkg",
            "run",
            self.name,
            f"--report-plist={self._result.file_path()}",
            f"--prefs={prefs_file_path}",
        ]

        cmd.extend([f"--preprocessor={item}" for item in self._settings.pre_processors])
        cmd.extend(
            [f"--postprocessor={item}" for item in self._settings.post_processors]
        )

        if self._settings.verbosity_int(-1) > 0:
            cmd.append(self._settings.verbosity_str(-1))

        if check:
            cmd.append("--check")

        return cmd

    async def _create_placeholder_cache_files(self) -> None:
        """Creates placeholder cache files for the recipe.

        This asynchronous private method calls the `file_utils.create_placeholder_files`
        utility function. It creates empty placeholder files in AutoPkg's cache
        directory, which simulates an existing cache. This can be crucial for
        recipes that expect certain files to exist or to optimize runs by
        avoiding unnecessary downloads.
        """
        # Use hasattr and a flag to ensure this is run only once per recipe instance
        if (
            not hasattr(self, "_placeholder_files_created")
            or self._placeholder_files_created is not True
        ):
            await file_utils.create_placeholder_files([self.name])
            self._placeholder_files_created = True

    @staticmethod
    def _extract_download_paths(download_items: list[dict[str, Any]]) -> list[str]:
        """Extracts 'download_path' values from a list of downloaded items.

        This static private helper function processes a list of dictionaries,
        each representing a downloaded item as reported by AutoPkg. It
        iterates through the `downloaded_items` structure and extracts the
        `"download_path"` string from each entry. This is useful for
        identifying the actual files that were downloaded during a recipe run.

        Args:
            download_items: A `list` of `dict`s. Each `dict` is expected to have
                a structure similar to
                `{'downloaded_items': [{'download_path': 'path_to_file'}]}`.

        Returns:
            A `list` of strings, where each string is the `download_path` value
            extracted from the `downloaded_items` list of each input dictionary.
            Returns an empty list if the input `download_items` is empty, or if
            any of the intermediate keys (`"downloaded_items"`, `"download_path"`)
            are missing, or if the `downloaded_items` list itself is empty.
        """
        if not download_items:
            return []

        return [item["download_path"] for item in download_items]

    def _get_contents(self) -> RecipeContents:
        """Read and parse the recipe file.

        This private method reads the entire content of the recipe file from
        `self._path` and then dispatches to either `_get_contents_yaml` or
        `_get_contents_plist` based on the determined `self._format`. It
        ensures the recipe's contents are loaded into the `_contents` attribute.

        Returns:
            A `RecipeContents` TypedDict containing the recipe's parsed contents.

        Raises:
            InvalidPlistContents: If the recipe is a PLIST and its contents are
                malformed.
            InvalidYamlContents: If the recipe is a YAML and its contents are
                malformed.
        """
        file_contents: str = self._path.read_text()

        if self._format == RecipeFormat.YAML:
            return self._get_contents_yaml(file_contents)
        return self._get_contents_plist(file_contents)

    def _get_contents_plist(self, file_contents: str) -> RecipeContents:
        """Parse a recipe in PLIST format.

        This private method takes the raw string content of a recipe file
        and attempts to parse it as an Apple Property List (PLIST) using
        `plistlib.loads()`. It's designed to handle both XML and binary
        PLIST formats.

        Args:
            file_contents: The recipe file contents as a `str`.

        Returns:
            A `RecipeContents` TypedDict containing the recipe's parsed contents.

        Raises:
            InvalidPlistContents: If `plistlib.loads()` encounters an
                `plistlib.InvalidFileException`, indicating that the file
                content is not a valid PLIST.
        """
        try:
            return plistlib.loads(file_contents.encode())
        except plistlib.InvalidFileException as exc:
            raise InvalidPlistContents(self._path) from exc

    def _get_contents_yaml(self, file_contents: str) -> RecipeContents:
        """Parse a recipe in YAML format.

        This private method takes the raw string content of a recipe file
        and attempts to parse it as a YAML document using `yaml.safe_load()`.
        This is used for recipes with a `.yaml` extension.

        Args:
            file_contents: The recipe file contents as a `str`.

        Returns:
            A `RecipeContents` TypedDict containing the recipe's parsed contents.

        Raises:
            InvalidYamlContents: If `yaml.safe_load()` encounters a `yaml.YAMLError`,
                indicating that the file content is not valid YAML.
        """
        try:
            return yaml.safe_load(file_contents)
        except yaml.YAMLError as exc:
            raise InvalidYamlContents(self._path) from exc

    async def _get_metadata(self, download_items: list[dict[str, str]]) -> RecipeCache:
        """Retrieves metadata for a list of downloaded items.

        This private asynchronous method processes a list of dictionaries,
        each representing information about a downloaded item from an AutoPkg
        run report. It asynchronously extracts relevant metadata (ETag, file size,
        last modified date) for each item by calling `_get_metadata_for_item`.
        The collected metadata is then compiled into a `RecipeCache` dictionary,
        including a timestamp of when the metadata was captured.

        Args:
            download_items: A `list` of `dict`s, where each dictionary
                contains information about a downloaded item, including its
                path (e.g., from `_extract_download_paths`).

        Returns:
            A `RecipeCache` dictionary containing a "timestamp" (`str`) and
            a `list` of `DownloadMetadata` dictionaries, one for each
            downloaded item for which metadata was successfully retrieved.
        """
        download_paths: list[str] = self._extract_download_paths(download_items)
        metadata_list: list[DownloadMetadata] = await asyncio.gather(
            *[self._get_metadata_for_item(path) for path in download_paths]
        )

        return {
            "timestamp": str(datetime.now(tz=timezone.utc)),
            "metadata": metadata_list,
        }

    @staticmethod
    async def _get_metadata_for_item(downloaded_item: str) -> DownloadMetadata:
        """Retrieves metadata for a single downloaded item.

        This static asynchronous private method takes the `str` path to a
        downloaded file and asynchronously retrieves its ETag, file size,
        and last modified date. It uses `file_utils.get_file_metadata` for
        extended attributes (ETag, last modified) and `file_utils.get_file_size`
        for the file's size. These operations are run concurrently using
        `asyncio.gather` for efficiency.

        Args:
            downloaded_item: The `str` path to the downloaded item.

        Returns:
            A `DownloadMetadata` dictionary containing the ETag (`str` or `None`),
            file size (`int` or `None`), last modified date (`str` or `None`),
            and the original file path (`str`) of the downloaded item.
        """
        downloaded_item_path: Path = Path(downloaded_item)

        etag_task = file_utils.get_file_metadata(
            downloaded_item_path, "com.github.autopkg.etag"
        )
        file_size_task = file_utils.get_file_size(downloaded_item_path)
        last_modified_task = file_utils.get_file_metadata(
            downloaded_item_path, "com.github.autopkg.last-modified"
        )

        # Run the tasks concurrently and await all of them to finish
        etag, file_size, last_modified = await asyncio.gather(
            etag_task, file_size_task, last_modified_task
        )

        return {
            "etag": etag,
            "file_size": file_size,
            "last_modified": last_modified,
            "file_path": downloaded_item,
        }

    def compile_report(self) -> ConsolidatedReport:
        """Compiles a consolidated report from the recipe report file.

        This method first ensures that the internal `_result` (which is a
        `RecipeReport` object) has its contents refreshed by re-reading the
        report file generated by AutoPkg. It then calls the `consolidate_report()`
        method on `_result` to aggregate information about failed items,
        downloaded items, package builds, and Munki imports into a single,
        easy-to-consume format.

        Returns:
            A `ConsolidatedReport` object containing summarized information
            about the AutoPkg recipe run.
        """
        self._result.refresh_contents()
        return self._result.consolidate_report()

    def format(self) -> RecipeFormat:
        """Determine the recipe's format based on its file extension.

        This method inspects the file extension of the recipe path (`self._path`)
        to determine whether it is a YAML-formatted recipe (e.g., `.yaml`) or
        a PLIST-formatted recipe (e.g., `.plist`, `.recipe`).

        Returns:
            A `RecipeFormat` enum value (either `RecipeFormat.YAML` or
            `RecipeFormat.PLIST`) indicating the identified format.

        Raises:
            RecipeFormatException: If the file extension is not recognized
                as a valid AutoPkg recipe format (i.e., not `.yaml`, `.plist`, or
                `.recipe`).
        """
        if self._path.suffix == ".yaml":
            return RecipeFormat.YAML
        if self._path.suffix in {".plist", ".recipe"}:
            return RecipeFormat.PLIST
        raise RecipeFormatException(self._path.suffix)

    async def run(self) -> ConsolidatedReport:
        """Runs the recipe and saves metadata.

        This asynchronous method orchestrates the execution of an AutoPkg recipe.
        It first performs a "check phase" to determine if there are any updates
        available. If updates lead to downloaded items, it then extracts metadata from
        these downloaded files, saves this metadata to the configured cache using the
        `metadata_cache` plugin, and finally performs a full run of the recipe. If no
        new downloads are detected during the check phase, the full run is skipped, and
        the report from the check phase is returned.

        Returns:
            A `ConsolidatedReport` object containing the results of the
            recipe run, encompassing either the check phase or the full run.
        """
        # First, run the check phase to see if there are any updates
        output: ConsolidatedReport = await self.run_check_phase()

        # If the check phase indicates new downloads, process metadata and run full
        if output["downloaded_items"]:
            self._logger.info(
                "New downloads detected for %s. Running full recipe.", self.name
            )
            metadata: RecipeCache = await self._get_metadata(output["downloaded_items"])
            metadata_cache_manager = metadata_cache.get_cache_plugin()
            await metadata_cache_manager.set_item(self.name, metadata)

            return await self.run_full()

        self._logger.info(
            "No new downloads for %s. Skipping full recipe run.", self.name
        )
        return output

    async def run_check_phase(self) -> ConsolidatedReport:
        """Performs the check phase of the recipe.

        This asynchronous method involves invoking the `autopkg` command-line
        tool with the `--check` flag. This action determines if there are any
        updates available for the software managed by the recipe without fully
        processing package builds. It ensures placeholder cache files are present.

        Returns:
            A `ConsolidatedReport` object containing the results and status
            information gathered during the check phase. This includes details
            on any potential downloads that would occur in a full run.
        """
        self._logger.debug("Performing Check Phase on %s...", self.name)

        await self._create_placeholder_cache_files()

        returncode, _stdout, stderr = await shell.run_cmd(
            await self._autopkg_run_cmd(check=True), check=False
        )

        if returncode != 0:
            if not stderr:
                stderr = "<Unknown Error>"
            self._logger.error(
                "An error occurred while running 'check phase' on %s: %s",
                self.name,
                stderr,
            )

        self._logger.debug("Check phase for %s completed.", self.name)
        return self.compile_report()

    async def run_full(
        self,
    ) -> ConsolidatedReport:
        """Performs an `autopkg run` of the recipe.

        This asynchronous method executes the full AutoPkg recipe. This involves
        all process steps defined in the recipe, including downloading files,
        building packages, and importing items into Munki (if configured),
        depending on the recipe's definition. It ensures placeholder cache files
        are present before execution.

        Returns:
            A `ConsolidatedReport` object containing the comprehensive results
            of the full recipe run, including actual downloads, package builds,
            and Munki imports.
        """
        self._logger.debug("Performing AutoPkg Run on %s...", self.name)

        await self._create_placeholder_cache_files()

        returncode, _stdout, stderr = await shell.run_cmd(
            await self._autopkg_run_cmd(check=False), check=False
        )

        if returncode != 0:
            if not stderr:
                stderr = "<Unknown Error>"
            self._logger.error(
                "An error occurred while running %s: %s", self.name, stderr
            )

        self._logger.debug("Full run for %s completed.", self.name)
        return self.compile_report()

    async def update_trust_info(self) -> bool:
        """Update trust info for the recipe.

        This asynchronous method invokes the `autopkg update-trust-info` command
        for the specific recipe. This command is used to update the trust information
        (signatures, checksums) for a recipe, particularly for overrides, to ensure
        the integrity and authenticity of its parent recipes and processors.

        Returns:
            `True` if the trust information was successfully updated (AutoPkg
            returns a 0 exit code), `False` otherwise. The internal
            `_trusted` state is reset to `UNTESTED` after an update attempt.
        """
        self._logger.debug("Updating trust info for %s...", self.name)

        prefs_file_path: Path = await self._autopkg_prefs.to_json_file(indent=2)

        cmd: list[str] = [
            "/usr/local/bin/autopkg",
            "update-trust-info",
            self.name,
            f"--override-dir={self._path.parent}",
            f"--prefs={prefs_file_path}",
        ]

        returncode, stdout, _stderr = await shell.run_cmd(cmd)

        self._logger.info(stdout)
        self._trusted = TrustInfoVerificationState.UNTESTED

        if returncode == 0:
            self._logger.info("Trust info update for %s successful.", self.name)
            return True

        self._logger.warning("Trust info update for %s failed.", self.name)
        return False

    async def verify_trust_info(self) -> bool:
        """Verify the trust info.

        This asynchronous method verifies the trust information for the recipe
        by invoking the `autopkg verify-trust-info` command. It first checks
        if the trust information has already been tested. If not, it executes
        the command and updates the internal `_trusted` state based on AutoPkg's
        exit code. This ensures that only trusted recipes are processed.

        Returns:
            `True` if the trust info is verified as trusted, `False` if it is
            untrusted or if the verification process fails. The returned boolean
            corresponds to the `bool` value of the `TrustInfoVerificationState` enum.
        """
        if self._trusted == TrustInfoVerificationState.UNTESTED:
            self._logger.debug("Verifying trust info for %s...", self.name)

            prefs_file_path: Path = await self._autopkg_prefs.to_json_file(indent=2)

            cmd: list[str] = [
                "/usr/local/bin/autopkg",
                "verify-trust-info",
                self.name,
                f"--override-dir={self._path.parent}",
                f"--prefs={prefs_file_path}",
            ]

            if self._settings.verbosity_int() > 0:
                cmd.append(self._settings.verbosity_str())

            returncode, _stdout, _stderr = await shell.run_cmd(cmd, check=False)

            if returncode == 0:
                self._logger.info(
                    "Trust info verification for %s successful.", self.name
                )
                self._trusted = TrustInfoVerificationState.TRUSTED
            else:
                self._logger.warning(
                    "Trust info verification for %s failed.", self.name
                )
                self._trusted = TrustInfoVerificationState.FAILED

        return self._trusted.value


class TrustInfoVerificationState(Enum):
    """Enum for whether trust info is tested, successful, or failed.

    This `Enum` represents the possible states of trust information verification
    for an AutoPkg recipe. It's used internally by the `Recipe` class to track
    the verification status and avoid redundant checks. Each state also has an
    associated boolean value for convenience.

    Values:
        UNTESTED: The trust information has not yet been verified.
        FAILED: The trust information verification process failed, or the recipe
            was found to be untrusted. Corresponds to `False`.
        TRUSTED: The trust information was successfully verified, and the recipe
            is considered trusted. Corresponds to `True`.
    """

    UNTESTED = auto()
    FAILED = False
    TRUSTED = True
