from enum import Enum
from transitions.core import EventData
from .state_machine import StateMachine
from .position import Position


class States(Enum):

    created = "created"
    pending = "pending"
    matched = "matched"

    waiting_pickup = "waiting_pickup"
    pickup = "pickup"

    waiting_dropoff = "waiting_dropoff"
    dropoff = "dropoff"

    # The last state _after_ dropoff
    complete = "complete"

    # by dispatcher
    # dispatcher_canceled = "dispatcher_canceled"
    # customer_canceled = "customer_canceled"

    expired = "expired"


state_transitions = [
    # format: [trigger, source, destination]
    ["set_pending", States.created, States.pending],
    ["set_matched", States.pending, States.matched],
    ["set_waiting_pickup", States.matched, States.waiting_pickup],
    # move to pickup or pickup if at the same location as vehicle
    ["set_pickup", [States.waiting_pickup, States.matched], States.pickup],
    ["set_waiting_dropoff", States.pickup, States.waiting_dropoff],
    ["set_dropoff", [States.waiting_dropoff, States.pickup], States.dropoff],
    ["set_complete", States.dropoff, States.complete],
    # booking can't be canceled after pickup
    # [
    #     "set_dispatcher_canceled",
    #     [States.pending, States.matched, States.waiting_pickup],
    #     States.dispatcher_canceled,
    # ],
    # [
    #     "set_customer_canceled",
    #     [States.pending, States.matched, States.waiting_pickup],
    #     States.customer_canceled,
    # ],
    ["set_expired", States.pending, States.expired],
]


class Booking(StateMachine):
    """
    Functions autogenerated by StateMachine

    booking.is_<STATE>()
    booking.set_<STATE>()

    booking.is_matched()
    booking.set_matched()

    booking.is_pickup()
    booking.set_pickup()

    ...
    """

    def __init__(
        self,
        clock,
        pickup: Position,
        dropoff: Position,
        seats: int = 4,
        preferences: dict = None,
        booking_id: str = None,
    ):

        states = [s for s in States]
        super().__init__(
            clock, state_transitions, states, States.created, object_id=booking_id
        )

        self.pickup = pickup
        self.dropoff = dropoff
        self.seats = seats
        self.preferences = preferences

        self.set_pending()

    def on_state_changed(self, event: EventData):
        if event.transition.dest == States.pending.value:
            # TODO: check if kwargs already have keys
            event.kwargs["position"] = self.pickup.to_dict()
            event.kwargs["dropoff"] = self.dropoff.to_dict()

        if event.transition.dest in (
            States.pending.value,
            States.matched.value,
            States.pickup.value,
            States.expired.value,
            States.waiting_dropoff.value,
            States.waiting_pickup.value,
            States.expired.value,
        ):

            # NOTE: created position can be different from pickup position
            # but this is not supported at the moment

            event.kwargs["position"] = self.pickup.to_dict()

        if event.transition.dest in (States.dropoff.value, States.complete.value):
            # NOTE: pickup
            event.kwargs["position"] = self.dropoff.to_dict()

        super().on_state_changed(event)
