# Copyright 2013 Camptocamp SA
# License AGPL-3.0 or later (https://www.gnu.org/licenses/agpl).
from odoo import api, fields, models
from odoo.exceptions import UserError


class StockQuantPackage(models.Model):
    _inherit = "stock.quant.package"

    postlogistics_manual_cod_amount = fields.Float(
        "PostLogistics Cash On Delivery Amount",
        help="If the cash on delivery amount for this package is different "
        "than the total of the sales order, write the amount there.",
    )
    parcel_tracking = fields.Char()
    package_carrier_type = fields.Selection(
        related="package_type_id.package_carrier_type",
        string="Packaging's Carrier",
    )

    @api.returns("stock.picking")
    def _get_origin_pickings(self):
        self.ensure_one()
        move_line_model = self.env["stock.move.line"]
        move_line = move_line_model.search([("package_id", "=", self.id)])
        return move_line.mapped("picking_id.sale_id.picking_ids")

    def postlogistics_cod_amount(self):
        """Return the PostLogistics Cash on Delivery amount of a package

        If we have only 1 package which delivers the whole sales order
        we use the total amount of the sales order.

        Otherwise we don't know the value of each package so we raise an
        error and the user has to write the prices on the packages.

        When a price is manually set on a package, it's always used as the
        cash on delivery amount.
        """
        self.ensure_one()
        if self.postlogistics_manual_cod_amount:
            return self.postlogistics_manual_cod_amount

        pickings = self._get_origin_pickings()
        if len(pickings) > 1:
            raise UserError(
                self.env._(
                    "The cash on delivery amount must be manually specified "
                    "on the packages when a sales order is delivered "
                    "in several delivery orders."
                )
            )

        order = pickings.sale_id
        if not order:
            return 0.0
        if len(order) > 1:
            raise UserError(
                self.env._(
                    "The cash on delivery amount must be manually specified "
                    "on the packages when a package contains products "
                    "from different sales orders."
                )
            )

        return order.amount_total
