"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const kinesis_generated_1 = require("./kinesis.generated");
/**
 * Represents a Kinesis Stream.
 *
 * Streams can be either defined within this stack:
 *
 *   new Stream(this, 'MyStream', { props });
 *
 * Or imported from an existing stream:
 *
 *   Stream.import(this, 'MyImportedStream', { streamArn: ... });
 *
 * You can also export a stream and import it into another stack:
 *
 *   const ref = myStream.export();
 *   Stream.import(this, 'MyImportedStream', ref);
 *
 */
class StreamBase extends core_1.Resource {
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, 'kinesis:DescribeStream', 'kinesis:GetRecords', 'kinesis:GetShardIterator');
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, 'kinesis:DescribeStream', 'kinesis:PutRecord', 'kinesis:PutRecords');
        if (this.encryptionKey) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        return ret;
    }
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, 'kinesis:DescribeStream', 'kinesis:GetRecords', 'kinesis:GetShardIterator', 'kinesis:PutRecord', 'kinesis:PutRecords');
        if (this.encryptionKey) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        return ret;
    }
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.streamArn],
            scope: this,
        });
    }
}
/**
 * A Kinesis stream. Can be encrypted with a KMS key.
 */
class Stream extends StreamBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.streamName,
        });
        const shardCount = props.shardCount || 1;
        const retentionPeriodHours = props.retentionPeriodHours || 24;
        if (retentionPeriodHours < 24 && retentionPeriodHours > 168) {
            throw new Error("retentionPeriodHours must be between 24 and 168 hours");
        }
        const { streamEncryption, encryptionKey } = this.parseEncryption(props);
        this.stream = new kinesis_generated_1.CfnStream(this, "Resource", {
            name: this.physicalName,
            retentionPeriodHours,
            shardCount,
            streamEncryption
        });
        this.streamArn = this.getResourceArnAttribute(this.stream.attrArn, {
            service: 'kinesis',
            resource: 'stream',
            resourceName: this.physicalName,
        });
        this.streamName = this.getResourceNameAttribute(this.stream.ref);
        this.encryptionKey = encryptionKey;
    }
    static fromStreamArn(scope, id, streamArn) {
        return Stream.fromStreamAttributes(scope, id, { streamArn });
    }
    /**
     * Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties
     */
    static fromStreamAttributes(scope, id, attrs) {
        class Import extends StreamBase {
            constructor() {
                super(...arguments);
                this.streamArn = attrs.streamArn;
                this.streamName = core_1.Stack.of(scope).parseArn(attrs.streamArn).resourceName;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default to unencrypted.
        const encryptionType = props.encryption || StreamEncryption.UNENCRYPTED;
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== StreamEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === StreamEncryption.UNENCRYPTED) {
            return { streamEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === StreamEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`
            });
            const streamEncryption = {
                encryptionType: 'KMS',
                keyId: encryptionKey.keyArn
            };
            return { encryptionKey, streamEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Stream = Stream;
/**
 * What kind of server-side encryption to apply to this stream
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Records in the stream are not encrypted.
     */
    StreamEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    StreamEncryption["KMS"] = "KMS";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
//# sourceMappingURL=data:application/json;base64,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