"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default stream'(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1
                    }
                }
            }
        });
        test.done();
    },
    'stream from attributes'(test) {
        const stack = new core_1.Stack();
        const s = lib_1.Stream.fromStreamAttributes(stack, 'MyStream', {
            streamArn: 'arn:aws:kinesis:region:account-id:stream/stream-name'
        });
        test.equals(s.streamArn, 'arn:aws:kinesis:region:account-id:stream/stream-name');
        test.done();
    },
    "uses explicit shard count"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            shardCount: 2
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 2
                    }
                }
            }
        });
        test.done();
    },
    "uses explicit retention period"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            retentionPeriodHours: 168
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 168,
                        "ShardCount": 1
                    }
                }
            }
        });
        test.done();
    },
    "retention period must be between 24 and 168 hours"(test) {
        test.throws({
            block: () => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriodHours: 169
                });
            },
            message: "retentionPeriodHours must be between 24 and 168 hours"
        });
        test.throws({
            block: () => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriodHours: 23
                });
            },
            message: "retentionPeriodHours must be between 24 and 168 hours"
        });
        test.done();
    },
    "auto-creates KMS key if encryption type is KMS but no key is provided"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            encryption: lib_1.StreamEncryption.KMS
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStreamKey76F3300E": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "Created by MyStream",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey",
                                        "kms:TagResource",
                                        "kms:UntagResource"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1,
                        "StreamEncryption": {
                            "EncryptionType": "KMS",
                            "KeyId": {
                                "Fn::GetAtt": [
                                    "MyStreamKey76F3300E",
                                    "Arn"
                                ]
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    "uses explicit KMS key if encryption type is KMS and a key is provided"(test) {
        const stack = new core_1.Stack();
        const explicitKey = new kms.Key(stack, 'ExplicitKey', {
            description: `Explicit Key`
        });
        new lib_1.Stream(stack, 'MyStream', {
            encryption: lib_1.StreamEncryption.KMS,
            encryptionKey: explicitKey
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "ExplicitKey7DF42F37": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "Explicit Key",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey",
                                        "kms:TagResource",
                                        "kms:UntagResource"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1,
                        "StreamEncryption": {
                            "EncryptionType": "KMS",
                            "KeyId": {
                                "Fn::GetAtt": [
                                    "ExplicitKey7DF42F37",
                                    "Arn"
                                ]
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    "permissions": {
        "with encryption": {
            "grantRead creates and attaches a policy with read only access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantRead(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion",
                                                "kms:GenerateDataKey",
                                                "kms:TagResource",
                                                "kms:UntagResource"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": "kms:Decrypt",
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": "kms:Decrypt",
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantWrite creates and attaches a policy with write only access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion",
                                                "kms:GenerateDataKey",
                                                "kms:TagResource",
                                                "kms:UntagResource"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": [
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": [
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*",
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantReadWrite creates and attaches a policy with access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantReadWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion",
                                                "kms:GenerateDataKey",
                                                "kms:TagResource",
                                                "kms:UntagResource"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": [
                                                "kms:Decrypt",
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": [
                                                "kms:Decrypt",
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            }
        },
        "with no encryption": {
            "grantRead creates and associates a policy with read only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantRead(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantWrite creates and attaches a policy with write only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "greatReadWrite creates and attaches a policy with write only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantReadWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            }
        }
    },
    "cross-stack permissions": {
        "no encryption"(test) {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream');
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            assert_1.expect(stackA).toMatch({
                "Resources": {
                    "MyStream5C050E93": {
                        "Type": "AWS::Kinesis::Stream",
                        "Properties": {
                            "RetentionPeriodHours": 24,
                            "ShardCount": 1
                        }
                    }
                },
                "Outputs": {
                    "ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD": {
                        "Value": {
                            "Fn::GetAtt": [
                                "MyStream5C050E93",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "stackA:ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD"
                        }
                    }
                }
            });
            assert_1.expect(stackB).toMatch({
                "Resources": {
                    "UserWhoNeedsAccessF8959C3D": {
                        "Type": "AWS::IAM::User"
                    },
                    "UserWhoNeedsAccessDefaultPolicy6A9EB530": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "kinesis:DescribeStream",
                                            "kinesis:GetRecords",
                                            "kinesis:GetShardIterator"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::ImportValue": "stackA:ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD"
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "UserWhoNeedsAccessDefaultPolicy6A9EB530",
                            "Users": [
                                {
                                    "Ref": "UserWhoNeedsAccessF8959C3D"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        "fails with encryption due to cyclic dependency"(test) {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS
            });
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            test.throws(() => app.synth(), /'stackB' depends on 'stackA'/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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