
# Copyright (c) [2025] [OpenAI]
# Copyright (c) [2025] [ByteDance Ltd. and/or its affiliates.]
# SPDX-License-Identifier: Apache-2.0
#
# This file has been modified by [ByteDance Ltd. and/or its affiliates.] on 2025.7
#
# Original file was released under Apache License Version 2.0, with the full license text
# available at https://github.com/openai/openai-python/blob/main/LICENSE.
#
# This modified file is released under the same license.

from typing import List, Union, Optional
from typing_extensions import Literal, TypeAlias

from .tool import Tool
from ..._models import BaseModel
from .response_error import ResponseError
from .response_usage import ResponseUsage
from .response_status import ResponseStatus
from volcenginesdkarkruntime.types.shared.thinking import Thinking
from .tool_choice_options import ToolChoiceOptions
from .response_output_item import ResponseOutputItem
from .response_text_config import ResponseTextConfig
from .tool_choice_function import ToolChoiceFunction
from .response_incomplete_details import IncompleteDetails
from .response_caching import ResponseCaching


__all__ = ["Response", "ToolChoice"]


ToolChoice: TypeAlias = Union[ToolChoiceOptions, ToolChoiceFunction]


class Response(BaseModel):
    id: str
    """Unique identifier for this Response."""

    created_at: float
    """Unix timestamp (in seconds) of when this Response was created."""

    error: Optional[ResponseError] = None
    """An error object returned when the model fails to generate a Response."""

    incomplete_details: Optional[IncompleteDetails] = None
    """Details about why the response is incomplete."""

    instructions: Optional[str] = None
    """
    Inserts a system (or developer) message as the first item in the model's
    context.

    When using along with `previous_response_id`, the instructions from a previous
    response will not be carried over to the next response. This makes it simple to
    swap out system (or developer) messages in new responses.
    """

    model: str
    """Model Name or endpoint id."""

    object: Literal["response"]
    """The object type of this resource - always set to `response`."""

    output: List[ResponseOutputItem]
    """An array of content items generated by the model.

    - The length and order of items in the `output` array is dependent on the
      model's response.
    - Rather than accessing the first item in the `output` array and assuming it's
      an `assistant` message with the content generated by the model, you might
      consider using the `output_text` property where supported in SDKs.
    """

    parallel_tool_calls: bool
    """Whether to allow the model to run tool calls in parallel."""

    temperature: Optional[float] = None
    """What sampling temperature to use, between 0 and 2.

    Higher values like 0.8 will make the output more random, while lower values like
    0.2 will make it more focused and deterministic. We generally recommend altering
    this or `top_p` but not both.
    """

    tool_choice: ToolChoice
    """
    How the model should select which tool (or tools) to use when generating a
    response. See the `tools` parameter to see how to specify which tools the model
    can call.
    """

    tools: List[Tool]
    """
    An array of tools the model may call while generating a response.

    You can specify which tool to use by setting the `tool_choice` parameter.
    """

    top_p: Optional[float] = None
    """
    An alternative to sampling with temperature, called nucleus sampling, where the
    model considers the results of the tokens with top_p probability mass. So 0.1
    means only the tokens comprising the top 10% probability mass are considered.

    We generally recommend altering this or `temperature` but not both.
    """

    max_output_tokens: Optional[int] = None
    """
    An upper bound for the number of tokens that can be generated for a response.
    """

    previous_response_id: Optional[str] = None
    """The unique ID of the previous response to the model.

    Use this to create multi-turn conversations.
    """

    thinking: Optional[Thinking] = None
    """
    The thinking mode to use for the model.
    """

    service_tier: Optional[Literal["auto", "default"]] = None
    """
    Specifies the latency tier to use for processing the request.
    """

    status: Optional[ResponseStatus] = None
    """The status of the response generation.

    One of `completed`, `failed`, `in_progress`, or `incomplete`.
    """

    text: Optional[ResponseTextConfig] = None
    """Configuration options for a text response from the model.

    Can be plain text or json object

    """

    usage: Optional[ResponseUsage] = None
    """
    Represents token usage details including input tokens, output tokens, a
    breakdown of output tokens, and the total tokens used.
    """

    caching: Optional[ResponseCaching] = None
    """
    Represents the caching configuration for the response.
    """
