"""Base adapter interface for data sources."""

from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import Any

from nlql.text.units import TextUnit


@dataclass
class QueryPlan:
    """Represents a query plan for data retrieval from an adapter.

    QueryPlan is used to communicate retrieval requirements from the Executor
    to the Adapter. It contains ONLY the information needed for data retrieval,
    not for filtering, sorting, or limiting results.

    Design Philosophy:
    - Adapters are responsible for DATA RETRIEVAL only
    - Executors are responsible for QUERY LOGIC (WHERE, ORDER BY, LIMIT)
    - QueryPlan bridges these two layers

    Attributes:
        query_text: Text query for semantic/vector search (e.g., for SIMILAR_TO)
                   Only used by adapters that support semantic search.
        filters: Simple key-value filters that CAN be pushed down to the adapter
                for optimization (e.g., metadata filters for vector databases).
                Currently unused - all filtering is done in the Executor.
        limit: NOT USED. Limiting is handled by the Executor after filtering.
              This field is kept for future optimization scenarios.
        metadata: Additional adapter-specific parameters for advanced use cases.

    Note:
        In the current implementation (Simple Mode):
        - query_text: Used for semantic search adapters
        - filters: None (all filtering in Executor)
        - limit: None (all limiting in Executor)

        In future optimized implementations:
        - The Executor may analyze WHERE clauses and push down simple filters
        - The QueryRouter will determine what can be safely pushed down
    """

    query_text: str | None = None
    filters: dict[str, Any] | None = None
    limit: int | None = None
    metadata: dict[str, Any] | None = None


class BaseAdapter(ABC):
    """Abstract base class for data source adapters.

    Design Philosophy - Adapters are DATA RETRIEVERS, not QUERY PROCESSORS:

    Adapters are responsible for:
    1. Retrieving data from the underlying data source
    2. Converting data to TextUnit objects (NLQL's internal format)
    3. Declaring capabilities (semantic search, etc.)

    Adapters are NOT responsible for:
    1. Filtering results based on WHERE clauses (handled by Executor)
    2. Sorting results based on ORDER BY (handled by Executor)
    3. Limiting results based on LIMIT (handled by Executor)
    4. Granularity transformations like SENTENCE/SPAN (handled by Executor)

    This separation of concerns ensures:
    - Simple adapter implementation (just focus on data retrieval)
    - Consistent query semantics across all data sources
    - Easy testing and debugging
    - Low mental overhead for adapter developers

    Example Implementations:
    - MemoryAdapter: Returns all chunks from in-memory list
    - ChromaAdapter (future): Executes semantic search, returns top-k results
    - FAISSAdapter (future): Executes vector search, returns neighbors
    - SQLAdapter (future): Executes SQL query, returns rows as TextUnits

    Note: While adapters CAN apply optimizations (e.g., using QueryPlan.filters
    for database-level filtering), they should always return semantically
    correct results. The Executor will apply additional filtering as needed.
    """

    @abstractmethod
    def query(self, plan: QueryPlan) -> list[TextUnit]:
        """Retrieve data from the data source.

        This method should focus ONLY on data retrieval, not on filtering,
        sorting, or limiting. The Executor will handle all query logic.

        Typical implementations:
        - MemoryAdapter: Return all chunks
        - VectorDBAdapter: Execute semantic search with plan.query_text,
          return top-k similar chunks
        - SQLAdapter: Execute SQL query, return all matching rows

        Args:
            plan: Query plan containing retrieval parameters
                 - plan.query_text: For semantic/vector search
                 - plan.filters: Optional optimization hints (can be ignored)
                 - plan.limit: Should be ignored (Executor handles limiting)

        Returns:
            List of TextUnit objects retrieved from the data source.
            Do NOT apply WHERE filtering, ORDER BY sorting, or LIMIT here.

        Raises:
            NLQLAdapterError: If data retrieval fails
        """
        pass

    @abstractmethod
    def supports_semantic_search(self) -> bool:
        """Check if this adapter supports semantic/vector similarity search.

        This indicates whether the adapter can handle plan.query_text for
        semantic search (e.g., finding similar documents using embeddings).

        Returns:
            True if the adapter can perform semantic search (e.g., vector DB)
            False if the adapter only returns raw data (e.g., MemoryAdapter)

        Examples:
            - MemoryAdapter: False (no embeddings)
            - ChromaAdapter: True (has vector search)
            - FAISSAdapter: True (has vector search)
        """
        pass

    @abstractmethod
    def supports_metadata_filter(self) -> bool:
        """Check if this adapter can optimize metadata filtering.

        This indicates whether the adapter can use plan.filters to optimize
        data retrieval (e.g., database-level WHERE clauses).

        Note: Returning False does NOT mean metadata filtering is unsupported.
        It just means the adapter doesn't optimize it - the Executor will
        handle all filtering in memory.

        Returns:
            True if the adapter can apply plan.filters for optimization
            False if the adapter ignores plan.filters (Executor handles it)

        Examples:
            - MemoryAdapter: False (returns all data, Executor filters)
            - ChromaAdapter: True (can use Chroma's where clause)
            - SQLAdapter: True (can use SQL WHERE clause)
        """
        pass

    def get_capabilities(self) -> dict[str, bool]:
        """Get a dictionary of adapter capabilities.

        Returns:
            Dictionary mapping capability names to boolean values
        """
        return {
            "semantic_search": self.supports_semantic_search(),
            "metadata_filter": self.supports_metadata_filter(),
        }

