"""Query routing and pushdown logic.

This module analyzes WHERE clauses and determines what can be pushed down
to the data source vs what needs to be executed in-memory.
"""

from dataclasses import dataclass, field
from typing import Any

from nlql.ast.nodes import ASTNode, SelectStatement


@dataclass
class RoutingPlan:
    """Plan for routing query execution.

    Attributes:
        pushdown_filters: Filters that can be pushed to the data source
        in_memory_filters: Filters that must be executed in-memory
        requires_embedding: Whether semantic search is needed
        metadata: Additional routing metadata
    """

    pushdown_filters: dict[str, Any] = field(default_factory=dict)
    in_memory_filters: list[ASTNode] = field(default_factory=list)
    requires_embedding: bool = False
    metadata: dict[str, Any] = field(default_factory=dict)


class QueryRouter:
    """Analyzes queries and creates routing plans.

    This determines which parts of a query can be pushed down to the
    data source and which parts need in-memory execution.
    """

    def __init__(self, adapter_capabilities: dict[str, bool]) -> None:
        """Initialize the router.

        Args:
            adapter_capabilities: Capabilities of the data source adapter
        """
        self.capabilities = adapter_capabilities

    def create_plan(self, ast: SelectStatement) -> RoutingPlan:
        """Create a routing plan from an AST.

        Args:
            ast: SELECT statement AST

        Returns:
            Routing plan with pushdown and in-memory components
        """
        plan = RoutingPlan()

        # TODO: Implement WHERE clause analysis
        # For now, return empty plan
        # Future implementation will:
        # 1. Walk the WHERE clause AST
        # 2. Identify SIMILAR_TO, META, etc.
        # 3. Check adapter capabilities
        # 4. Split into pushdown vs in-memory

        return plan

