"""Core type system implementation for NLQL.

This module defines the base type abstraction and built-in type classes
that support comparison operations in WHERE clauses.
"""

from abc import ABC, abstractmethod
from datetime import datetime
from typing import Any


class BaseType(ABC):
    """Abstract base class for all NLQL types.

    Types must implement comparison methods to support WHERE clause operations.
    """

    def __init__(self, value: Any) -> None:
        self.value = value

    @abstractmethod
    def __lt__(self, other: Any) -> bool:
        """Less than comparison."""
        pass

    @abstractmethod
    def __le__(self, other: Any) -> bool:
        """Less than or equal comparison."""
        pass

    @abstractmethod
    def __gt__(self, other: Any) -> bool:
        """Greater than comparison."""
        pass

    @abstractmethod
    def __ge__(self, other: Any) -> bool:
        """Greater than or equal comparison."""
        pass

    @abstractmethod
    def __eq__(self, other: Any) -> bool:
        """Equality comparison."""
        pass

    def __ne__(self, other: Any) -> bool:
        """Not equal comparison."""
        return not self.__eq__(other)

    def __repr__(self) -> str:
        return f"{self.__class__.__name__}({self.value!r})"


class NumberType(BaseType):
    """Numeric type supporting arithmetic comparisons."""

    def __init__(self, value: int | float) -> None:
        super().__init__(float(value))

    def __lt__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value < float(other_val)

    def __le__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value <= float(other_val)

    def __gt__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value > float(other_val)

    def __ge__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value >= float(other_val)

    def __eq__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value == float(other_val)


class TextType(BaseType):
    """Text type supporting lexicographic comparisons."""

    def __init__(self, value: str) -> None:
        super().__init__(str(value))

    def __lt__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value < str(other_val)

    def __le__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value <= str(other_val)

    def __gt__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value > str(other_val)

    def __ge__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value >= str(other_val)

    def __eq__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, BaseType) else other
        return self.value == str(other_val)


class DateType(BaseType):
    """Date/datetime type supporting temporal comparisons."""

    def __init__(self, value: datetime | str) -> None:
        if isinstance(value, str):
            # Simple ISO format parsing, can be extended
            value = datetime.fromisoformat(value)
        super().__init__(value)

    def __lt__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, DateType) else other
        if isinstance(other_val, str):
            other_val = datetime.fromisoformat(other_val)
        return self.value < other_val

    def __le__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, DateType) else other
        if isinstance(other_val, str):
            other_val = datetime.fromisoformat(other_val)
        return self.value <= other_val

    def __gt__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, DateType) else other
        if isinstance(other_val, str):
            other_val = datetime.fromisoformat(other_val)
        return self.value > other_val

    def __ge__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, DateType) else other
        if isinstance(other_val, str):
            other_val = datetime.fromisoformat(other_val)
        return self.value >= other_val

    def __eq__(self, other: Any) -> bool:
        other_val = other.value if isinstance(other, DateType) else other
        if isinstance(other_val, str):
            other_val = datetime.fromisoformat(other_val)
        return self.value == other_val

