"""META field type registration and management."""

from typing import Any, Callable, Type

from nlql.errors import NLQLTypeError
from nlql.types.core import BaseType, NumberType, TextType


class MetaFieldRegistry:
    """Registry for META field type information.

    This allows users to register type information for metadata fields,
    enabling type-safe comparisons in WHERE clauses.
    """

    def __init__(self) -> None:
        self._field_types: dict[str, Type[BaseType]] = {}

    def register(self, field_name: str, type_class: Type[BaseType]) -> None:
        """Register a META field with its type class.

        Args:
            field_name: Name of the metadata field
            type_class: Type class (e.g., NumberType, TextType, DateType)

        Raises:
            NLQLTypeError: If type_class is not a valid BaseType subclass
        """
        if not issubclass(type_class, BaseType):
            raise NLQLTypeError(
                f"Type class must be a subclass of BaseType, got {type_class}"
            )
        self._field_types[field_name] = type_class

    def get_type(self, field_name: str) -> Type[BaseType] | None:
        """Get the registered type class for a field.

        Args:
            field_name: Name of the metadata field

        Returns:
            Type class if registered, None otherwise
        """
        return self._field_types.get(field_name)

    def wrap_value(self, field_name: str, value: Any) -> BaseType:
        """Wrap a raw value in its registered type class.

        Args:
            field_name: Name of the metadata field
            value: Raw value from data source

        Returns:
            Wrapped value as a BaseType instance

        Raises:
            NLQLTypeError: If field is not registered or wrapping fails
        """
        type_class = self.get_type(field_name)
        if type_class is None:
            # Default to TextType if not registered
            type_class = TextType

        try:
            return type_class(value)
        except Exception as e:
            raise NLQLTypeError(
                f"Failed to wrap value {value!r} for field '{field_name}' "
                f"with type {type_class.__name__}: {e}"
            ) from e

    def clear(self) -> None:
        """Clear all registered field types."""
        self._field_types.clear()


# Global registry instance
_global_meta_registry = MetaFieldRegistry()


def register_meta_field(field_name: str, type_class: Type[BaseType]) -> None:
    """Register a META field type in the global registry.

    Args:
        field_name: Name of the metadata field
        type_class: Type class (e.g., NumberType, TextType, DateType)

    Example:
        >>> from nlql.types import register_meta_field, NumberType
        >>> register_meta_field("page_count", NumberType)
    """
    _global_meta_registry.register(field_name, type_class)


def get_meta_registry() -> MetaFieldRegistry:
    """Get the global META field registry."""
    return _global_meta_registry

