import json
import httpx
import argparse
from typing import Any
import markdown
from mcp.server.fastmcp import FastMCP

mcp = FastMCP("weatherserver")
# 7e6cfc5dc71cee3af065bcc59997e445
OPENWEATHER_API_KEY = "7e6cfc5dc71cee3af065bcc59997e445"
OPENWEATHER_API_URL = "https://api.openweathermap.org/data/2.5/weather"
USER_AGENT = "weather-app/1.0"


async def fetch_weather(city: str) -> dict[str, Any] | None:
    """
    从 OpenWeatherMap 获取天气数据
    """
    if OPENWEATHER_API_KEY is None:
        return {"error": "API key not set"}
    params = {
        "q": city,
        "appid": OPENWEATHER_API_KEY,
        "units": "metric",
        "lang": "zh_cn"
    }
    headers = {
        "User-Agent": USER_AGENT
    }
    async with httpx.AsyncClient() as client:
        try:
            response = await client.get(OPENWEATHER_API_URL, params=params, headers=headers)
            response.raise_for_status()
            return response.json()
        except httpx.HTTPStatusError as e:
            return {"error": f"http error: {e.response.status_code}"}
        except Exception as e:
            return {"error": f"请求失败: {e}"}


def format_weather(data: dict[str, Any] | str) -> str:
    """
    格式化天气数据
    """
    if isinstance(data, str):
        try:
            data = json.loads(data)
        except Exception as e:
            return f"无法解析天气数据: {e}"
    if "error" in data:
        return f"{data['error']}"

    city = data.get("name", "未知城市")
    country = data.get("sys", {}).get("country", "未知国家")
    temp = data.get("main", {}).get("temp", "N/A")
    humidity = data.get("main", {}).get("humidity", "N/A")
    wind_speed = data.get("wind", {}).get("speed", "N/A")
    wind_deg = data.get("wind", {}).get("deg", "N/A")
    weather_list = data.get("weather", [{}])
    description = weather_list[0].get("description", "未知天气")
    return (
        f"# {city} ({country})\n"
        f"温度: {temp}°C\n"
        f"湿度: {humidity}%\n"
        f"风速: {wind_speed} m/s\n"
        f"天气: {description}\n"
    )


@mcp.tool()
async def query_weather(city: str) -> str:
    """
    输入指定的城市的英文名称，返回今日天气的查询结果
    """
    data = await fetch_weather(city)
    return format_weather(data)


def main():
    parser = argparse.ArgumentParser(description="Weather Server")
    # parser.add_argument("--port", type=int, default=8080, help="端口")
    # parser.add_argument("--host", type=str, default="127.0.0.1", help="主机")
    parser.add_argument("--openweather_api_key", type=str,
                        required=True, help="OpenWeatherMap API Key")
    args = parser.parse_args()
    global OPENWEATHER_API_KEY
    OPENWEATHER_API_KEY = args.openweather_api_key
    args = parser.parse_args()
    mcp.run(transport="stdio")


if __name__ == "__main__":
    main()
