import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import iam = require('@aws-cdk/aws-iam');
import lambda = require('@aws-cdk/aws-lambda');
import cdk = require('@aws-cdk/cdk');
import { AutoRollbackConfig } from '../rollback-config';
import { ILambdaApplication } from './application';
import { ILambdaDeploymentConfig } from './deployment-config';
/**
 * Interface for a Lambda deployment groups.
 */
export interface ILambdaDeploymentGroup extends cdk.IConstruct {
    /**
     * The reference to the CodeDeploy Lambda Application that this Deployment Group belongs to.
     */
    readonly application: ILambdaApplication;
    /**
     * The physical name of the CodeDeploy Deployment Group.
     */
    readonly deploymentGroupName: string;
    /**
     * The ARN of this Deployment Group.
     */
    readonly deploymentGroupArn: string;
    /**
     * Export this Deployment Group for use in another stack or application.
     */
    export(): LambdaDeploymentGroupImportProps;
}
/**
 * Construction properties for {@link LambdaDeploymentGroup}.
 */
export interface LambdaDeploymentGroupProps {
    /**
     * The reference to the CodeDeploy Lambda Application that this Deployment Group belongs to.
     *
     * @default one will be created for you
     */
    readonly application?: ILambdaApplication;
    /**
     * The physical, human-readable name of the CodeDeploy Deployment Group.
     *
     * @default an auto-generated name will be used
     */
    readonly deploymentGroupName?: string;
    /**
     * The Deployment Configuration this Deployment Group uses.
     *
     * @default LambdaDeploymentConfig#AllAtOnce
     */
    readonly deploymentConfig?: ILambdaDeploymentConfig;
    /**
     * The CloudWatch alarms associated with this Deployment Group.
     * CodeDeploy will stop (and optionally roll back)
     * a deployment if during it any of the alarms trigger.
     *
     * Alarms can also be added after the Deployment Group is created using the {@link #addAlarm} method.
     *
     * @default []
     * @see https://docs.aws.amazon.com/codedeploy/latest/userguide/monitoring-create-alarms.html
     */
    readonly alarms?: cloudwatch.Alarm[];
    /**
     * The service Role of this Deployment Group.
     *
     * @default a new Role will be created.
     */
    readonly role?: iam.IRole;
    /**
     * Lambda Alias to shift traffic. Updating the version
     * of the alias will trigger a CodeDeploy deployment.
     */
    readonly alias: lambda.Alias;
    /**
     * The Lambda function to run before traffic routing starts.
     */
    readonly preHook?: lambda.IFunction;
    /**
     * The Lambda function to run after traffic routing starts.
     */
    readonly postHook?: lambda.IFunction;
    /**
     * Whether to continue a deployment even if fetching the alarm status from CloudWatch failed.
     *
     * @default false
     */
    readonly ignorePollAlarmsFailure?: boolean;
    /**
     * The auto-rollback configuration for this Deployment Group.
     */
    readonly autoRollback?: AutoRollbackConfig;
}
export declare class LambdaDeploymentGroup extends cdk.Construct implements ILambdaDeploymentGroup {
    /**
     * Import an Lambda Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param props the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static import(scope: cdk.Construct, id: string, props: LambdaDeploymentGroupImportProps): ILambdaDeploymentGroup;
    readonly application: ILambdaApplication;
    readonly deploymentGroupName: string;
    readonly deploymentGroupArn: string;
    readonly role: iam.IRole;
    private readonly alarms;
    private preHook?;
    private postHook?;
    constructor(scope: cdk.Construct, id: string, props: LambdaDeploymentGroupProps);
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm: cloudwatch.Alarm): void;
    /**
     * Associate a function to run before deployment begins.
     * @param preHook function to run before deployment beings
     * @throws an error if a pre-hook function is already configured
     */
    onPreHook(preHook: lambda.IFunction): void;
    /**
     * Associate a function to run after deployment completes.
     * @param preHook function to run after deployment completes
     * @throws an error if a post-hook function is already configured
     */
    onPostHook(postHook: lambda.IFunction): void;
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus
     * on this deployment group resource.
     * @param principal to grant permission to
     */
    grantPutLifecycleEventHookExecutionStatus(principal?: iam.IPrincipal): void;
    export(): LambdaDeploymentGroupImportProps;
}
/**
 * Properties of a reference to a CodeDeploy Lambda Deployment Group.
 *
 * @see LambdaDeploymentGroup#import
 * @see ILambdaDeploymentGroup#export
 */
export interface LambdaDeploymentGroupImportProps {
    /**
     * The reference to the CodeDeploy Lambda Application
     * that this Deployment Group belongs to.
     */
    readonly application: ILambdaApplication;
    /**
     * The physical, human-readable name of the CodeDeploy Lambda Deployment Group
     * that we are referencing.
     */
    readonly deploymentGroupName: string;
}
