"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
class LambdaDeploymentGroup extends cdk.Construct {
    /**
     * Import an Lambda Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param props the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static import(scope, id, props) {
        return new ImportedLambdaDeploymentGroup(scope, id, props);
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com')
        });
        this.role.attachManagedPolicy('arn:aws:iam::aws:policy/service-role/AWSCodeDeployRoleForLambda');
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: props.deploymentGroupName,
            deploymentConfigName: (props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.AllAtOnce).deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL'
            },
            alarmConfiguration: new cdk.Token(() => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure)),
            autoRollbackConfiguration: new cdk.Token(() => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback)),
        });
        this.deploymentGroupName = resource.deploymentGroupName;
        this.deploymentGroupArn = utils_1.deploymentGroupNameToArn(this.application.applicationName, this.deploymentGroupName, this);
        if (props.preHook) {
            this.onPreHook(props.preHook);
        }
        if (props.postHook) {
            this.onPostHook(props.postHook);
        }
        props.alias.node.findChild('Resource').options.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.deploymentGroupName,
                beforeAllowTrafficHook: new cdk.Token(() => this.preHook === undefined ? undefined : this.preHook.functionName).toString(),
                afterAllowTrafficHook: new cdk.Token(() => this.postHook === undefined ? undefined : this.postHook.functionName).toString()
            }
        };
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * Associate a function to run before deployment begins.
     * @param preHook function to run before deployment beings
     * @throws an error if a pre-hook function is already configured
     */
    onPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook.role);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * Associate a function to run after deployment completes.
     * @param preHook function to run after deployment completes
     * @throws an error if a post-hook function is already configured
     */
    onPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook.role);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus
     * on this deployment group resource.
     * @param principal to grant permission to
     */
    grantPutLifecycleEventHookExecutionStatus(principal) {
        if (principal) {
            principal.addToPolicy(new iam.PolicyStatement()
                .addResource(this.deploymentGroupArn)
                .addAction('codedeploy:PutLifecycleEventHookExecutionStatus'));
        }
    }
    export() {
        return {
            application: this.application,
            deploymentGroupName: new cdk.CfnOutput(this, 'DeploymentGroupName', {
                value: this.deploymentGroupName
            }).makeImportValue().toString()
        };
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
class ImportedLambdaDeploymentGroup extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.deploymentGroupNameToArn(props.application.applicationName, props.deploymentGroupName, this);
    }
    export() {
        return this.props;
    }
}
//# sourceMappingURL=data:application/json;base64,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