"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
class ImportedServerDeploymentConfig extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.deploymentConfigName = props.deploymentConfigName;
    }
    deploymentConfigArn(scope) {
        return utils_1.arnForDeploymentConfigName(this.deploymentConfigName, scope);
    }
    export() {
        return this.props;
    }
}
class DefaultServerDeploymentConfig {
    constructor(deploymentConfigName) {
        this.deploymentConfigName = deploymentConfigName;
    }
    deploymentConfigArn(scope) {
        return utils_1.arnForDeploymentConfigName(this.deploymentConfigName, scope);
    }
    export() {
        return {
            deploymentConfigName: this.deploymentConfigName
        };
    }
}
/**
 * A custom Deployment Configuration for an EC2/on-premise Deployment Group.
 */
class ServerDeploymentConfig extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new codedeploy_generated_1.CfnDeploymentConfig(this, 'Resource', {
            deploymentConfigName: props.deploymentConfigName,
            minimumHealthyHosts: this.minimumHealthyHosts(props),
        });
        this.deploymentConfigName = resource.ref.toString();
    }
    /**
     * Import a custom Deployment Configuration for an EC2/on-premise Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param props the properties of the referenced custom Deployment Configuration
     * @returns a Construct representing a reference to an existing custom Deployment Configuration
     */
    static import(scope, id, props) {
        return new ImportedServerDeploymentConfig(scope, id, props);
    }
    deploymentConfigArn(scope) {
        return utils_1.arnForDeploymentConfigName(this.deploymentConfigName, scope);
    }
    export() {
        return {
            deploymentConfigName: new cdk.CfnOutput(this, 'DeploymentConfigName', {
                value: this.deploymentConfigName,
            }).makeImportValue().toString(),
        };
    }
    minimumHealthyHosts(props) {
        if (props.minHealthyHostCount === undefined && props.minHealthyHostPercentage === undefined) {
            throw new Error('At least one of minHealthyHostCount or minHealthyHostPercentage must be specified when creating ' +
                'a custom Server DeploymentConfig');
        }
        if (props.minHealthyHostCount !== undefined && props.minHealthyHostPercentage !== undefined) {
            throw new Error('Both minHealthyHostCount and minHealthyHostPercentage cannot be specified when creating ' +
                'a custom Server DeploymentConfig');
        }
        return {
            type: props.minHealthyHostCount !== undefined ? 'HOST_COUNT' : 'FLEET_PERCENT',
            value: props.minHealthyHostCount !== undefined ? props.minHealthyHostCount : props.minHealthyHostPercentage,
        };
    }
}
ServerDeploymentConfig.OneAtATime = new DefaultServerDeploymentConfig('CodeDeployDefault.OneAtATime');
ServerDeploymentConfig.HalfAtATime = new DefaultServerDeploymentConfig('CodeDeployDefault.HalfAtATime');
ServerDeploymentConfig.AllAtOnce = new DefaultServerDeploymentConfig('CodeDeployDefault.AllAtOnce');
exports.ServerDeploymentConfig = ServerDeploymentConfig;
//# sourceMappingURL=data:application/json;base64,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