"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codedeploylb = require("@aws-cdk/aws-codedeploy-api");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const codedeploy_generated_1 = require("../codedeploy.generated");
const pipeline_action_1 = require("../pipeline-action");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Construct {
    constructor(scope, id, deploymentConfig) {
        super(scope, id);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.OneAtATime;
    }
    toCodePipelineDeployAction(props) {
        return new pipeline_action_1.PipelineDeployAction(Object.assign({}, props, { deploymentGroup: this }));
    }
}
exports.ServerDeploymentGroupBase = ServerDeploymentGroupBase;
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.props = props;
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.deploymentGroupNameToArn(props.application.applicationName, props.deploymentGroupName, this);
    }
    export() {
        return this.props;
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK,
     * or in a different CDK Stack and exported using the {@link #export} method.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param props the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static import(scope, id, props) {
        return new ImportedServerDeploymentGroup(scope, id, props);
    }
    constructor(scope, id, props = {}) {
        super(scope, id, props.deploymentConfig);
        this.application = props.application || new application_1.ServerApplication(this, 'Application');
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicyArns: ['arn:aws:iam::aws:policy/service-role/AWSCodeDeployRole'],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent === undefined ? true : props.installAgent;
        this.codeDeployBucket = s3.Bucket.import(this, 'CodeDeployBucket', {
            bucketName: `aws-codedeploy-${this.node.stack.region}`,
        });
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: props.deploymentGroupName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: new cdk.Token(() => this._autoScalingGroups.length === 0
                ? undefined
                : this._autoScalingGroups.map(asg => asg.autoScalingGroupName)).toList(),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: new cdk.Token(() => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure)),
            autoRollbackConfiguration: new cdk.Token(() => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback)),
        });
        this.deploymentGroupName = resource.deploymentGroupName;
        this.deploymentGroupArn = utils_1.deploymentGroupNameToArn(this.application.applicationName, this.deploymentGroupName, this);
    }
    export() {
        return {
            application: this.application,
            deploymentGroupName: new cdk.CfnOutput(this, 'DeploymentGroupName', {
                value: this.deploymentGroupName
            }).makeImportValue().toString(),
            deploymentConfig: this.deploymentConfig,
        };
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg.role, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.Linux:
                asg.addUserData('PKG_CMD=`which yum 2>/dev/null`', 'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG=CMD=yum', 'fi', '$PKG_CMD update -y', '$PKG_CMD install -y ruby2.0', 'if [ $? -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', '$PKG_CMD install -y awscli', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${this.node.stack.region}/latest/install . --region ${this.node.stack.region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.Windows:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${this.node.stack.region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, '$TEMPDIR\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(lbProvider) {
        if (!lbProvider) {
            return undefined;
        }
        const lb = lbProvider.asCodeDeployLoadBalancer();
        switch (lb.generation) {
            case codedeploylb.LoadBalancerGeneration.First:
                return {
                    elbInfoList: [
                        { name: lb.name },
                    ],
                };
            case codedeploylb.LoadBalancerGeneration.Second:
                return {
                    targetGroupInfoList: [
                        { name: lb.name },
                    ]
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = [];
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,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