"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/cdk");
const codedeploy = require("../../lib");
const lib_1 = require("../../lib");
function mockFunction(stack, id) {
    return new lambda.Function(stack, id, {
        code: lambda.Code.inline('mock'),
        handler: 'index.handler',
        runtime: lambda.Runtime.NodeJS810
    });
}
function mockAlias(stack) {
    return new lambda.Alias(stack, 'Alias', {
        aliasName: 'my-alias',
        version: new lambda.Version(stack, 'Version', {
            lambda: mockFunction(stack, 'Function')
        })
    });
}
module.exports = {
    "CodeDeploy Lambda DeploymentGroup": {
        "can be created with default AllAtOnce IN_PLACE configuration"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: "MyApp3CE31C26"
                },
                ServiceRoleArn: {
                    "Fn::GetAtt": [
                        "MyDGServiceRole5E94FD88",
                        "Arn"
                    ]
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE"
                    ]
                },
                DeploymentConfigName: "CodeDeployDefault.LambdaAllAtOnce",
                DeploymentStyle: {
                    DeploymentOption: "WITH_TRAFFIC_CONTROL",
                    DeploymentType: "BLUE_GREEN"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
                Type: "AWS::Lambda::Alias",
                Properties: {
                    FunctionName: {
                        Ref: "Function76856677"
                    },
                    FunctionVersion: {
                        "Fn::GetAtt": [
                            "Version6A868472",
                            "Version"
                        ]
                    },
                    Name: "my-alias"
                },
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: "sts:AssumeRole",
                            Effect: "Allow",
                            Principal: {
                                Service: { "Fn::Join": ["", ["codedeploy.", { Ref: "AWS::Region" }, ".", { Ref: "AWS::URLSuffix" }]] }
                            }
                        }],
                    Version: "2012-10-17"
                },
                ManagedPolicyArns: ['arn:aws:iam::aws:policy/service-role/AWSCodeDeployRoleForLambda']
            }));
            test.done();
        },
        "can be created with explicit name"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce,
                deploymentGroupName: 'test'
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                DeploymentGroupName: "test",
            }));
            test.done();
        },
        "can be created with explicit role"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const serviceRole = new iam.Role(stack, 'MyRole', {
                assumedBy: new iam.ServicePrincipal('not-codedeploy.test')
            });
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce,
                role: serviceRole
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: "sts:AssumeRole",
                            Effect: "Allow",
                            Principal: {
                                Service: "not-codedeploy.test"
                            }
                        }],
                    Version: "2012-10-17"
                },
                ManagedPolicyArns: ['arn:aws:iam::aws:policy/service-role/AWSCodeDeployRoleForLambda']
            }));
            test.done();
        },
        "can configure blue/green traffic shifting"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.Linear10PercentEvery1Minute
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: "MyApp3CE31C26"
                },
                ServiceRoleArn: {
                    "Fn::GetAtt": [
                        "MyDGServiceRole5E94FD88",
                        "Arn"
                    ]
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE"
                    ]
                },
                DeploymentConfigName: "CodeDeployDefault.LambdaLinear10PercentEvery1Minute",
                DeploymentStyle: {
                    DeploymentOption: "WITH_TRAFFIC_CONTROL",
                    DeploymentType: "BLUE_GREEN"
                }
            }));
            test.done();
        },
        "can rollback on alarm"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.AllAtOnce,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GreaterThanThreshold,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: "Failures8A3E1A2F"
                            }
                        }],
                    Enabled: true
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE",
                        "DEPLOYMENT_STOP_ON_ALARM"
                    ]
                },
            }));
            test.done();
        },
        "onPreHook throws error if pre-hook already defined"(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce
            });
            test.throws(() => group.onPreHook(mockFunction(stack, 'PreHook2')));
            test.done();
        },
        "onPostHook throws error if post-hook already defined"(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce
            });
            test.throws(() => group.onPostHook(mockFunction(stack, 'PostHook2')));
            test.done();
        },
        "can run pre hook lambda function before deployment"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        BeforeAllowTrafficHook: {
                            Ref: "PreHook8B53F672"
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PreHook8B53F672",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                }
            }));
            test.done();
        },
        "can add pre hook lambda function after creating the deployment group"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce
            });
            group.onPreHook(mockFunction(stack, 'PreHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        BeforeAllowTrafficHook: {
                            Ref: "PreHook8B53F672"
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PreHook8B53F672",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                }
            }));
            test.done();
        },
        "can run post hook lambda function before deployment"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PostHookF2E49B30",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                },
            }));
            test.done();
        },
        "can add post hook lambda function after creating the deployment group"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce
            });
            group.onPostHook(mockFunction(stack, 'PostHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PostHookF2E49B30",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                },
            }));
            test.done();
        },
        "can disable rollback when alarm polling fails"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce,
                ignorePollAlarmsFailure: true,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GreaterThanThreshold,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: "Failures8A3E1A2F"
                            }
                        }],
                    Enabled: true,
                    IgnorePollAlarmFailure: true
                },
            }));
            test.done();
        },
        "can disable rollback when deployment fails"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce,
                autoRollback: {
                    failedDeployment: false
                }
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: "MyApp3CE31C26"
                },
                ServiceRoleArn: {
                    "Fn::GetAtt": [
                        "MyDGServiceRole5E94FD88",
                        "Arn"
                    ]
                },
                DeploymentConfigName: "CodeDeployDefault.LambdaAllAtOnce",
                DeploymentStyle: {
                    DeploymentOption: "WITH_TRAFFIC_CONTROL",
                    DeploymentType: "BLUE_GREEN"
                }
            }));
            test.done();
        },
        "can enable rollback when deployment stops"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce,
                autoRollback: {
                    stoppedDeployment: true
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE",
                        "DEPLOYMENT_STOP_ON_REQUEST"
                    ]
                },
            }));
            test.done();
        },
        "can disable rollback when alarm in failure state"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.AllAtOnce,
                autoRollback: {
                    deploymentInAlarm: false
                },
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GreaterThanThreshold,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE",
                    ]
                },
            }));
            test.done();
        },
    }
};
//# sourceMappingURL=data:application/json;base64,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