"use strict";
const assert_1 = require("@aws-cdk/assert");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const lbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cdk = require("@aws-cdk/cdk");
const codedeploy = require("../../lib");
module.exports = {
    'CodeDeploy Server Deployment Group': {
        "can be created by explicitly passing an Application"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.ServerApplication(stack, 'MyApp');
            new codedeploy.ServerDeploymentGroup(stack, 'MyDG', {
                application,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "ApplicationName": {
                    "Ref": "MyApp3CE31C26"
                },
            }));
            test.done();
        },
        'can be imported'(test) {
            const stack = new cdk.Stack();
            const application = codedeploy.ServerApplication.import(stack, 'MyApp', {
                applicationName: 'MyApp',
            });
            const deploymentGroup = codedeploy.ServerDeploymentGroup.import(stack, 'MyDG', {
                application,
                deploymentGroupName: 'MyDG',
            });
            test.notEqual(deploymentGroup, undefined);
            test.done();
        },
        "created with ASGs contains the ASG names"(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: new ec2.InstanceTypePair(ec2.InstanceClass.Standard3, ec2.InstanceSize.Small),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.VpcNetwork(stack, 'VPC'),
            });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoScalingGroups: [asg],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoScalingGroups": [
                    {
                        "Ref": "ASG46ED3070",
                    },
                ]
            }));
            test.done();
        },
        "created without ASGs but adding them later contains the ASG names"(test) {
            const stack = new cdk.Stack();
            const asg = new autoscaling.AutoScalingGroup(stack, 'ASG', {
                instanceType: new ec2.InstanceTypePair(ec2.InstanceClass.Standard3, ec2.InstanceSize.Small),
                machineImage: new ec2.AmazonLinuxImage(),
                vpc: new ec2.VpcNetwork(stack, 'VPC'),
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAutoScalingGroup(asg);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoScalingGroups": [
                    {
                        "Ref": "ASG46ED3070",
                    },
                ]
            }));
            test.done();
        },
        'can be created with an ALB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const alb = new lbv2.ApplicationLoadBalancer(stack, 'ALB', {
                vpc: new ec2.VpcNetwork(stack, 'VPC'),
            });
            const listener = alb.addListener('Listener', { protocol: lbv2.ApplicationProtocol.Http });
            const targetGroup = listener.addTargets('Fleet', { protocol: lbv2.ApplicationProtocol.Http });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: targetGroup,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "LoadBalancerInfo": {
                    "TargetGroupInfoList": [
                        {
                            "Name": {
                                "Fn::GetAtt": [
                                    "ALBListenerFleetGroup008CEEE4",
                                    "TargetGroupName",
                                ],
                            },
                        },
                    ],
                },
                "DeploymentStyle": {
                    "DeploymentOption": "WITH_TRAFFIC_CONTROL",
                },
            }));
            test.done();
        },
        'can be created with an NLB Target Group as the load balancer'(test) {
            const stack = new cdk.Stack();
            const nlb = new lbv2.NetworkLoadBalancer(stack, 'NLB', {
                vpc: new ec2.VpcNetwork(stack, 'VPC'),
            });
            const listener = nlb.addListener('Listener', { port: 80 });
            const targetGroup = listener.addTargets('Fleet', { port: 80 });
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                loadBalancer: targetGroup,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "LoadBalancerInfo": {
                    "TargetGroupInfoList": [
                        {
                            "Name": {
                                "Fn::GetAtt": [
                                    "NLBListenerFleetGroupB882EC86",
                                    "TargetGroupName",
                                ],
                            },
                        },
                    ],
                },
                "DeploymentStyle": {
                    "DeploymentOption": "WITH_TRAFFIC_CONTROL",
                },
            }));
            test.done();
        },
        'can be created with a single EC2 instance tag set with a single or no value'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                ec2InstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value'],
                    'other-key': [],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "Ec2TagSet": {
                    "Ec2TagSetList": [
                        {
                            "Ec2TagGroup": [
                                {
                                    "Key": "some-key",
                                    "Value": "some-value",
                                    "Type": "KEY_AND_VALUE",
                                },
                                {
                                    "Key": "other-key",
                                    "Type": "KEY_ONLY",
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'can be created with two on-premise instance tag sets with multiple values or without a key'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                    'some-key': ['some-value', 'another-value'],
                }, {
                    '': ['keyless-value'],
                }),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "OnPremisesTagSet": {
                    "OnPremisesTagSetList": [
                        {
                            "OnPremisesTagGroup": [
                                {
                                    "Key": "some-key",
                                    "Value": "some-value",
                                    "Type": "KEY_AND_VALUE",
                                },
                                {
                                    "Key": "some-key",
                                    "Value": "another-value",
                                    "Type": "KEY_AND_VALUE",
                                },
                            ],
                        },
                        {
                            "OnPremisesTagGroup": [
                                {
                                    "Value": "keyless-value",
                                    "Type": "VALUE_ONLY",
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'cannot be created with an instance tag set containing a keyless, valueless filter'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({
                        '': [],
                    }),
                });
            });
            test.done();
        },
        'cannot be created with an instance tag set containing 4 instance tag groups'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                    onPremiseInstanceTags: new codedeploy.InstanceTagSet({}, {}, {}, {}),
                });
            }, /3/);
            test.done();
        },
        'can have alarms added to it after being created'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AlarmConfiguration": {
                    "Alarms": [
                        {
                            "Name": {
                                "Ref": "Alarm1F9009D71",
                            },
                        },
                    ],
                    "Enabled": true,
                },
            }));
            test.done();
        },
        'only automatically rolls back failed deployments by default'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoRollbackConfiguration": {
                    "Enabled": true,
                    "Events": [
                        "DEPLOYMENT_FAILURE",
                    ],
                },
            }));
            test.done();
        },
        'rolls back alarmed deployments if at least one alarm has been added'(test) {
            const stack = new cdk.Stack();
            const alarm = new cloudwatch.Alarm(stack, 'Alarm1', {
                metric: new cloudwatch.Metric({
                    metricName: 'Errors',
                    namespace: 'my.namespace',
                }),
                threshold: 1,
                evaluationPeriods: 1,
            });
            const deploymentGroup = new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    failedDeployment: false,
                },
            });
            deploymentGroup.addAlarm(alarm);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                "AutoRollbackConfiguration": {
                    "Enabled": true,
                    "Events": [
                        "DEPLOYMENT_STOP_ON_ALARM",
                    ],
                },
            }));
            test.done();
        },
        'setting to roll back on alarms without providing any results in an exception'(test) {
            const stack = new cdk.Stack();
            new codedeploy.ServerDeploymentGroup(stack, 'DeploymentGroup', {
                autoRollback: {
                    deploymentInAlarm: true,
                },
            });
            test.throws(() => assert_1.SynthUtils.toCloudFormation(stack), /deploymentInAlarm/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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