"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const path = require("path");
const lambda = require("../lib");
function defineFunction(code, runtime = lambda.Runtime.NODEJS_10_X) {
    const stack = new cdk.Stack();
    return new lambda.Function(stack, 'Func', {
        handler: 'foom',
        code, runtime
    });
}
function generateRandomString(bytes) {
    let s = '';
    for (let i = 0; i < bytes; ++i) {
        s += String.fromCharCode(Math.round(Math.random() * 256));
    }
    return s;
}
module.exports = {
    'lambda.Code.fromInline': {
        'fails if used with unsupported runtimes'(test) {
            test.throws(() => defineFunction(lambda.Code.fromInline('boom'), lambda.Runtime.GO_1_X), /Inline source not allowed for go1\.x/);
            test.throws(() => defineFunction(lambda.Code.fromInline('boom'), lambda.Runtime.JAVA_8), /Inline source not allowed for java8/);
            test.done();
        },
        'fails if larger than 4096 bytes'(test) {
            test.throws(() => defineFunction(lambda.Code.fromInline(generateRandomString(4097)), lambda.Runtime.NODEJS_10_X), /Lambda source is too large, must be <= 4096 but is 4097/);
            test.done();
        }
    },
    'lambda.Code.fromAsset': {
        'fails if a non-zip asset is used'(test) {
            // GIVEN
            const fileAsset = lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler', 'index.py'));
            // THEN
            test.throws(() => defineFunction(fileAsset), /Asset must be a \.zip file or a directory/);
            test.done();
        },
        'only one Asset object gets created even if multiple functions use the same AssetCode'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'MyStack');
            const directoryAsset = lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler'));
            // WHEN
            new lambda.Function(stack, 'Func1', {
                handler: 'foom',
                runtime: lambda.Runtime.NODEJS_10_X,
                code: directoryAsset
            });
            new lambda.Function(stack, 'Func2', {
                handler: 'foom',
                runtime: lambda.Runtime.NODEJS_10_X,
                code: directoryAsset
            });
            // THEN
            const assembly = app.synth();
            const synthesized = assembly.stacks[0];
            // Func1 has an asset, Func2 does not
            test.deepEqual(synthesized.assets.length, 1);
            test.done();
        },
        'adds code asset metadata'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
            const location = path.join(__dirname, 'my-lambda-handler');
            // WHEN
            new lambda.Function(stack, 'Func1', {
                code: lambda.Code.fromAsset(location),
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: 'foom',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Metadata: {
                    [cxapi.ASSET_RESOURCE_METADATA_PATH_KEY]: 'asset.9678c34eca93259d11f2d714177347afd66c50116e1e08996eff893d3ca81232',
                    [cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY]: 'Code'
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        }
    },
    'lambda.Code.fromCfnParameters': {
        "automatically creates the Bucket and Key parameters when it's used in a Function"(test) {
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode();
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: "FunctionLambdaSourceBucketNameParameter9E9E108F",
                    },
                    S3Key: {
                        Ref: "FunctionLambdaSourceObjectKeyParameter1C7AED11",
                    },
                },
            }));
            test.equal(stack.resolve(code.bucketNameParam), 'FunctionLambdaSourceBucketNameParameter9E9E108F');
            test.equal(stack.resolve(code.objectKeyParam), 'FunctionLambdaSourceObjectKeyParameter1C7AED11');
            test.done();
        },
        'does not allow accessing the Parameter properties before being used in a Function'(test) {
            const code = new lambda.CfnParametersCode();
            test.throws(() => {
                test.notEqual(code.bucketNameParam, undefined);
            }, /bucketNameParam/);
            test.throws(() => {
                test.notEqual(code.objectKeyParam, undefined);
            }, /objectKeyParam/);
            test.done();
        },
        'allows passing custom Parameters when creating it'(test) {
            const stack = new cdk.Stack();
            const bucketNameParam = new cdk.CfnParameter(stack, 'BucketNameParam', {
                type: 'String',
            });
            const bucketKeyParam = new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                type: 'String',
            });
            const code = lambda.Code.fromCfnParameters({
                bucketNameParam,
                objectKeyParam: bucketKeyParam,
            });
            test.equal(stack.resolve(code.bucketNameParam), 'BucketNameParam');
            test.equal(stack.resolve(code.objectKeyParam), 'ObjectKeyParam');
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: "BucketNameParam",
                    },
                    S3Key: {
                        Ref: "ObjectKeyParam",
                    },
                },
            }));
            test.done();
        },
        'can assign parameters'(test) {
            // given
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode({
                bucketNameParam: new cdk.CfnParameter(stack, 'BucketNameParam', {
                    type: 'String',
                }),
                objectKeyParam: new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                    type: 'String',
                }),
            });
            // when
            const overrides = stack.resolve(code.assign({
                bucketName: 'SomeBucketName',
                objectKey: 'SomeObjectKey',
            }));
            // then
            test.equal(overrides['BucketNameParam'], 'SomeBucketName');
            test.equal(overrides['ObjectKeyParam'], 'SomeObjectKey');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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