"use strict";
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit_1 = require("nodeunit");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'add incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS]
        });
        // THEN
        test.throws(() => func.addLayers(layer), /This lambda function uses a runtime that is incompatible with this layer/);
        test.done();
    },
    'add compatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.PYTHON_3_7]
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'add compatible layer for deep clone'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const runtime = lambda.Runtime.PYTHON_3_7;
        const func = new lambda.Function(stack, 'myFunc', {
            runtime,
            handler: 'index.handler',
            code,
        });
        const clone = _.cloneDeep(runtime);
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [clone]
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'empty inline code is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('')
        }), /Lambda inline code cannot be empty/);
        test.done();
    },
    'logGroup is correctly returned'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const logGroup = fn.logGroup;
        test.ok(logGroup.logGroupName);
        test.ok(logGroup.logGroupArn);
        test.done();
    },
    'dlq is returned when provided by user'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14)
        });
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueue: dlQueue,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueName);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueUrl);
        test.done();
    },
    'dlq is returned when setup by cdk'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueueEnabled: true,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueName);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueUrl);
        test.done();
    },
    'dlq is undefined when not setup'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === undefined);
        test.done();
    },
    'one and only one child LogRetention construct will be created'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            logRetention: logs.RetentionDays.FIVE_DAYS,
        });
        // tslint:disable:no-unused-expression
        // Call logGroup a few times. If more than one instance of LogRetention was created,
        // the second call will fail on duplicate constructs.
        fn.logGroup;
        fn.logGroup;
        fn.logGroup;
        // tslint:enable:no-unused-expression
        test.done();
    },
    'fails when inline code is specified on an incompatible runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.PROVIDED,
            code: lambda.Code.fromInline('foo')
        }), /Inline source not allowed for/);
        test.done();
    }
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5mdW5jdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuZnVuY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDBDQUEwQztBQUMxQyxzQ0FBc0M7QUFDdEMsd0NBQXdDO0FBQ3hDLHFDQUFxQztBQUNyQyw0QkFBNEI7QUFDNUIsdUNBQXdDO0FBQ3hDLGlDQUFpQztBQUVqQyxpQkFBUyxtQkFBUSxDQUFDO0lBQ2hCLHdCQUF3QixDQUFDLElBQVU7UUFDakMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDcEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM5QyxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBRXBELE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2hELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVU7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsSUFBSTtTQUNMLENBQUMsQ0FBQztRQUNILE1BQU0sS0FBSyxHQUFHLElBQUksTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ3RELElBQUk7WUFDSixrQkFBa0IsRUFBRSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDO1NBQzVDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEVBQ3JDLDBFQUEwRSxDQUFDLENBQUM7UUFFOUUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELHNCQUFzQixDQUFDLElBQVU7UUFDL0IsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDcEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM5QyxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBRXBELE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2hELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVU7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsSUFBSTtTQUNMLENBQUMsQ0FBQztRQUNILE1BQU0sS0FBSyxHQUFHLElBQUksTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ3RELElBQUk7WUFDSixrQkFBa0IsRUFBRSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO1NBQ2hELENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxtQkFBbUI7UUFDbkIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV0QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QscUNBQXFDLENBQUMsSUFBVTtRQUM5QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNwRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzlDLE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFFcEQsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUM7UUFDMUMsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEQsT0FBTztZQUNQLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLElBQUk7U0FDTCxDQUFDLENBQUM7UUFDSCxNQUFNLEtBQUssR0FBRyxDQUFDLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ25DLE1BQU0sS0FBSyxHQUFHLElBQUksTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ3RELElBQUk7WUFDSixrQkFBa0IsRUFBRSxDQUFDLEtBQUssQ0FBQztTQUM1QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsbUJBQW1CO1FBQ25CLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFdEIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtDQUFrQyxDQUFDLElBQVU7UUFDM0MsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLFlBQVk7UUFDWixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1lBQ2pELE9BQU8sRUFBRSxLQUFLO1lBQ2QsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDO1NBQ2pDLENBQUMsRUFBRSxvQ0FBb0MsQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxnQ0FBZ0MsQ0FBQyxJQUFVO1FBQ3pDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1lBQzFDLE9BQU8sRUFBRSxLQUFLO1lBQ2QsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1NBQ3BDLENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLEVBQUUsQ0FBQyxRQUFRLENBQUM7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDL0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDOUIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVDQUF1QyxDQUFDLElBQVU7UUFDaEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtZQUN0RCxTQUFTLEVBQUUsY0FBYztZQUN6QixlQUFlLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1NBQ3ZDLENBQUMsQ0FBQztRQUVILE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1lBQzFDLE9BQU8sRUFBRSxLQUFLO1lBQ2QsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ25DLGVBQWUsRUFBRSxPQUFPO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sZUFBZSxHQUFHLEVBQUUsQ0FBQyxlQUFlLENBQUM7UUFDM0MsSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLGFBQWYsZUFBZSx1QkFBZixlQUFlLENBQUUsUUFBUSxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLGFBQWYsZUFBZSx1QkFBZixlQUFlLENBQUUsU0FBUyxDQUFDLENBQUM7UUFDcEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLGFBQWYsZUFBZSx1QkFBZixlQUFlLENBQUUsUUFBUSxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1DQUFtQyxDQUFDLElBQVU7UUFDNUMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxFQUFFLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7WUFDMUMsT0FBTyxFQUFFLEtBQUs7WUFDZCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbkMsc0JBQXNCLEVBQUUsSUFBSTtTQUM3QixDQUFDLENBQUM7UUFDSCxNQUFNLGVBQWUsR0FBRyxFQUFFLENBQUMsZUFBZSxDQUFDO1FBQzNDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxhQUFmLGVBQWUsdUJBQWYsZUFBZSxDQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ25DLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxhQUFmLGVBQWUsdUJBQWYsZUFBZSxDQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxhQUFmLGVBQWUsdUJBQWYsZUFBZSxDQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ25DLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpQ0FBaUMsQ0FBQyxJQUFVO1FBQzFDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1lBQzFDLE9BQU8sRUFBRSxLQUFLO1lBQ2QsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1NBQ3BDLENBQUMsQ0FBQztRQUNILE1BQU0sZUFBZSxHQUFHLEVBQUUsQ0FBQyxlQUFlLENBQUM7UUFDM0MsSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLEtBQUssU0FBUyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtEQUErRCxDQUFDLElBQVU7UUFDeEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxFQUFFLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7WUFDMUMsT0FBTyxFQUFFLEtBQUs7WUFDZCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbkMsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztTQUMzQyxDQUFDLENBQUM7UUFFSCxzQ0FBc0M7UUFDdEMsb0ZBQW9GO1FBQ3BGLHFEQUFxRDtRQUNyRCxFQUFFLENBQUMsUUFBUSxDQUFDO1FBQ1osRUFBRSxDQUFDLFFBQVEsQ0FBQztRQUNaLEVBQUUsQ0FBQyxRQUFRLENBQUM7UUFDWixxQ0FBcUM7UUFFckMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdFQUFnRSxDQUFDLElBQVU7UUFDekUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtZQUNqRCxPQUFPLEVBQUUsS0FBSztZQUNkLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVE7WUFDaEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztTQUNwQyxDQUFDLEVBQUUsK0JBQStCLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgbG9ncyBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgc3FzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zcXMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgXyBmcm9tICdsb2Rhc2gnO1xuaW1wb3J0IHtUZXN0LCB0ZXN0Q2FzZX0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHRlc3RDYXNlKHtcbiAgJ2FkZCBpbmNvbXBhdGlibGUgbGF5ZXInKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsICdUZXN0U3RhY2snKTtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnQnVja2V0Jyk7XG4gICAgY29uc3QgY29kZSA9IG5ldyBsYW1iZGEuUzNDb2RlKGJ1Y2tldCwgJ09iamVjdEtleScpO1xuXG4gICAgY29uc3QgZnVuYyA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdteUZ1bmMnLCB7XG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM183LFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgY29kZSxcbiAgICB9KTtcbiAgICBjb25zdCBsYXllciA9IG5ldyBsYW1iZGEuTGF5ZXJWZXJzaW9uKHN0YWNrLCAnbXlMYXllcicsIHtcbiAgICAgIGNvZGUsXG4gICAgICBjb21wYXRpYmxlUnVudGltZXM6IFtsYW1iZGEuUnVudGltZS5OT0RFSlNdXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gZnVuYy5hZGRMYXllcnMobGF5ZXIpLFxuICAgICAgL1RoaXMgbGFtYmRhIGZ1bmN0aW9uIHVzZXMgYSBydW50aW1lIHRoYXQgaXMgaW5jb21wYXRpYmxlIHdpdGggdGhpcyBsYXllci8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdhZGQgY29tcGF0aWJsZSBsYXllcicodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgJ1Rlc3RTdGFjaycpO1xuICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdCdWNrZXQnKTtcbiAgICBjb25zdCBjb2RlID0gbmV3IGxhbWJkYS5TM0NvZGUoYnVja2V0LCAnT2JqZWN0S2V5Jyk7XG5cbiAgICBjb25zdCBmdW5jID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ215RnVuYycsIHtcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzcsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBjb2RlLFxuICAgIH0pO1xuICAgIGNvbnN0IGxheWVyID0gbmV3IGxhbWJkYS5MYXllclZlcnNpb24oc3RhY2ssICdteUxheWVyJywge1xuICAgICAgY29kZSxcbiAgICAgIGNvbXBhdGlibGVSdW50aW1lczogW2xhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzddXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgLy8gc2hvdWxkIG5vdCB0aHJvd1xuICAgIGZ1bmMuYWRkTGF5ZXJzKGxheWVyKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnYWRkIGNvbXBhdGlibGUgbGF5ZXIgZm9yIGRlZXAgY2xvbmUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsICdUZXN0U3RhY2snKTtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnQnVja2V0Jyk7XG4gICAgY29uc3QgY29kZSA9IG5ldyBsYW1iZGEuUzNDb2RlKGJ1Y2tldCwgJ09iamVjdEtleScpO1xuXG4gICAgY29uc3QgcnVudGltZSA9IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzc7XG4gICAgY29uc3QgZnVuYyA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdteUZ1bmMnLCB7XG4gICAgICBydW50aW1lLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgY29kZSxcbiAgICB9KTtcbiAgICBjb25zdCBjbG9uZSA9IF8uY2xvbmVEZWVwKHJ1bnRpbWUpO1xuICAgIGNvbnN0IGxheWVyID0gbmV3IGxhbWJkYS5MYXllclZlcnNpb24oc3RhY2ssICdteUxheWVyJywge1xuICAgICAgY29kZSxcbiAgICAgIGNvbXBhdGlibGVSdW50aW1lczogW2Nsb25lXVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIC8vIHNob3VsZCBub3QgdGhyb3dcbiAgICBmdW5jLmFkZExheWVycyhsYXllcik7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZW1wdHkgaW5saW5lIGNvZGUgaXMgbm90IGFsbG93ZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTi9USEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ2ZuJywge1xuICAgICAgaGFuZGxlcjogJ2ZvbycsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJycpXG4gICAgfSksIC9MYW1iZGEgaW5saW5lIGNvZGUgY2Fubm90IGJlIGVtcHR5Lyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2xvZ0dyb3VwIGlzIGNvcnJlY3RseSByZXR1cm5lZCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGZuID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ2ZuJywge1xuICAgICAgaGFuZGxlcjogJ2ZvbycsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2ZvbycpLFxuICAgIH0pO1xuICAgIGNvbnN0IGxvZ0dyb3VwID0gZm4ubG9nR3JvdXA7XG4gICAgdGVzdC5vayhsb2dHcm91cC5sb2dHcm91cE5hbWUpO1xuICAgIHRlc3Qub2sobG9nR3JvdXAubG9nR3JvdXBBcm4pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkbHEgaXMgcmV0dXJuZWQgd2hlbiBwcm92aWRlZCBieSB1c2VyJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBjb25zdCBkbFF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ0RlYWRMZXR0ZXJRdWV1ZScsIHtcbiAgICAgIHF1ZXVlTmFtZTogJ015TGFtYmRhX0RMUScsXG4gICAgICByZXRlbnRpb25QZXJpb2Q6IGNkay5EdXJhdGlvbi5kYXlzKDE0KVxuICAgIH0pO1xuXG4gICAgY29uc3QgZm4gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnZm4nLCB7XG4gICAgICBoYW5kbGVyOiAnZm9vJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnZm9vJyksXG4gICAgICBkZWFkTGV0dGVyUXVldWU6IGRsUXVldWUsXG4gICAgfSk7XG4gICAgY29uc3QgZGVhZExldHRlclF1ZXVlID0gZm4uZGVhZExldHRlclF1ZXVlO1xuICAgIHRlc3Qub2soZGVhZExldHRlclF1ZXVlPy5xdWV1ZUFybik7XG4gICAgdGVzdC5vayhkZWFkTGV0dGVyUXVldWU/LnF1ZXVlTmFtZSk7XG4gICAgdGVzdC5vayhkZWFkTGV0dGVyUXVldWU/LnF1ZXVlVXJsKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZGxxIGlzIHJldHVybmVkIHdoZW4gc2V0dXAgYnkgY2RrJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgZm4gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnZm4nLCB7XG4gICAgICBoYW5kbGVyOiAnZm9vJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnZm9vJyksXG4gICAgICBkZWFkTGV0dGVyUXVldWVFbmFibGVkOiB0cnVlLFxuICAgIH0pO1xuICAgIGNvbnN0IGRlYWRMZXR0ZXJRdWV1ZSA9IGZuLmRlYWRMZXR0ZXJRdWV1ZTtcbiAgICB0ZXN0Lm9rKGRlYWRMZXR0ZXJRdWV1ZT8ucXVldWVBcm4pO1xuICAgIHRlc3Qub2soZGVhZExldHRlclF1ZXVlPy5xdWV1ZU5hbWUpO1xuICAgIHRlc3Qub2soZGVhZExldHRlclF1ZXVlPy5xdWV1ZVVybCk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2RscSBpcyB1bmRlZmluZWQgd2hlbiBub3Qgc2V0dXAnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBmbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdmbicsIHtcbiAgICAgIGhhbmRsZXI6ICdmb28nLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCdmb28nKSxcbiAgICB9KTtcbiAgICBjb25zdCBkZWFkTGV0dGVyUXVldWUgPSBmbi5kZWFkTGV0dGVyUXVldWU7XG4gICAgdGVzdC5vayhkZWFkTGV0dGVyUXVldWUgPT09IHVuZGVmaW5lZCk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ29uZSBhbmQgb25seSBvbmUgY2hpbGQgTG9nUmV0ZW50aW9uIGNvbnN0cnVjdCB3aWxsIGJlIGNyZWF0ZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBmbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdmbicsIHtcbiAgICAgIGhhbmRsZXI6ICdmb28nLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCdmb28nKSxcbiAgICAgIGxvZ1JldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLkZJVkVfREFZUyxcbiAgICB9KTtcblxuICAgIC8vIHRzbGludDpkaXNhYmxlOm5vLXVudXNlZC1leHByZXNzaW9uXG4gICAgLy8gQ2FsbCBsb2dHcm91cCBhIGZldyB0aW1lcy4gSWYgbW9yZSB0aGFuIG9uZSBpbnN0YW5jZSBvZiBMb2dSZXRlbnRpb24gd2FzIGNyZWF0ZWQsXG4gICAgLy8gdGhlIHNlY29uZCBjYWxsIHdpbGwgZmFpbCBvbiBkdXBsaWNhdGUgY29uc3RydWN0cy5cbiAgICBmbi5sb2dHcm91cDtcbiAgICBmbi5sb2dHcm91cDtcbiAgICBmbi5sb2dHcm91cDtcbiAgICAvLyB0c2xpbnQ6ZW5hYmxlOm5vLXVudXNlZC1leHByZXNzaW9uXG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZmFpbHMgd2hlbiBpbmxpbmUgY29kZSBpcyBzcGVjaWZpZWQgb24gYW4gaW5jb21wYXRpYmxlIHJ1bnRpbWUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnZm4nLCB7XG4gICAgICBoYW5kbGVyOiAnZm9vJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBST1ZJREVELFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnZm9vJylcbiAgICB9KSwgL0lubGluZSBzb3VyY2Ugbm90IGFsbG93ZWQgZm9yLyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH1cbn0pO1xuIl19