"use strict";
const AWS = require("aws-sdk-mock");
const nock = require("nock");
const sinon = require("sinon");
const provider = require("../lib/log-retention-provider");
AWS.setSDK(require.resolve('aws-sdk'));
const eventCommon = {
    ServiceToken: 'token',
    ResponseURL: 'https://localhost',
    StackId: 'stackId',
    RequestId: 'requestId',
    LogicalResourceId: 'logicalResourceId',
    PhysicalResourceId: 'group',
    ResourceType: 'Custom::LogRetention',
};
const context = {
    functionName: 'provider'
};
function createRequest(type) {
    return nock('https://localhost')
        .put('/', (body) => body.Status === type && body.PhysicalResourceId === 'group')
        .reply(200);
}
module.exports = {
    'tearDown'(callback) {
        AWS.restore();
        nock.cleanAll();
        callback();
    },
    async 'create event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group'
            }
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group'
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 30
        });
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: '/aws/lambda/provider'
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: '/aws/lambda/provider',
            retentionInDays: 1
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with new log retention'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Update',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '365',
                LogGroupName: 'group'
            },
            OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '30'
            }
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group'
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 365
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with log retention undefined'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Update',
            PhysicalResourceId: 'group',
            ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group'
            },
            OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '365'
            }
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group'
        });
        sinon.assert.calledWith(deleteRetentionPolicyFake, {
            logGroupName: 'group'
        });
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'delete event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Delete',
            PhysicalResourceId: 'group',
            ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group'
            }
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.notCalled(createLogGroupFake);
        sinon.assert.notCalled(putRetentionPolicyFake);
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'responds with FAILED on error'(test) {
        const createLogGroupFake = sinon.fake.rejects(new Error('UnkownError'));
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group'
            }
        };
        const request = createRequest('FAILED');
        await provider.handler(event, context);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'does not fail when operations on provider log group fail'(test) {
        const createLogGroupFake = (params) => {
            if (params.logGroupName === '/aws/lambda/provider') {
                return Promise.reject(new Error('OperationAbortedException'));
            }
            return Promise.resolve({});
        };
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group'
            }
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'response data contains the log group name'(test) {
        AWS.mock('CloudWatchLogs', 'createLogGroup', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', sinon.fake.resolves({}));
        const event = {
            ...eventCommon,
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group'
            }
        };
        async function withOperation(operation) {
            const request = nock('https://localhost')
                .put('/', (body) => { var _a; return ((_a = body.Data) === null || _a === void 0 ? void 0 : _a.LogGroupName) === 'group'; })
                .reply(200);
            const opEvent = { ...event, RequestType: operation };
            await provider.handler(opEvent, context);
            test.equal(request.isDone(), true);
        }
        await withOperation('Create');
        await withOperation('Update');
        await withOperation('Delete');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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