"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaAtEdge = void 0;
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const retrieve_lambda_at_edge_1 = require("./retrieve-lambda-at-edge");
class LambdaAtEdge extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        const { account, stackName, region } = this.stack;
        const { role, code, name, parameterName = `/edge-lambda/${stackName}/${name}`, } = props;
        let lambdaAtEdgeStack = this.stack;
        if (region !== 'us-east-1') {
            const stackId = `lambda-at-edge-support-stack`;
            const app = this.requireApp();
            lambdaAtEdgeStack = app.node.tryFindChild(stackId);
            if (!lambdaAtEdgeStack) {
                lambdaAtEdgeStack = new core_1.Stack(app, stackId, {
                    env: {
                        account,
                        region: 'us-east-1',
                    },
                    synthesizer: this.getCrossRegionSupportSynthesizer(),
                });
                // the stack containing the edge lambdas must be deployed before
                this.stack.addDependency(lambdaAtEdgeStack);
            }
        }
        const lambdaAtEdgeExecutionRole = role !== null && role !== void 0 ? role : new aws_iam_1.Role(lambdaAtEdgeStack, 'LambdaAtEdgeExecutionRole', {
            assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal('edgelambda.amazonaws.com'), new aws_iam_1.ServicePrincipal('lambda.amazonaws.com')),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const lambdaFunction = new aws_lambda_1.Function(lambdaAtEdgeStack, name, {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code,
            role: lambdaAtEdgeExecutionRole,
        });
        this.parameter = new aws_ssm_1.StringParameter(lambdaAtEdgeStack, 'StringParameter', {
            parameterName,
            description: 'Parameter stored for cross region Lambda@Edge',
            stringValue: lambdaFunction.functionArn,
        });
    }
    retrieve(scope) {
        const { lambdaFunction } = new retrieve_lambda_at_edge_1.RetrieveLambdaAtEdge(scope, 'Retrieve', {
            parameter: this.parameter,
        });
        return lambdaFunction;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer
                    .cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
}
exports.LambdaAtEdge = LambdaAtEdge;
//# sourceMappingURL=data:application/json;base64,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