import numpy as np
from torch.utils.data import DataLoader, TensorDataset, Dataset
from torch import Tensor
from concurrent.futures import as_completed
from sklearn.model_selection import train_test_split
import torch
from concurrent.futures import wait
from os import listdir
from concurrent.futures import ThreadPoolExecutor
import os
import logging
from PIL import Image
from concurrent.futures import ThreadPoolExecutor
from os.path import isfile, join


class ZelinkaFractalDataset(Dataset):
    def __init__(self, files, labels, image_resolution=1024):
        self.files = files
        self.labels = labels.astype('float32')
        self.image_res = image_resolution

    def __len__(self):
        return len(self.files)

    def __getitem__(self, idx):
        if self.image_res != 1024:
            image = Image.open(self.files[idx]).resize((self.image_res, self.image_res))
        else:
            image = Image.open(self.files[idx])
        # make the image into a 3 channel image in numpy array
        image = np.array(image).reshape(3, image.size[0], image.size[1]).astype('float32')
        return image, np.argmax(self.labels[idx])


# Dataset from https://github.com/Zelinkai/FractalMalware
def zelinka_fractal(
        path_to_dataset: str = '',
        batch_size: int = 16,
        image_resolution: int = 1024,
        datasets=None
):
    # Validate that the parameters are within bounds
    if datasets is None:
        datasets = ['train', 'val', 'test']
    split = [0.7, 0.15, 0.15]
    if path_to_dataset == '':
        path_to_dataset = '/Datasets'
    if len(datasets) > 3 or len(datasets) < 1:
        raise Exception("datasets must be a list of length 1-3")
    if not np.in1d(datasets, ['train', 'val', 'test']).all():
        raise Exception("datasets must be a list of 'train', 'val', and/or 'test'")
    # specify the known labels of the dataset
    labels = ['malicious', 'benign']
    # define the loaders dictionary with the metadata of what will be loaded
    loaders = {
        'image_resolution': image_resolution,
        'batch_size': batch_size,
        'labels': labels,
        'input_channels': 3
    }
    path_to_dataset = path_to_dataset + '/Zelinka-Fractal'
    logging.info('--- BEGIN LOADING DATASET ---')
    features = []
    full_labels = []
    for index, label in enumerate(labels):
        if label == 'malicious':
            path_label = 'MW1'
        else:
            path_label = 'GW1'
        logging.info(f'LOADING DATASETSET: CLASS {index + 1} OF {len(labels)}')
        path = path_to_dataset + f'/{path_label}'
        # files = [f for f in listdir(path) if isfile(join(path, f))]
        files_full = []
        for path, subdirs, files in os.walk(path):
            files = [f for f in listdir(path) if isfile(join(path, f))]
            for name in files:
                if isfile(os.path.join(path, name)) and not name.startswith('.'):
                    files_full.append(os.path.join(path, name))
        files_np = [np.asarray(img) for img in files_full]
        features.extend(files_np)
        full_labels.extend([index for i in range(len(files_np))])
    # Resize the data and check that the samples are in the correct size
    logging.info(f'--- DATASET LOADING COMPLETE ---')
    logging.info(f'--- BEGIN DATASET PREPROCESSING ---')
    features = np.asarray([feature for feature in features])
    full_labels = np.array([label for label in full_labels])
    # Shuffles the dataset
    np.random.seed(10)
    idx = np.random.choice(len(full_labels), len(full_labels))
    full_labels = full_labels[idx]
    features = features[idx]

    full_labels = full_labels.reshape((len(full_labels), 1))
    X_train = features
    new_y_train = np.zeros((len(X_train), len(labels)))
    # Set the labels value to the index of the classes
    for ind, val in enumerate(full_labels):
        tmp = np.zeros(len(labels))
        tmp[val] = 1
        new_y_train[ind] = tmp
    Y_train = new_y_train
    # make three dataloaders with the train, val, and test sets
    train_split_size = split[0]
    val_split_size = split[1]
    test_split_size = split[2]
    # get the first split size of data and put into an array
    x_train = X_train[:int(train_split_size * len(X_train))]
    y_train = Y_train[:int(train_split_size * len(Y_train))]
    # get the second split size of data and put into an array
    x_val = X_train[int(train_split_size * len(X_train)):int((train_split_size + val_split_size) * len(X_train))]
    y_val = Y_train[int(train_split_size * len(Y_train)):int((train_split_size + val_split_size) * len(Y_train))]
    # get the third split size of data and put into an array
    x_test = X_train[int((train_split_size + val_split_size) * len(X_train)):]
    y_test = Y_train[int((train_split_size + val_split_size) * len(Y_train)):]

    train = ZelinkaFractalDataset(files=x_train, labels=y_train, image_resolution=image_resolution)
    val = ZelinkaFractalDataset(files=x_val, labels=y_val, image_resolution=image_resolution)
    test = ZelinkaFractalDataset(files=x_test, labels=y_test, image_resolution=image_resolution)

    # create the loaders dictionary
    if 'train' in datasets:
        loaders['train'] = DataLoader(train,
                                      batch_size=batch_size,
                                      shuffle=False)
    if 'val' in datasets:
        loaders['val'] = DataLoader(val,
                                    batch_size=batch_size,
                                    shuffle=False)
    if 'test' in datasets:
        loaders['test'] = DataLoader(test,
                                     batch_size=batch_size,
                                     shuffle=False)
    logging.info(f'--- DATASET PREPROCESSING COMPLETE ---')
    return loaders

