"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerVolumeConsistency = exports.BundlingDockerImage = void 0;
const child_process_1 = require("child_process");
/**
 * A Docker image used for asset bundling
 */
class BundlingDockerImage {
    /** @param image The Docker image */
    constructor(image) {
        this.image = image;
    }
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name
     */
    static fromRegistry(image) {
        return new BundlingDockerImage(image);
    }
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file
     * @param options Docker build options
     */
    static fromAsset(path, options = {}) {
        const buildArgs = options.buildArgs || {};
        const dockerArgs = [
            'build', '-q',
            ...flatten(Object.entries(buildArgs).map(([k, v]) => ['--build-arg', `${k}=${v}`])),
            path,
        ];
        const docker = dockerExec(dockerArgs);
        const match = docker.stdout.toString().match(/sha256:([a-z0-9]+)/);
        if (!match) {
            throw new Error('Failed to extract image ID from Docker build output');
        }
        return new BundlingDockerImage(match[1]);
    }
    /**
     * Runs a Docker image
     *
     * @internal
     */
    _run(options = {}) {
        const volumes = options.volumes || [];
        const environment = options.environment || {};
        const command = options.command || [];
        const dockerArgs = [
            'run', '--rm',
            ...options.user
                ? ['-u', options.user]
                : [],
            ...flatten(volumes.map(v => { var _a; return ['-v', `${v.hostPath}:${v.containerPath}:${(_a = v.consistency) !== null && _a !== void 0 ? _a : DockerVolumeConsistency.DELEGATED}`]; })),
            ...flatten(Object.entries(environment).map(([k, v]) => ['--env', `${k}=${v}`])),
            ...options.workingDirectory
                ? ['-w', options.workingDirectory]
                : [],
            this.image,
            ...command,
        ];
        dockerExec(dockerArgs, {
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
        });
    }
}
exports.BundlingDockerImage = BundlingDockerImage;
/**
 * Supported Docker volume consistency types. Only valid on macOS due to the way file storage works on Mac
 */
var DockerVolumeConsistency;
(function (DockerVolumeConsistency) {
    /**
     * Read/write operations inside the Docker container are applied immediately on the mounted host machine volumes
     */
    DockerVolumeConsistency["CONSISTENT"] = "consistent";
    /**
     * Read/write operations on mounted Docker volumes are first written inside the container and then synchronized to the host machine
     */
    DockerVolumeConsistency["DELEGATED"] = "delegated";
    /**
     * Read/write operations on mounted Docker volumes are first applied on the host machine and then synchronized to the container
     */
    DockerVolumeConsistency["CACHED"] = "cached";
})(DockerVolumeConsistency = exports.DockerVolumeConsistency || (exports.DockerVolumeConsistency = {}));
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
function dockerExec(args, options) {
    var _a, _b, _c;
    const prog = (_a = process.env.CDK_DOCKER) !== null && _a !== void 0 ? _a : 'docker';
    const proc = child_process_1.spawnSync(prog, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_b = proc.stdout) === null || _b === void 0 ? void 0 : _b.toString().trim()}\n\n\nstderr: ${(_c = proc.stderr) === null || _c === void 0 ? void 0 : _c.toString().trim()}`);
        }
        throw new Error(`${prog} exited with status ${proc.status}`);
    }
    return proc;
}
//# sourceMappingURL=data:application/json;base64,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