"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk', '@aws-solutions-constructs'];
// list of NPM packages included in version reporting
const WHITELIST_PACKAGES = ['aws-rfdk'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        foundMatch = foundMatch || WHITELIST_PACKAGES.includes(name);
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!(mod === null || mod === void 0 ? void 0 : mod.paths)) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod === null || mod === void 0 ? void 0 : mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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