"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RemoveTag = exports.Tags = exports.Tag = void 0;
// import * as cxapi from '@aws-cdk/cx-api';
const aspect_1 = require("./aspect");
const tag_manager_1 = require("./tag-manager");
/**
 * The common functionality for Tag and Remove Tag Aspects
 */
class TagBase {
    constructor(key, props = {}) {
        this.key = key;
        this.props = props;
    }
    visit(construct) {
        if (tag_manager_1.TagManager.isTaggable(construct)) {
            this.applyTag(construct);
        }
    }
}
/**
 * The Tag Aspect will handle adding a tag to this node and cascading tags to children
 */
class Tag extends TagBase {
    constructor(key, value, props = {}) {
        super(key, props);
        this.defaultPriority = 100;
        if (value === undefined) {
            throw new Error('Tag must have a value');
        }
        this.value = value;
    }
    /**
     * DEPRECATED: add tags to the node of a construct and all its the taggable children
     *
     * @deprecated use `Tags.of(scope).add()`
     */
    static add(scope, key, value, props = {}) {
        Tags.of(scope).add(key, value, props);
    }
    /**
     * DEPRECATED: remove tags to the node of a construct and all its the taggable children
     *
     * @deprecated use `Tags.of(scope).remove()`
     */
    static remove(scope, key, props = {}) {
        Tags.of(scope).remove(key, props);
    }
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.setTag(this.key, this.value, this.props.priority !== undefined ? this.props.priority : this.defaultPriority, this.props.applyToLaunchedInstances !== false);
        }
    }
}
exports.Tag = Tag;
/**
 * Manages AWS tags for all resources within a construct scope.
 */
class Tags {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the tags API for this scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Tags(scope);
    }
    /**
     * add tags to the node of a construct and all its the taggable children
     */
    add(key, value, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new Tag(key, value, props));
    }
    /**
     * remove tags to the node of a construct and all its the taggable children
     */
    remove(key, props = {}) {
        aspect_1.Aspects.of(this.scope).add(new RemoveTag(key, props));
    }
}
exports.Tags = Tags;
/**
 * The RemoveTag Aspect will handle removing tags from this node and children
 */
class RemoveTag extends TagBase {
    constructor(key, props = {}) {
        super(key, props);
        this.defaultPriority = 200;
    }
    applyTag(resource) {
        if (resource.tags.applyTagAspectHere(this.props.includeResourceTypes, this.props.excludeResourceTypes)) {
            resource.tags.removeTag(this.key, this.props.priority !== undefined ? this.props.priority : this.defaultPriority);
        }
    }
}
exports.RemoveTag = RemoveTag;
//# sourceMappingURL=data:application/json;base64,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