"use strict";
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const lib_1 = require("../lib");
const annotations_1 = require("../lib/annotations");
const util_1 = require("./util");
function createTree(context) {
    const root = new lib_1.App();
    const highChild = new lib_1.Construct(root, 'HighChild');
    if (context) {
        Object.keys(context).forEach(key => highChild.node.setContext(key, context[key]));
    }
    const child1 = new lib_1.Construct(highChild, 'Child1');
    const child2 = new lib_1.Construct(highChild, 'Child2');
    const child1_1 = new lib_1.Construct(child1, 'Child11');
    const child1_2 = new lib_1.Construct(child1, 'Child12');
    const child1_1_1 = new lib_1.Construct(child1_1, 'Child111');
    const child2_1 = new lib_1.Construct(child2, 'Child21');
    return {
        root, child1, child2, child1_1, child1_2, child1_1_1, child2_1,
    };
}
class MyBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
}
/**
 * Returns a string with a tree representation of this construct and it's children.
 */
function toTreeString(node, depth = 0) {
    let out = '';
    for (let i = 0; i < depth; ++i) {
        out += '  ';
    }
    const name = node.node.id || '';
    out += `${node.constructor.name}${name.length > 0 ? ' [' + name + ']' : ''}\n`;
    for (const child of node.node.children) {
        out += toTreeString(child, depth + 1);
    }
    return out;
}
module.exports = {
    'the "Root" construct is a special construct which can be used as the root of the tree'(test) {
        const root = new lib_1.App();
        test.equal(root.node.id, '', 'if not specified, name of a root construct is an empty string');
        test.ok(!root.node.scope, 'no parent');
        test.equal(root.node.children.length, 1);
        test.done();
    },
    'constructs cannot be created with an empty name unless they are root'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.Construct(root, ''));
        test.done();
    },
    'construct.name returns the name of the construct'(test) {
        const t = createTree();
        test.equal(t.child1.node.id, 'Child1');
        test.equal(t.child2.node.id, 'Child2');
        test.equal(t.child1_1.node.id, 'Child11');
        test.equal(t.child1_2.node.id, 'Child12');
        test.equal(t.child1_1_1.node.id, 'Child111');
        test.equal(t.child2_1.node.id, 'Child21');
        test.done();
    },
    'construct id can use any character except the path separator'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'valid');
        new lib_1.Construct(root, 'ValiD');
        new lib_1.Construct(root, 'Va123lid');
        new lib_1.Construct(root, 'v');
        new lib_1.Construct(root, '  invalid');
        new lib_1.Construct(root, 'invalid   ');
        new lib_1.Construct(root, '123invalid');
        new lib_1.Construct(root, 'in valid');
        new lib_1.Construct(root, 'in_Valid');
        new lib_1.Construct(root, 'in-Valid');
        new lib_1.Construct(root, 'in\\Valid');
        new lib_1.Construct(root, 'in.Valid');
        test.done();
    },
    'if construct id contains path seperators, they will be replaced by double-dash'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Boom/Boom/Bam');
        test.deepEqual(c.node.id, 'Boom--Boom--Bam');
        test.done();
    },
    'if "undefined" is forcefully used as an "id", it will be treated as an empty string'(test) {
        const c = new lib_1.Construct(undefined, undefined);
        test.deepEqual(c.node.id, '');
        test.done();
    },
    'dont allow unresolved tokens to be used in construct IDs'(test) {
        // GIVEN
        const root = new lib_1.App();
        const token = lib_1.Lazy.stringValue({ produce: () => 'lazy' });
        // WHEN + THEN
        test.throws(() => new lib_1.Construct(root, `MyID: ${token}`), /Cannot use tokens in construct ID: MyID: \${Token/);
        test.done();
    },
    'construct.uniqueId returns a tree-unique alphanumeric id of this construct'(test) {
        const root = new lib_1.App();
        const child1 = new lib_1.Construct(root, 'This is the first child');
        const child2 = new lib_1.Construct(child1, 'Second level');
        const c1 = new lib_1.Construct(child2, 'My construct');
        const c2 = new lib_1.Construct(child1, 'My construct');
        test.deepEqual(c1.node.path, 'This is the first child/Second level/My construct');
        test.deepEqual(c2.node.path, 'This is the first child/My construct');
        test.deepEqual(c1.node.uniqueId, 'ThisisthefirstchildSecondlevelMyconstruct202131E0');
        test.deepEqual(c2.node.uniqueId, 'ThisisthefirstchildMyconstruct8C288DF9');
        test.done();
    },
    'cannot calculate uniqueId if the construct path is ["Default"]'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Default');
        test.throws(() => c.node.uniqueId, /Unable to calculate a unique id for an empty set of components/);
        test.done();
    },
    'construct.getChildren() returns an array of all children'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Child1');
        new lib_1.Construct(root, 'Child2');
        test.equal(child.node.children.length, 0, 'no children');
        test.equal(root.node.children.length, 3, 'three children are expected');
        test.done();
    },
    'construct.findChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.tryFindChild(child.node.id), child, 'findChild(name) can be used to retrieve the child from a parent');
        test.ok(!root.node.tryFindChild('NotFound'), 'findChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.getChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.findChild(child.node.id), child, 'getChild(name) can be used to retrieve the child from a parent');
        test.throws(() => {
            root.node.findChild('NotFound');
        }, '', 'getChild(name) returns undefined if the child is not found');
        test.done();
    },
    'can remove children from the tree using tryRemoveChild()'(test) {
        const root = new lib_1.App();
        const childrenBeforeAdding = root.node.children.length; // Invariant to adding 'Metadata' resource or not
        // Add & remove
        const child = new lib_1.Construct(root, 'Construct');
        test.equals(true, root.node.tryRemoveChild(child.node.id));
        test.equals(false, root.node.tryRemoveChild(child.node.id)); // Second time does nothing
        test.equals(undefined, root.node.tryFindChild(child.node.id));
        test.equals(childrenBeforeAdding, root.node.children.length);
        test.done();
    },
    'construct.toString() and construct.toTreeString() can be used for diagnostics'(test) {
        const t = createTree();
        test.equal(t.root.toString(), '<root>');
        test.equal(t.child1_1_1.toString(), 'HighChild/Child1/Child11/Child111');
        test.equal(t.child2.toString(), 'HighChild/Child2');
        test.equal(toTreeString(t.root), 'App\n  TreeMetadata [Tree]\n  Construct [HighChild]\n    Construct [Child1]\n      Construct [Child11]\n        Construct [Child111]\n      Construct [Child12]\n    Construct [Child2]\n      Construct [Child21]\n');
        test.done();
    },
    'construct.getContext(key) can be used to read a value from context defined at the root level'(test) {
        const context = {
            ctx1: 12,
            ctx2: 'hello',
        };
        const t = createTree(context);
        test.equal(t.child1_2.node.tryGetContext('ctx1'), 12);
        test.equal(t.child1_1_1.node.tryGetContext('ctx2'), 'hello');
        test.done();
    },
    // eslint-disable-next-line max-len
    'construct.setContext(k,v) sets context at some level and construct.getContext(key) will return the lowermost value defined in the stack'(test) {
        const root = new lib_1.App();
        const highChild = new lib_1.Construct(root, 'highChild');
        highChild.node.setContext('c1', 'root');
        highChild.node.setContext('c2', 'root');
        const child1 = new lib_1.Construct(highChild, 'child1');
        child1.node.setContext('c2', 'child1');
        child1.node.setContext('c3', 'child1');
        const child2 = new lib_1.Construct(highChild, 'child2');
        const child3 = new lib_1.Construct(child1, 'child1child1');
        child3.node.setContext('c1', 'child3');
        child3.node.setContext('c4', 'child3');
        test.equal(highChild.node.tryGetContext('c1'), 'root');
        test.equal(highChild.node.tryGetContext('c2'), 'root');
        test.equal(highChild.node.tryGetContext('c3'), undefined);
        test.equal(child1.node.tryGetContext('c1'), 'root');
        test.equal(child1.node.tryGetContext('c2'), 'child1');
        test.equal(child1.node.tryGetContext('c3'), 'child1');
        test.equal(child2.node.tryGetContext('c1'), 'root');
        test.equal(child2.node.tryGetContext('c2'), 'root');
        test.equal(child2.node.tryGetContext('c3'), undefined);
        test.equal(child3.node.tryGetContext('c1'), 'child3');
        test.equal(child3.node.tryGetContext('c2'), 'child1');
        test.equal(child3.node.tryGetContext('c3'), 'child1');
        test.equal(child3.node.tryGetContext('c4'), 'child3');
        test.done();
    },
    'construct.setContext(key, value) can only be called before adding any children'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'child1');
        test.throws(() => root.node.setContext('k', 'v'));
        test.done();
    },
    'fails if context key contains unresolved tokens'(test) {
        const root = new lib_1.App();
        test.throws(() => root.node.setContext(`my-${lib_1.Aws.REGION}`, 'foo'), /Invalid context key/);
        test.throws(() => root.node.tryGetContext(lib_1.Aws.REGION), /Invalid context key/);
        test.done();
    },
    'construct.pathParts returns an array of strings of all names from root to node'(test) {
        const tree = createTree();
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'HighChild/Child1/Child11/Child111');
        test.deepEqual(tree.child2.node.path, 'HighChild/Child2');
        test.done();
    },
    'if a root construct has a name, it should be included in the path'(test) {
        const tree = createTree({});
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'HighChild/Child1/Child11/Child111');
        test.done();
    },
    'construct can not be created with the name of a sibling'(test) {
        const root = new lib_1.App();
        // WHEN
        new lib_1.Construct(root, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(root, 'SameName');
        }, /There is already a Construct with name 'SameName' in App/);
        // WHEN
        const c0 = new lib_1.Construct(root, 'c0');
        new lib_1.Construct(c0, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(c0, 'SameName');
        }, /There is already a Construct with name 'SameName' in Construct \[c0\]/);
        test.done();
    },
    'addMetadata(type, data) can be used to attach metadata to constructs FIND_ME'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        test.deepEqual(con.node.metadata, [], 'starts empty');
        con.node.addMetadata('key', 'value');
        con.node.addMetadata('number', 103);
        con.node.addMetadata('array', [123, 456]);
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, 'key');
        test.deepEqual(con.node.metadata[0].data, 'value');
        test.deepEqual(con.node.metadata[1].data, 103);
        test.deepEqual(con.node.metadata[2].data, [123, 456]);
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace[1].indexOf('FIND_ME') !== -1, 'First stack line should include this function\s name');
        test.done();
    },
    'addMetadata(type, undefined/null) is ignored'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'Foo');
        con.node.addMetadata('Null', null);
        con.node.addMetadata('Undefined', undefined);
        con.node.addMetadata('True', true);
        con.node.addMetadata('False', false);
        con.node.addMetadata('Empty', '');
        const exists = (key) => con.node.metadata.find(x => x.type === key);
        test.ok(!exists('Null'));
        test.ok(!exists('Undefined'));
        test.ok(exists('True'));
        test.ok(exists('False'));
        test.ok(exists('Empty'));
        test.done();
    },
    'addWarning(message) can be used to add a "WARNING" message entry to the construct'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        annotations_1.Annotations.of(con).addWarning('This construct is deprecated, use the other one instead');
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.WARN);
        test.deepEqual(con.node.metadata[0].data, 'This construct is deprecated, use the other one instead');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addError(message) can be used to add a "ERROR" message entry to the construct'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        annotations_1.Annotations.of(con).addError('Stop!');
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.ERROR);
        test.deepEqual(con.node.metadata[0].data, 'Stop!');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addInfo(message) can be used to add an "INFO" message entry to the construct'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        annotations_1.Annotations.of(con).addInfo('Hey there, how do you do?');
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.INFO);
        test.deepEqual(con.node.metadata[0].data, 'Hey there, how do you do?');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'multiple children of the same type, with explicit names are welcome'(test) {
        const root = new lib_1.App();
        new MyBeautifulConstruct(root, 'mbc1');
        new MyBeautifulConstruct(root, 'mbc2');
        new MyBeautifulConstruct(root, 'mbc3');
        new MyBeautifulConstruct(root, 'mbc4');
        test.ok(root.node.children.length >= 4);
        test.done();
    },
    // eslint-disable-next-line max-len
    'construct.validate() can be implemented to perform validation, ConstructNode.validate(construct.node) will return all errors from the subtree (DFS)'(test) {
        class MyConstruct extends lib_1.Construct {
            validate() {
                return ['my-error1', 'my-error2'];
            }
        }
        class YourConstruct extends lib_1.Construct {
            validate() {
                return ['your-error1'];
            }
        }
        class TheirConstruct extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new YourConstruct(this, 'YourConstruct');
            }
            validate() {
                return ['their-error'];
            }
        }
        class TestStack extends lib_1.App {
            constructor() {
                super();
                new MyConstruct(this, 'MyConstruct');
                new TheirConstruct(this, 'TheirConstruct');
            }
            validate() {
                return ['stack-error'];
            }
        }
        const stack = new TestStack();
        const errors = lib_1.ConstructNode.validate(stack.node).map((v) => ({ path: v.source.node.path, message: v.message }));
        // validate DFS
        test.deepEqual(errors, [
            { path: 'MyConstruct', message: 'my-error1' },
            { path: 'MyConstruct', message: 'my-error2' },
            { path: 'TheirConstruct/YourConstruct', message: 'your-error1' },
            { path: 'TheirConstruct', message: 'their-error' },
            { path: '', message: 'stack-error' },
        ]);
        test.done();
    },
    'construct.lock() protects against adding children anywhere under this construct (direct or indirect)'(test) {
        class LockableConstruct extends lib_1.Construct {
            lockMe() {
                this.node._actualNode._lock();
            }
            unlockMe() {
                this.node._actualNode._unlock();
            }
        }
        const stack = new lib_1.App();
        const c0a = new LockableConstruct(stack, 'c0a');
        const c0b = new lib_1.Construct(stack, 'c0b');
        const c1a = new lib_1.Construct(c0a, 'c1a');
        const c1b = new lib_1.Construct(c0a, 'c1b');
        c0a.lockMe();
        // now we should still be able to add children to c0b, but not to c0a or any its children
        new lib_1.Construct(c0b, 'c1a');
        test.throws(() => new lib_1.Construct(c0a, 'fail1'), /Cannot add children to "c0a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1a, 'fail2'), /Cannot add children to "c0a\/c1a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1b, 'fail3'), /Cannot add children to "c0a\/c1b" during synthesis/);
        c0a.unlockMe();
        new lib_1.Construct(c0a, 'c0aZ');
        new lib_1.Construct(c1a, 'c1aZ');
        new lib_1.Construct(c1b, 'c1bZ');
        test.done();
    },
    'findAll returns a list of all children in either DFS or BFS'(test) {
        // GIVEN
        const c1 = new lib_1.Construct(undefined, '1');
        const c2 = new lib_1.Construct(c1, '2');
        new lib_1.Construct(c1, '3');
        new lib_1.Construct(c2, '4');
        new lib_1.Construct(c2, '5');
        // THEN
        test.deepEqual(c1.node.findAll().map(x => x.node.id), c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id)); // default is PreOrder
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id), ['1', '2', '4', '5', '3']);
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.POSTORDER).map(x => x.node.id), ['4', '5', '2', '3', '1']);
        test.done();
    },
    'ancestors returns a list of parents up to root'(test) {
        const { child1_1_1 } = createTree();
        test.deepEqual(child1_1_1.node.scopes.map(x => x.node.id), ['', 'HighChild', 'Child1', 'Child11', 'Child111']);
        test.done();
    },
    '"root" returns the root construct'(test) {
        const { child1, child2, child1_1_1, root } = createTree();
        test.ok(child1.node.root === root);
        test.ok(child2.node.root === root);
        test.ok(child1_1_1.node.root === root);
        test.done();
    },
    defaultChild: {
        'returns the child with id "Resource"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Resource');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns the child with id "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'can override defaultChild'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'Resource');
            const defaultChild = new lib_1.Construct(root, 'OtherResource');
            root.node.defaultChild = defaultChild;
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns "undefined" if there is no default'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'child2');
            test.equal(root.node.defaultChild, undefined);
            test.done();
        },
        'fails if there are both "Resource" and "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            new lib_1.Construct(root, 'Resource');
            test.throws(() => root.node.defaultChild, /Cannot determine default child for . There is both a child with id "Resource" and id "Default"/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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