"use strict";
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const sinon = require("sinon");
const lib_1 = require("../lib");
const STUB_INPUT_FILE = '/tmp/docker-stub.input';
var DockerStubCommand;
(function (DockerStubCommand) {
    DockerStubCommand["SUCCESS"] = "DOCKER_STUB_SUCCESS";
    DockerStubCommand["FAIL"] = "DOCKER_STUB_FAIL";
    DockerStubCommand["SUCCESS_NO_OUTPUT"] = "DOCKER_STUB_SUCCESS_NO_OUTPUT";
})(DockerStubCommand || (DockerStubCommand = {}));
const userInfo = os.userInfo();
const USER_ARG = `-u ${userInfo.uid}:${userInfo.gid}`;
// this is a way to provide a custom "docker" command for staging.
process.env.CDK_DOCKER = `${__dirname}/docker-stub.sh`;
function readDockerStubInput() {
    const out = fs.readFileSync(STUB_INPUT_FILE, 'utf-8').trim();
    return out
        .replace(/-v ([^:]+):\/asset-input/, '-v /input:/asset-input')
        .replace(/-v ([^:]+):\/asset-output/, '-v /output:/asset-output');
}
module.exports = {
    'tearDown'(cb) {
        if (fs.existsSync(STUB_INPUT_FILE)) {
            fs.unlinkSync(STUB_INPUT_FILE);
        }
        cb();
        sinon.restore();
    },
    'base case'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), 'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.done();
    },
    'staging can be disabled through context'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.node.setContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT, true);
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), sourcePath);
        test.done();
    },
    'files are copied to the output directory during synth'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const file = path.join(__dirname, 'fs', 'fixtures.tar.gz');
        // WHEN
        new lib_1.AssetStaging(stack, 's1', { sourcePath: directory });
        new lib_1.AssetStaging(stack, 'file', { sourcePath: file });
        // THEN
        const assembly = app.synth();
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00',
            'asset.af10ac04b3b607b0f8659c8f0cee8c343025ee75baf0b146f10f0e5311d2c46b.gz',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'allow specifying extra data to include in the source hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const withoutExtra = new lib_1.AssetStaging(stack, 'withoutExtra', { sourcePath: directory });
        const withExtra = new lib_1.AssetStaging(stack, 'withExtra', { sourcePath: directory, extraHash: 'boom' });
        // THEN
        test.notEqual(withoutExtra.sourceHash, withExtra.sourceHash);
        test.deepEqual(withoutExtra.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(withExtra.sourceHash, 'c95c915a5722bb9019e2c725d11868e5a619b55f36172f76bcbcaa8bb2d10c5f');
        test.done();
    },
    'with bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const ensureDirSyncSpy = sinon.spy(fs, 'ensureDirSync');
        const mkdtempSyncSpy = sinon.spy(fs, 'mkdtempSync');
        const chmodSyncSpy = sinon.spy(fs, 'chmodSync');
        const processStdErrWriteSpy = sinon.spy(process.stderr, 'write');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        // asset is bundled in a directory inside .cdk.staging
        const stagingTmp = path.join('.', '.cdk.staging');
        test.ok(ensureDirSyncSpy.calledWith(stagingTmp));
        test.ok(mkdtempSyncSpy.calledWith(sinon.match(path.join(stagingTmp, 'asset-bundle-'))));
        test.ok(chmodSyncSpy.calledWith(sinon.match(path.join(stagingTmp, 'asset-bundle-')), 0o777));
        // shows a message before bundling
        test.ok(processStdErrWriteSpy.calledWith('Bundling asset stack/Asset...\n'));
        test.done();
    },
    'bundling throws when /asset-ouput is empty'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS_NO_OUTPUT],
            },
        }), /Bundling did not produce any output/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS_NO_OUTPUT`);
        test.done();
    },
    'bundling with BUNDLE asset hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHashType: lib_1.AssetHashType.BUNDLE,
        });
        // THEN
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.equal(asset.assetHash, '33cbf2cae5432438e0f046bc45ba8c3cef7b6afcf47b59d1c183775c1918fb1f');
        test.done();
    },
    'custom hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHash: 'my-custom-hash',
        });
        // THEN
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.equal(asset.assetHash, 'b9c77053f5b83bbe5ba343bc18e92db939a49017010813225fea91fa892c4823'); // hash of 'my-custom-hash'
        test.done();
    },
    'throws with assetHash and not CUSTOM hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHash: 'my-custom-hash',
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot specify `bundle` for `assetHashType`/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.done();
    },
    'throws with BUNDLE hash type and no bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.done();
    },
    'throws with CUSTOM and no hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.CUSTOM,
        }), /`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false); // "docker" not executed
        test.done();
    },
    'throws when bundling fails'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('this-is-an-invalid-docker-image'),
                command: [DockerStubCommand.FAIL],
            },
        }), /Failed to bundle asset stack\/Asset/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input:delegated -v /output:/asset-output:delegated -w /asset-input this-is-an-invalid-docker-image DOCKER_STUB_FAIL`);
        test.done();
    },
    'with local bundling'(test) {
        var _a;
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        let dir;
        let opts;
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
                local: {
                    tryBundle(outputDir, options) {
                        dir = outputDir;
                        opts = options;
                        fs.writeFileSync(path.join(outputDir, 'hello.txt'), 'hello'); // output cannot be empty
                        return true;
                    },
                },
            },
        });
        // THEN
        test.ok(dir && /asset-bundle-/.test(dir));
        test.equals((_a = opts === null || opts === void 0 ? void 0 : opts.command) === null || _a === void 0 ? void 0 : _a[0], DockerStubCommand.SUCCESS);
        test.throws(() => readDockerStubInput());
        if (dir) {
            fs.removeSync(path.join(dir, 'hello.txt'));
        }
        test.done();
    },
    'with local bundling returning false'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
                local: {
                    tryBundle(_bundleDir) {
                        return false;
                    },
                },
            },
        });
        // THEN
        test.ok(readDockerStubInput());
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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