"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_1 = require("./construct");
const runtime_info_1 = require("./private/runtime-info");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps_and_stacks.html
 */
class App extends construct_1.Construct {
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        super(undefined, '');
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        // both are reverse logic
        this.runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? false : true;
        this.outdir = props.outdir || process.env[cxapi.OUTDIR_ENV];
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
    }
    /**
     * Synthesizes a cloud assembly for this app. Emits it to the directory
     * specified by `outdir`.
     *
     * @returns a `CloudAssembly` which can be used to inspect synthesized
     * artifacts such as CloudFormation templates and assets.
     */
    synth() {
        // we already have a cloud assembly, no-op for you
        if (this._assembly) {
            return this._assembly;
        }
        const assembly = construct_1.ConstructNode.synth(this.node, {
            outdir: this.outdir,
            runtimeInfo: this.runtimeInfo ? runtime_info_1.collectRuntimeInformation() : undefined
        });
        this._assembly = assembly;
        return assembly;
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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