"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.props = props;
        this.type = props.type || 'String';
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this.props.noEcho;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.props.default,
                    AllowedPattern: this.props.allowedPattern,
                    AllowedValues: this.props.allowedValues,
                    ConstraintDescription: this.props.constraintDescription,
                    Description: this.props.description,
                    MaxLength: this.props.maxLength,
                    MaxValue: this.props.maxValue,
                    MinLength: this.props.minLength,
                    MinValue: this.props.minValue,
                    NoEcho: this.props.noEcho
                }
            }
        };
    }
    resolve(_context) {
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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