"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A set of constructs to be used as a dependable
 *
 * This class can be used when a set of constructs which are disjoint in the
 * construct tree needs to be combined to be used as a single dependable.
 */
class ConcreteDependable {
    constructor() {
        this._dependencyRoots = new Array();
        const self = this;
        DependableTrait.implement(this, {
            get dependencyRoots() { return self._dependencyRoots; },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(construct) {
        this._dependencyRoots.push(construct);
    }
}
exports.ConcreteDependable = ConcreteDependable;
const DEPENDABLE_SYMBOL = Symbol.for('@aws-cdk/core.DependableTrait');
/**
 * Trait for IDependable
 *
 * Traits are interfaces that are privately implemented by objects. Instead of
 * showing up in the public interface of a class, they need to be queried
 * explicitly. This is used to implement certain framework features that are
 * not intended to be used by Construct consumers, and so should be hidden
 * from accidental use.
 *
 * @example
 *
 * // Usage
 * const roots = DependableTrait.get(construct).dependencyRoots;
 *
 * // Definition
 * DependableTrait.implement(construct, {
 *   get dependencyRoots() { return []; }
 * });
 */
class DependableTrait {
    /**
     * Register `instance` to have the given DependableTrait
     *
     * Should be called in the class constructor.
     */
    static implement(instance, trait) {
        // I would also like to reference classes (to cut down on the list of objects
        // we need to manage), but we can't do that either since jsii doesn't have the
        // concept of a class reference.
        instance[DEPENDABLE_SYMBOL] = trait;
    }
    /**
     * Return the matching DependableTrait for the given class instance.
     */
    static get(instance) {
        const ret = instance[DEPENDABLE_SYMBOL];
        if (!ret) {
            throw new Error(`${instance} does not implement DependableTrait`);
        }
        return ret;
    }
}
exports.DependableTrait = DependableTrait;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwZW5kZW5jeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGVuZGVuY3kudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFpQkE7Ozs7O0dBS0c7QUFDSCxNQUFhLGtCQUFrQjtJQUc3QjtRQUZpQixxQkFBZ0IsR0FBRyxJQUFJLEtBQUssRUFBYyxDQUFDO1FBRzFELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixlQUFlLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRTtZQUM5QixJQUFJLGVBQWUsS0FBSyxPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7U0FDeEQsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksR0FBRyxDQUFDLFNBQXFCO1FBQzlCLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDeEMsQ0FBQztDQUNGO0FBaEJELGdEQWdCQztBQUVELE1BQU0saUJBQWlCLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0FBRXRFOzs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FrQkc7QUFDSCxNQUFzQixlQUFlO0lBQ25DOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQXFCLEVBQUUsS0FBc0I7UUFDbkUsNkVBQTZFO1FBQzdFLDhFQUE4RTtRQUM5RSxnQ0FBZ0M7UUFDL0IsUUFBZ0IsQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEtBQUssQ0FBQztJQUMvQyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRyxDQUFDLFFBQXFCO1FBQ3JDLE1BQU0sR0FBRyxHQUFJLFFBQWdCLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ1IsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLFFBQVEscUNBQXFDLENBQUMsQ0FBQztTQUNuRTtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztDQVNGO0FBL0JELDBDQStCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElDb25zdHJ1Y3QgfSBmcm9tIFwiLi9jb25zdHJ1Y3RcIjtcblxuLyoqXG4gKiBUcmFpdCBtYXJrZXIgZm9yIGNsYXNzZXMgdGhhdCBjYW4gYmUgZGVwZW5kZWQgdXBvblxuICpcbiAqIFRoZSBwcmVzZW5jZSBvZiB0aGlzIGludGVyZmFjZSBpbmRpY2F0ZXMgdGhhdCBhbiBvYmplY3QgaGFzXG4gKiBhbiBgSURlcGVuZGFibGVUcmFpdGAgaW1wbGVtZW50YXRpb24uXG4gKlxuICogVGhpcyBpbnRlcmZhY2UgY2FuIGJlIHVzZWQgdG8gdGFrZSBhbiAob3JkZXJpbmcpIGRlcGVuZGVuY3kgb24gYSBzZXQgb2ZcbiAqIGNvbnN0cnVjdHMuIEFuIG9yZGVyaW5nIGRlcGVuZGVuY3kgaW1wbGllcyB0aGF0IHRoZSByZXNvdXJjZXMgcmVwcmVzZW50ZWQgYnlcbiAqIHRob3NlIGNvbnN0cnVjdHMgYXJlIGRlcGxveWVkIGJlZm9yZSB0aGUgcmVzb3VyY2VzIGRlcGVuZGluZyBPTiB0aGVtIGFyZVxuICogZGVwbG95ZWQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSURlcGVuZGFibGUge1xuICAvLyBFbXB0eSwgdGhpcyBpbnRlcmZhY2UgaXMgYSB0cmFpdCBtYXJrZXJcbn1cblxuLyoqXG4gKiBBIHNldCBvZiBjb25zdHJ1Y3RzIHRvIGJlIHVzZWQgYXMgYSBkZXBlbmRhYmxlXG4gKlxuICogVGhpcyBjbGFzcyBjYW4gYmUgdXNlZCB3aGVuIGEgc2V0IG9mIGNvbnN0cnVjdHMgd2hpY2ggYXJlIGRpc2pvaW50IGluIHRoZVxuICogY29uc3RydWN0IHRyZWUgbmVlZHMgdG8gYmUgY29tYmluZWQgdG8gYmUgdXNlZCBhcyBhIHNpbmdsZSBkZXBlbmRhYmxlLlxuICovXG5leHBvcnQgY2xhc3MgQ29uY3JldGVEZXBlbmRhYmxlIGltcGxlbWVudHMgSURlcGVuZGFibGUge1xuICBwcml2YXRlIHJlYWRvbmx5IF9kZXBlbmRlbmN5Um9vdHMgPSBuZXcgQXJyYXk8SUNvbnN0cnVjdD4oKTtcblxuICBjb25zdHJ1Y3RvcigpIHtcbiAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICBEZXBlbmRhYmxlVHJhaXQuaW1wbGVtZW50KHRoaXMsIHtcbiAgICAgIGdldCBkZXBlbmRlbmN5Um9vdHMoKSB7IHJldHVybiBzZWxmLl9kZXBlbmRlbmN5Um9vdHM7IH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgY29uc3RydWN0IHRvIHRoZSBkZXBlbmRlbmN5IHJvb3RzXG4gICAqL1xuICBwdWJsaWMgYWRkKGNvbnN0cnVjdDogSUNvbnN0cnVjdCkge1xuICAgIHRoaXMuX2RlcGVuZGVuY3lSb290cy5wdXNoKGNvbnN0cnVjdCk7XG4gIH1cbn1cblxuY29uc3QgREVQRU5EQUJMRV9TWU1CT0wgPSBTeW1ib2wuZm9yKCdAYXdzLWNkay9jb3JlLkRlcGVuZGFibGVUcmFpdCcpO1xuXG4vKipcbiAqIFRyYWl0IGZvciBJRGVwZW5kYWJsZVxuICpcbiAqIFRyYWl0cyBhcmUgaW50ZXJmYWNlcyB0aGF0IGFyZSBwcml2YXRlbHkgaW1wbGVtZW50ZWQgYnkgb2JqZWN0cy4gSW5zdGVhZCBvZlxuICogc2hvd2luZyB1cCBpbiB0aGUgcHVibGljIGludGVyZmFjZSBvZiBhIGNsYXNzLCB0aGV5IG5lZWQgdG8gYmUgcXVlcmllZFxuICogZXhwbGljaXRseS4gVGhpcyBpcyB1c2VkIHRvIGltcGxlbWVudCBjZXJ0YWluIGZyYW1ld29yayBmZWF0dXJlcyB0aGF0IGFyZVxuICogbm90IGludGVuZGVkIHRvIGJlIHVzZWQgYnkgQ29uc3RydWN0IGNvbnN1bWVycywgYW5kIHNvIHNob3VsZCBiZSBoaWRkZW5cbiAqIGZyb20gYWNjaWRlbnRhbCB1c2UuXG4gKlxuICogQGV4YW1wbGVcbiAqXG4gKiAvLyBVc2FnZVxuICogY29uc3Qgcm9vdHMgPSBEZXBlbmRhYmxlVHJhaXQuZ2V0KGNvbnN0cnVjdCkuZGVwZW5kZW5jeVJvb3RzO1xuICpcbiAqIC8vIERlZmluaXRpb25cbiAqIERlcGVuZGFibGVUcmFpdC5pbXBsZW1lbnQoY29uc3RydWN0LCB7XG4gKiAgIGdldCBkZXBlbmRlbmN5Um9vdHMoKSB7IHJldHVybiBbXTsgfVxuICogfSk7XG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBEZXBlbmRhYmxlVHJhaXQge1xuICAvKipcbiAgICogUmVnaXN0ZXIgYGluc3RhbmNlYCB0byBoYXZlIHRoZSBnaXZlbiBEZXBlbmRhYmxlVHJhaXRcbiAgICpcbiAgICogU2hvdWxkIGJlIGNhbGxlZCBpbiB0aGUgY2xhc3MgY29uc3RydWN0b3IuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGltcGxlbWVudChpbnN0YW5jZTogSURlcGVuZGFibGUsIHRyYWl0OiBEZXBlbmRhYmxlVHJhaXQpIHtcbiAgICAvLyBJIHdvdWxkIGFsc28gbGlrZSB0byByZWZlcmVuY2UgY2xhc3NlcyAodG8gY3V0IGRvd24gb24gdGhlIGxpc3Qgb2Ygb2JqZWN0c1xuICAgIC8vIHdlIG5lZWQgdG8gbWFuYWdlKSwgYnV0IHdlIGNhbid0IGRvIHRoYXQgZWl0aGVyIHNpbmNlIGpzaWkgZG9lc24ndCBoYXZlIHRoZVxuICAgIC8vIGNvbmNlcHQgb2YgYSBjbGFzcyByZWZlcmVuY2UuXG4gICAgKGluc3RhbmNlIGFzIGFueSlbREVQRU5EQUJMRV9TWU1CT0xdID0gdHJhaXQ7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIHRoZSBtYXRjaGluZyBEZXBlbmRhYmxlVHJhaXQgZm9yIHRoZSBnaXZlbiBjbGFzcyBpbnN0YW5jZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZ2V0KGluc3RhbmNlOiBJRGVwZW5kYWJsZSk6IERlcGVuZGFibGVUcmFpdCB7XG4gICAgY29uc3QgcmV0ID0gKGluc3RhbmNlIGFzIGFueSlbREVQRU5EQUJMRV9TWU1CT0xdO1xuICAgIGlmICghcmV0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCR7aW5zdGFuY2V9IGRvZXMgbm90IGltcGxlbWVudCBEZXBlbmRhYmxlVHJhaXRgKTtcbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgc2V0IG9mIGNvbnN0cnVjdHMgdGhhdCBmb3JtIHRoZSByb290IG9mIHRoaXMgZGVwZW5kYWJsZVxuICAgKlxuICAgKiBBbGwgcmVzb3VyY2VzIHVuZGVyIGFsbCByZXR1cm5lZCBjb25zdHJ1Y3RzIGFyZSBpbmNsdWRlZCBpbiB0aGUgb3JkZXJpbmdcbiAgICogZGVwZW5kZW5jeS5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBkZXBlbmRlbmN5Um9vdHM6IElDb25zdHJ1Y3RbXTtcbn1cbiJdfQ==