"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const stack_1 = require("../stack");
const token_1 = require("../token");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(resource.node.uniqueId, 24);
    let region = stack.region;
    if (token_1.Token.isUnresolved(region)) {
        region = undefined;
    }
    let account = stack.account;
    if (token_1.Token.isUnresolved(account)) {
        account = undefined;
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region || '')
        .update(account || '');
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGh5c2ljYWwtbmFtZS1nZW5lcmF0b3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwaHlzaWNhbC1uYW1lLWdlbmVyYXRvci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGlDQUFrQztBQUVsQyxvQ0FBaUM7QUFDakMsb0NBQWlDO0FBRWpDLFNBQWdCLG9CQUFvQixDQUFDLFFBQW1CO0lBQ3RELE1BQU0sS0FBSyxHQUFHLGFBQUssQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDakMsTUFBTSxTQUFTLEdBQUcsSUFBSSxjQUFjLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMxRCxNQUFNLE1BQU0sR0FBRyxJQUFJLGNBQWMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUU5RCxJQUFJLE1BQU0sR0FBdUIsS0FBSyxDQUFDLE1BQU0sQ0FBQztJQUM5QyxJQUFJLGFBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDOUIsTUFBTSxHQUFHLFNBQVMsQ0FBQztLQUNwQjtJQUVELElBQUksT0FBTyxHQUF1QixLQUFLLENBQUMsT0FBTyxDQUFDO0lBQ2hELElBQUksYUFBSyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsRUFBRTtRQUMvQixPQUFPLEdBQUcsU0FBUyxDQUFDO0tBQ3JCO0lBRUQsTUFBTSxLQUFLLEdBQUcsQ0FBQyxTQUFTLEVBQUUsTUFBTSxDQUFDO1NBQzlCLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0lBRWhDLE1BQU0sVUFBVSxHQUFHLEVBQUUsQ0FBQztJQUN0QixNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQztTQUN2QyxNQUFNLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQztTQUN6QixNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQztTQUN0QixNQUFNLENBQUMsTUFBTSxJQUFJLEVBQUUsQ0FBQztTQUNwQixNQUFNLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3pCLE1BQU0sSUFBSSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxVQUFVLENBQUMsQ0FBQztJQUV2RCxNQUFNLEdBQUcsR0FBRyxDQUFDLEdBQUcsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUV0QyxPQUFPLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztBQUMzQixDQUFDO0FBN0JELG9EQTZCQztBQUVELE1BQWUsUUFBUTtJQUdyQixZQUFZLE9BQWU7UUFDekIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7SUFDekIsQ0FBQztDQUdGO0FBRUQsTUFBTSxjQUFlLFNBQVEsUUFBUTtJQUNuQyxZQUFZLE9BQWUsRUFBbUIsWUFBb0I7UUFDaEUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRDZCLGlCQUFZLEdBQVosWUFBWSxDQUFRO0lBRWxFLENBQUM7SUFFTSxRQUFRO1FBQ2IsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQ2xELENBQUM7Q0FDRjtBQUVELE1BQU0sY0FBZSxTQUFRLFFBQVE7SUFDbkMsWUFBWSxHQUFXLEVBQW1CLFlBQW9CO1FBQzVELEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUQ2QixpQkFBWSxHQUFaLFlBQVksQ0FBUTtJQUU5RCxDQUFDO0lBRU0sUUFBUTtRQUNiLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDO1FBQ25DLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDM0QsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxVQUFVLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDaEQsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGNyeXB0byA9IHJlcXVpcmUoJ2NyeXB0bycpO1xuaW1wb3J0IHsgSVJlc291cmNlIH0gZnJvbSAnLi4vcmVzb3VyY2UnO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICcuLi9zdGFjayc7XG5pbXBvcnQgeyBUb2tlbiB9IGZyb20gJy4uL3Rva2VuJztcblxuZXhwb3J0IGZ1bmN0aW9uIGdlbmVyYXRlUGh5c2ljYWxOYW1lKHJlc291cmNlOiBJUmVzb3VyY2UpOiBzdHJpbmcge1xuICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHJlc291cmNlKTtcbiAgY29uc3Qgc3RhY2tQYXJ0ID0gbmV3IFByZWZpeE5hbWVQYXJ0KHN0YWNrLnN0YWNrTmFtZSwgMjUpO1xuICBjb25zdCBpZFBhcnQgPSBuZXcgU3VmZml4TmFtZVBhcnQocmVzb3VyY2Uubm9kZS51bmlxdWVJZCwgMjQpO1xuXG4gIGxldCByZWdpb246IHN0cmluZyB8IHVuZGVmaW5lZCA9IHN0YWNrLnJlZ2lvbjtcbiAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZChyZWdpb24pKSB7XG4gICAgcmVnaW9uID0gdW5kZWZpbmVkO1xuICB9XG5cbiAgbGV0IGFjY291bnQ6IHN0cmluZyB8IHVuZGVmaW5lZCA9IHN0YWNrLmFjY291bnQ7XG4gIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQoYWNjb3VudCkpIHtcbiAgICBhY2NvdW50ID0gdW5kZWZpbmVkO1xuICB9XG5cbiAgY29uc3QgcGFydHMgPSBbc3RhY2tQYXJ0LCBpZFBhcnRdXG4gICAgLm1hcChwYXJ0ID0+IHBhcnQuZ2VuZXJhdGUoKSk7XG5cbiAgY29uc3QgaGFzaExlbmd0aCA9IDEyO1xuICBjb25zdCBzaGEyNTYgPSBjcnlwdG8uY3JlYXRlSGFzaCgnc2hhMjU2JylcbiAgICAudXBkYXRlKHN0YWNrUGFydC5iYXJlU3RyKVxuICAgIC51cGRhdGUoaWRQYXJ0LmJhcmVTdHIpXG4gICAgLnVwZGF0ZShyZWdpb24gfHwgJycpXG4gICAgLnVwZGF0ZShhY2NvdW50IHx8ICcnKTtcbiAgY29uc3QgaGFzaCA9IHNoYTI1Ni5kaWdlc3QoJ2hleCcpLnNsaWNlKDAsIGhhc2hMZW5ndGgpO1xuXG4gIGNvbnN0IHJldCA9IFsuLi5wYXJ0cywgaGFzaF0uam9pbignJyk7XG5cbiAgcmV0dXJuIHJldC50b0xvd2VyQ2FzZSgpO1xufVxuXG5hYnN0cmFjdCBjbGFzcyBOYW1lUGFydCB7XG4gIHB1YmxpYyByZWFkb25seSBiYXJlU3RyOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IoYmFyZVN0cjogc3RyaW5nKSB7XG4gICAgdGhpcy5iYXJlU3RyID0gYmFyZVN0cjtcbiAgfVxuXG4gIHB1YmxpYyBhYnN0cmFjdCBnZW5lcmF0ZSgpOiBzdHJpbmc7XG59XG5cbmNsYXNzIFByZWZpeE5hbWVQYXJ0IGV4dGVuZHMgTmFtZVBhcnQge1xuICBjb25zdHJ1Y3RvcihiYXJlU3RyOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgcHJlZml4TGVuZ3RoOiBudW1iZXIpIHtcbiAgICBzdXBlcihiYXJlU3RyKTtcbiAgfVxuXG4gIHB1YmxpYyBnZW5lcmF0ZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLmJhcmVTdHIuc2xpY2UoMCwgdGhpcy5wcmVmaXhMZW5ndGgpO1xuICB9XG59XG5cbmNsYXNzIFN1ZmZpeE5hbWVQYXJ0IGV4dGVuZHMgTmFtZVBhcnQge1xuICBjb25zdHJ1Y3RvcihzdHI6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBzdWZmaXhMZW5ndGg6IG51bWJlcikge1xuICAgIHN1cGVyKHN0cik7XG4gIH1cblxuICBwdWJsaWMgZ2VuZXJhdGUoKTogc3RyaW5nIHtcbiAgICBjb25zdCBzdHJMZW4gPSB0aGlzLmJhcmVTdHIubGVuZ3RoO1xuICAgIGNvbnN0IHN0YXJ0SW5kZXggPSBNYXRoLm1heChzdHJMZW4gLSB0aGlzLnN1ZmZpeExlbmd0aCwgMCk7XG4gICAgcmV0dXJuIHRoaXMuYmFyZVN0ci5zbGljZShzdGFydEluZGV4LCBzdHJMZW4pO1xuICB9XG59XG4iXX0=