"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const encoding_1 = require("./private/encoding");
const token_map_1 = require("./private/token-map");
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
class StringConcat {
    join(left, right) {
        if (left === undefined) {
            return right !== undefined ? `${right}` : undefined;
        }
        if (right === undefined) {
            return `${left}`;
        }
        return `${left}${right}`;
    }
}
exports.StringConcat = StringConcat;
/**
 * Default resolver implementation
 */
class DefaultTokenResolver {
    constructor(concat) {
        this.concat = concat;
    }
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t, context, postProcessor) {
        try {
            let resolved = t.resolve(context);
            // The token might have returned more values that need resolving, recurse
            resolved = context.resolve(resolved);
            resolved = postProcessor.postProcess(resolved, context);
            return resolved;
        }
        catch (e) {
            let message = `Resolution error: ${e.message}.`;
            if (t.creationStack && t.creationStack.length > 0) {
                message += `\nObject creation stack:\n  at ${t.creationStack.join('\n  at ')}`;
            }
            e.message = message;
            throw e;
        }
    }
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments, context) {
        return fragments.mapTokens({ mapToken: context.resolve }).join(this.concat);
    }
    resolveList(xs, context) {
        // Must be a singleton list token, because concatenation is not allowed.
        if (xs.length !== 1) {
            throw new Error(`Cannot add elements to list token, got: ${xs}`);
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const tokenMap = token_map_1.TokenMap.instance();
        const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
        if (fragments.length !== 1) {
            throw new Error(`Cannot concatenate strings in a tokenized string array, got: ${xs[0]}`);
        }
        return fragments.mapTokens({ mapToken: context.resolve }).firstValue;
    }
}
exports.DefaultTokenResolver = DefaultTokenResolver;
//# sourceMappingURL=data:application/json;base64,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