"use strict";
const lib_1 = require("../lib");
const context_provider_1 = require("../lib/context-provider");
/**
 * Get the expected context key from a stack with missing parameters
 */
function expectedContextKey(stack) {
    const missing = lib_1.ConstructNode.synth(stack.node).manifest.missing;
    if (!missing || missing.length !== 1) {
        throw new Error(`Expecting assembly to include a single missing context report`);
    }
    return missing[0].key;
}
module.exports = {
    'AvailabilityZoneProvider returns a list with dummy values if the context is not available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['dummy1a', 'dummy1b', 'dummy1c']);
        test.done();
    },
    'AvailabilityZoneProvider will return context list if available'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, ['us-east-1a', 'us-east-1b']);
        const azs = stack.availabilityZones;
        test.deepEqual(azs, ['us-east-1a', 'us-east-1b']);
        test.done();
    },
    'AvailabilityZoneProvider will complain if not given a list'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const before = stack.availabilityZones;
        test.deepEqual(before, ['dummy1a', 'dummy1b', 'dummy1c']);
        const key = expectedContextKey(stack);
        stack.node.setContext(key, 'not-a-list');
        test.throws(() => stack.availabilityZones);
        test.done();
    },
    'ContextProvider consistently generates a key'(test) {
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'ssm',
            props: {
                parameterName: 'foo',
                anyStringParam: 'bar'
            },
        });
        test.deepEqual(key, {
            key: 'ssm:account=12345:anyStringParam=bar:parameterName=foo:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                parameterName: 'foo',
                anyStringParam: 'bar'
            }
        });
        const complexKey = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'vpc',
            props: {
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            }
        });
        test.deepEqual(complexKey, {
            key: 'vpc:account=12345:cidrBlock=192.168.0.16:igw=false:region=us-east-1:tags.Env=Preprod:tags.Name=MyVPC',
            props: {
                account: '12345',
                region: 'us-east-1',
                cidrBlock: '192.168.0.16',
                tags: { Name: 'MyVPC', Env: 'Preprod' },
                igw: false,
            }
        });
        test.done();
    },
    'Key generation can contain arbitrarily deep structures'(test) {
        // GIVEN
        const stack = new lib_1.Stack(undefined, 'TestStack', { env: { account: '12345', region: 'us-east-1' } });
        // WHEN
        const key = context_provider_1.ContextProvider.getKey(stack, {
            provider: 'provider',
            props: {
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            }
        });
        // THEN
        test.deepEqual(key, {
            key: 'provider:account=12345:list.0.key=key1:list.0.value=value1:list.1.key=key2:list.1.value=value2:region=us-east-1',
            props: {
                account: '12345',
                region: 'us-east-1',
                list: [
                    { key: 'key1', value: 'value1' },
                    { key: 'key2', value: 'value2' },
                ],
            }
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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