"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdateRoute53AddOn = exports.ElasticFileSystemAddOn = exports.BucketSyncAddOn = exports.InstanceFirewallAddOn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const commands_1 = require("./commands");
/**
 * An add-on that configures an on-instance firewall.
 */
class InstanceFirewallAddOn {
    /**
     * An add-on that configures an on-instance firewall.
     *
     * @param rules - The instance firewall rules
     * @param props - Optional configuration properties
     */
    constructor(rules, props) {
        this.rules = rules;
        this.props = props;
    }
    configure(starter) {
        starter.addScript(this.props?.priority ?? 0, aws_cdk_lib_1.Lazy.string({ produce: () => this.rules.buildCommands().join("\n") }));
    }
}
exports.InstanceFirewallAddOn = InstanceFirewallAddOn;
_a = JSII_RTTI_SYMBOL_1;
InstanceFirewallAddOn[_a] = { fqn: "shady-island.configuration.InstanceFirewallAddOn", version: "0.1.51" };
/**
 * An add-on that synchronizes files from S3 to directories on the instance.
 *
 * This add-on also grants read access to the bucket.
 */
class BucketSyncAddOn {
    /**
     * An add-on that synchronizes files from S3 to directories on the instance.
     *
     * This add-on also grants read access to the bucket.
     *
     * @param bucket - The S3 bucket from which files can be downloaded
     * @param destinations - An object where keys are S3 object key prefixes and values are filesystem directories
     * @param props - Optional configuration properties
     */
    constructor(bucket, destinations, props) {
        this.bucket = bucket;
        this.destinations = destinations;
        this.props = props;
    }
    configure(starter) {
        this.bucket.grantRead(starter);
        starter.addScript(this.props?.priority ?? 10, ...commands_1.ShellCommands.syncFromBucket(this.bucket, this.destinations));
    }
}
exports.BucketSyncAddOn = BucketSyncAddOn;
_b = JSII_RTTI_SYMBOL_1;
BucketSyncAddOn[_b] = { fqn: "shady-island.configuration.BucketSyncAddOn", version: "0.1.51" };
/**
 * An add-on that configures a mount point for an EFS filesystem.
 *
 * This add-on will produce a startup script to:
 * - Create the mount directory
 * - Mount the NFS filesystem to the mount point
 * - Optionally change the mode or ownership of the mount point
 *
 * This visitor also configures the Security Groups on both ends.
 */
class ElasticFileSystemAddOn {
    /**
     * An add-on that configures a mount point for an EFS filesystem.
     *
     * This add-on will produce a startup script to:
     * - Create the mount directory
     * - Mount the NFS filesystem to the mount point
     * - Optionally change the mode or ownership of the mount point
     *
     * This add-on also configures the Security Groups on both ends.
     *
     * @param filesystem - The elastic filesystem to mount
     * @param destination - The directory to use as the mount point
     * @param props - Optional configuration properties
     */
    constructor(filesystem, destination, props) {
        this.filesystem = filesystem;
        this.destination = destination;
        this.props = props;
    }
    configure(starter) {
        const description = "Allow NFS traffic from EC2 to EFS";
        // The CDK source code definitely sets the default port to 2049.
        const port = this.filesystem.connections.defaultPort;
        // To prevent circular references in case the stacks are different.
        for (const fsg of this.filesystem.connections.securityGroups) {
            for (const isg of starter.connections.securityGroups) {
                fsg.addIngressRule(isg, port, description, true);
                isg.addEgressRule(fsg, port, description);
            }
        }
        const postMount = [];
        const { chown, chgrp, chmod, priority = 10 } = this.props || {};
        if (chmod) {
            postMount.push(`chmod ${chmod} "${this.destination}"`);
        }
        if (chown || chgrp) {
            postMount.push(...commands_1.ShellCommands.changeOwnership(this.destination, chown, chgrp));
        }
        starter.addScript(priority, ...commands_1.ShellCommands.mountElasticFileSystem(this.filesystem, this.destination), ...postMount);
    }
}
exports.ElasticFileSystemAddOn = ElasticFileSystemAddOn;
_c = JSII_RTTI_SYMBOL_1;
ElasticFileSystemAddOn[_c] = { fqn: "shady-island.configuration.ElasticFileSystemAddOn", version: "0.1.51" };
/**
 * An add-on that updates Route 53 with instance public-facing IP addresses.
 *
 * This add-on also configures the necessary IAM policy.
 */
class UpdateRoute53AddOn {
    /**
     * An add-on that updates Route 53 with instance public-facing IP addresses.
     *
     * This add-on also configures the necessary IAM policy.
     *
     * @param zone - The Route 53 hosted zone
     * @param subdomain - The subdomain of the DNS record
     * @param props - Optional configuration properties
     */
    constructor(zone, subdomain, props) {
        this.zone = zone;
        this.subdomain = subdomain;
        this.props = props;
    }
    configure(starter) {
        const { ipv4 = true, ipv6 = false, priority = 5 } = this.props || {};
        if (!ipv4 && !ipv6) {
            return;
        }
        const recordTypes = [];
        if (ipv4) {
            recordTypes.push("A");
        }
        if (ipv6) {
            recordTypes.push("AAAA");
        }
        const fqdn = `${this.subdomain}.${this.zone.zoneName}`;
        starter.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ["route53:changeResourceRecordSets"],
            resources: [this.zone.hostedZoneArn],
            conditions: {
                "ForAllValues:StringLike": {
                    "route53:ChangeResourceRecordSetsNormalizedRecordNames": [fqdn],
                },
                "ForAllValues:StringEquals": {
                    "route53:ChangeResourceRecordSetsActions": ["UPSERT"],
                    "route53:ChangeResourceRecordSetsRecordTypes": recordTypes,
                },
            },
        }));
        const commands = [
            'metaToken=$(curl -sX PUT "http://instance-data/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 300")',
            'metaTokenHeader="X-aws-ec2-metadata-token: $metaToken"',
            "IMD=http://instance-data/latest/meta-data",
            'IMDNIM="$IMD/network/interfaces/macs"',
            'myNetworkMac=$(curl -sH "$metaTokenHeader" "$IMDNIM/")',
            'IMDNIMC="$IMDNIM/$myNetworkMac"',
        ];
        const changeBatchChanges = [];
        if (ipv4) {
            commands.push('myHostIpPublic=$(curl -sH "$metaTokenHeader" "${IMDNIMC}public-ipv4s")');
            changeBatchChanges.push({
                Action: "UPSERT",
                ResourceRecordSet: {
                    Name: fqdn,
                    Type: "A",
                    TTL: 300,
                    ResourceRecords: [{ Value: "'$myHostIpPublic'" }],
                },
            });
        }
        if (ipv6) {
            commands.push('myHostIpv6=$(curl -sH "$metaTokenHeader" "${IMDNIMC}ipv6s")');
            changeBatchChanges.push({
                Action: "UPSERT",
                ResourceRecordSet: {
                    Name: fqdn,
                    Type: "AAAA",
                    TTL: 300,
                    ResourceRecords: [{ Value: "'$myHostIpv6'" }],
                },
            });
        }
        const changeBatch = { Changes: changeBatchChanges };
        starter.addScript(priority, ...commands, `aws route53 change-resource-record-sets --hosted-zone-id ${this.zone.hostedZoneId} --change-batch '${JSON.stringify(changeBatch)}'`);
    }
}
exports.UpdateRoute53AddOn = UpdateRoute53AddOn;
_d = JSII_RTTI_SYMBOL_1;
UpdateRoute53AddOn[_d] = { fqn: "shady-island.configuration.UpdateRoute53AddOn", version: "0.1.51" };
//# sourceMappingURL=data:application/json;base64,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