"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserDataBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
/**
 * A utility class to assist with composing instance User Data.
 *
 * This class allows multiple observers in code to add lines to the same end
 * result UserData without clobbering each other. Just like `conf.d` directories
 * with priority number prefixes, you can declare the proper execution order of
 * your UserData commands without having to add them in that order.
 */
class UserDataBuilder {
    constructor() {
        /**
         * The groups of prioritized command line entries.
         */
        this.lines = [];
    }
    /**
     * Returns a user data builder for GNU/Linux operating systems.
     *
     * @param options - The Linux UserData constructor options
     * @returns the new builder object
     */
    static forLinux(options) {
        return new LinuxUserDataBuilder(options);
    }
    /**
     * Returns a user data builder for Windows operating systems.
     *
     * @returns the new builder object
     */
    static forWindows() {
        return new WindowsUserDataBuilder();
    }
    /**
     * Add one or more commands to the user data with a priority of `0`.
     *
     * @param commands - The lines to add
     */
    addCommands(...commands) {
        this.lines.push({ priority: 0, lines: commands });
    }
    /**
     * Add one or more commands to the user data at a specific priority.
     *
     * @param priority - The priority of these lines (lower executes earlier)
     * @param commands - The lines to add
     */
    insertCommands(priority, ...commands) {
        this.lines.push({ priority, lines: commands });
    }
}
exports.UserDataBuilder = UserDataBuilder;
_a = JSII_RTTI_SYMBOL_1;
UserDataBuilder[_a] = { fqn: "shady-island.UserDataBuilder", version: "0.1.51" };
/**
 * A User Data builder that targets GNU/Linux operating systems.
 */
class LinuxUserDataBuilder extends UserDataBuilder {
    constructor(options = {}) {
        super();
        this.options = options;
    }
    buildUserData() {
        const lines = [...this.lines];
        lines.sort((a, b) => a.priority - b.priority);
        const userData = aws_ec2_1.UserData.forLinux(this.options);
        userData.addCommands(...lines.map((a) => a.lines).flat());
        return userData;
    }
}
/**
 * A User Data builder that targets Windows operating systems.
 */
class WindowsUserDataBuilder extends UserDataBuilder {
    buildUserData() {
        const lines = [...this.lines];
        lines.sort((a, b) => a.priority - b.priority);
        const userData = aws_ec2_1.UserData.forWindows();
        userData.addCommands(...lines.map((a) => a.lines).flat());
        return userData;
    }
}
//# sourceMappingURL=data:application/json;base64,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