"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// eslint-disable-next-line import/no-extraneous-dependencies
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
// eslint-disable-next-line import/no-extraneous-dependencies
const promise_1 = require("mysql2/promise");
const util_1 = require("./util");
const adminSecretArn = process.env.ADMIN_SECRET_ARN;
const databaseName = process.env.DB_NAME;
const characterSet = process.env.DB_CHARACTER_SET || "utf8mb4";
const collation = process.env.DB_COLLATION;
const secretsManagerClient = new client_secrets_manager_1.SecretsManagerClient({});
const ownerSecretArns = (0, util_1.parseJsonArrayFromEnv)("OWNER_SECRETS");
const readerSecretArns = (0, util_1.parseJsonArrayFromEnv)("READER_SECRETS");
const unprivilegedSecretArns = (0, util_1.parseJsonArrayFromEnv)("UNPRIVILEGED_SECRETS");
async function createUser(connection, user, password) {
    const escapedPassword = connection.escape(password);
    let createUserSql = `CREATE USER IF NOT EXISTS ${user} IDENTIFIED BY ${escapedPassword}`;
    console.log(`About to create user: ${user}`);
    await connection.execute(createUserSql, [password]);
    console.log(`User created: ${user}`);
}
async function applyGrant(connection, user, privileges) {
    let grantPrivSql = `GRANT ${privileges} ON ${databaseName}.* TO ${user}`;
    console.log(`About to grant privileges to user: ${user}`);
    await connection.execute(grantPrivSql);
    console.log(`Privileges granted: ${privileges} ON ${databaseName}.*`);
}
const handler = async () => {
    const adminSecret = await (0, util_1.fetchSecret)(adminSecretArn, secretsManagerClient);
    const region = process.env.AWS_REGION || process.env.AWS_DEFAULT_REGION;
    const caBundle = await (0, util_1.readRemote)(process.env.CA_CERTS_URL ||
        `https://truststore.pki.rds.amazonaws.com/${region}/${region}-bundle.pem`);
    const connection = await (0, promise_1.createConnection)({
        host: adminSecret.host,
        user: adminSecret.username,
        password: adminSecret.password,
        port: adminSecret.port,
        ssl: { ca: caBundle },
        connectTimeout: 40000,
    });
    let createDbSql = `CREATE DATABASE IF NOT EXISTS ${databaseName} CHARACTER SET ${characterSet}`;
    if (collation) {
        createDbSql += ` COLLATE ${collation}`;
    }
    console.log(`About to create database: ${databaseName}`);
    await connection.execute(createDbSql);
    console.log(`Database created: ${databaseName}`);
    const ownerSecrets = await (0, util_1.fetchAllSecrets)(ownerSecretArns, secretsManagerClient);
    for (const userSecret of ownerSecrets) {
        const mysqlUser = `'${userSecret.username}'@'%'`;
        await createUser(connection, mysqlUser, userSecret.password);
        await applyGrant(connection, mysqlUser, "ALL PRIVILEGES");
    }
    const readerSecrets = await (0, util_1.fetchAllSecrets)(readerSecretArns, secretsManagerClient);
    for (const userSecret of readerSecrets) {
        const mysqlUser = `'${userSecret.username}'@'%'`;
        await createUser(connection, mysqlUser, userSecret.password);
        await applyGrant(connection, mysqlUser, "SELECT, SHOW VIEW");
    }
    const unprivilegedSecrets = await (0, util_1.fetchAllSecrets)(unprivilegedSecretArns, secretsManagerClient);
    for (const userSecret of unprivilegedSecrets) {
        const mysqlUser = `'${userSecret.username}'@'%'`;
        await createUser(connection, mysqlUser, userSecret.password);
    }
    let flushSql = "FLUSH PRIVILEGES";
    console.log("About to flush privileges");
    await connection.execute(flushSql);
    console.log("Privileges flushed");
    console.log("About to close the connection");
    await connection.end();
    console.log("Connection closed");
};
module.exports = { handler };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibXlzcWwuaGFuZGxlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9yZHMvdHJpZ2dlcnMvbXlzcWwuaGFuZGxlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLDZEQUE2RDtBQUM3RCw0RUFBdUU7QUFDdkUsNkRBQTZEO0FBQzdELDRDQUE4RDtBQUU5RCxpQ0FLZ0I7QUFFaEIsTUFBTSxjQUFjLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxnQkFBaUIsQ0FBQztBQUNyRCxNQUFNLFlBQVksR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLE9BQVEsQ0FBQztBQUMxQyxNQUFNLFlBQVksR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLGdCQUFpQixJQUFJLFNBQVMsQ0FBQztBQUNoRSxNQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQztBQUUzQyxNQUFNLG9CQUFvQixHQUFHLElBQUksNkNBQW9CLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDMUQsTUFBTSxlQUFlLEdBQUcsSUFBQSw0QkFBcUIsRUFBQyxlQUFlLENBQUMsQ0FBQztBQUMvRCxNQUFNLGdCQUFnQixHQUFHLElBQUEsNEJBQXFCLEVBQUMsZ0JBQWdCLENBQUMsQ0FBQztBQUNqRSxNQUFNLHNCQUFzQixHQUFHLElBQUEsNEJBQXFCLEVBQUMsc0JBQXNCLENBQUMsQ0FBQztBQUU3RSxLQUFLLFVBQVUsVUFBVSxDQUN2QixVQUFzQixFQUN0QixJQUFZLEVBQ1osUUFBZ0I7SUFFaEIsTUFBTSxlQUFlLEdBQUcsVUFBVSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNwRCxJQUFJLGFBQWEsR0FBRyw2QkFBNkIsSUFBSSxrQkFBa0IsZUFBZSxFQUFFLENBQUM7SUFDekYsT0FBTyxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUM3QyxNQUFNLFVBQVUsQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztJQUNwRCxPQUFPLENBQUMsR0FBRyxDQUFDLGlCQUFpQixJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQ3ZDLENBQUM7QUFFRCxLQUFLLFVBQVUsVUFBVSxDQUN2QixVQUFzQixFQUN0QixJQUFZLEVBQ1osVUFBa0I7SUFFbEIsSUFBSSxZQUFZLEdBQUcsU0FBUyxVQUFVLE9BQU8sWUFBWSxTQUFTLElBQUksRUFBRSxDQUFDO0lBQ3pFLE9BQU8sQ0FBQyxHQUFHLENBQUMsc0NBQXNDLElBQUksRUFBRSxDQUFDLENBQUM7SUFDMUQsTUFBTSxVQUFVLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQ3ZDLE9BQU8sQ0FBQyxHQUFHLENBQUMsdUJBQXVCLFVBQVUsT0FBTyxZQUFZLElBQUksQ0FBQyxDQUFDO0FBQ3hFLENBQUM7QUFFRCxNQUFNLE9BQU8sR0FBRyxLQUFLLElBQUksRUFBRTtJQUN6QixNQUFNLFdBQVcsR0FBd0IsTUFBTSxJQUFBLGtCQUFXLEVBQ3hELGNBQWMsRUFDZCxvQkFBb0IsQ0FDckIsQ0FBQztJQUVGLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUM7SUFDeEUsTUFBTSxRQUFRLEdBQUcsTUFBTSxJQUFBLGlCQUFVLEVBQy9CLE9BQU8sQ0FBQyxHQUFHLENBQUMsWUFBWTtRQUN0Qiw0Q0FBNEMsTUFBTSxJQUFJLE1BQU0sYUFBYSxDQUM1RSxDQUFDO0lBRUYsTUFBTSxVQUFVLEdBQUcsTUFBTSxJQUFBLDBCQUFnQixFQUFDO1FBQ3hDLElBQUksRUFBRSxXQUFXLENBQUMsSUFBSTtRQUN0QixJQUFJLEVBQUUsV0FBVyxDQUFDLFFBQVE7UUFDMUIsUUFBUSxFQUFFLFdBQVcsQ0FBQyxRQUFRO1FBQzlCLElBQUksRUFBRSxXQUFXLENBQUMsSUFBSTtRQUN0QixHQUFHLEVBQUUsRUFBRSxFQUFFLEVBQUUsUUFBUSxFQUFFO1FBQ3JCLGNBQWMsRUFBRSxLQUFLO0tBQ3RCLENBQUMsQ0FBQztJQUVILElBQUksV0FBVyxHQUFHLGlDQUFpQyxZQUFZLGtCQUFrQixZQUFZLEVBQUUsQ0FBQztJQUNoRyxJQUFJLFNBQVMsRUFBRSxDQUFDO1FBQ2QsV0FBVyxJQUFJLFlBQVksU0FBUyxFQUFFLENBQUM7SUFDekMsQ0FBQztJQUNELE9BQU8sQ0FBQyxHQUFHLENBQUMsNkJBQTZCLFlBQVksRUFBRSxDQUFDLENBQUM7SUFDekQsTUFBTSxVQUFVLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ3RDLE9BQU8sQ0FBQyxHQUFHLENBQUMscUJBQXFCLFlBQVksRUFBRSxDQUFDLENBQUM7SUFFakQsTUFBTSxZQUFZLEdBQXVCLE1BQU0sSUFBQSxzQkFBZSxFQUM1RCxlQUFlLEVBQ2Ysb0JBQW9CLENBQ3JCLENBQUM7SUFDRixLQUFLLE1BQU0sVUFBVSxJQUFJLFlBQVksRUFBRSxDQUFDO1FBQ3RDLE1BQU0sU0FBUyxHQUFHLElBQUksVUFBVSxDQUFDLFFBQVEsT0FBTyxDQUFDO1FBQ2pELE1BQU0sVUFBVSxDQUFDLFVBQVUsRUFBRSxTQUFTLEVBQUUsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzdELE1BQU0sVUFBVSxDQUFDLFVBQVUsRUFBRSxTQUFTLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBRUQsTUFBTSxhQUFhLEdBQXVCLE1BQU0sSUFBQSxzQkFBZSxFQUM3RCxnQkFBZ0IsRUFDaEIsb0JBQW9CLENBQ3JCLENBQUM7SUFDRixLQUFLLE1BQU0sVUFBVSxJQUFJLGFBQWEsRUFBRSxDQUFDO1FBQ3ZDLE1BQU0sU0FBUyxHQUFHLElBQUksVUFBVSxDQUFDLFFBQVEsT0FBTyxDQUFDO1FBQ2pELE1BQU0sVUFBVSxDQUFDLFVBQVUsRUFBRSxTQUFTLEVBQUUsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzdELE1BQU0sVUFBVSxDQUFDLFVBQVUsRUFBRSxTQUFTLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBRUQsTUFBTSxtQkFBbUIsR0FBdUIsTUFBTSxJQUFBLHNCQUFlLEVBQ25FLHNCQUFzQixFQUN0QixvQkFBb0IsQ0FDckIsQ0FBQztJQUNGLEtBQUssTUFBTSxVQUFVLElBQUksbUJBQW1CLEVBQUUsQ0FBQztRQUM3QyxNQUFNLFNBQVMsR0FBRyxJQUFJLFVBQVUsQ0FBQyxRQUFRLE9BQU8sQ0FBQztRQUNqRCxNQUFNLFVBQVUsQ0FBQyxVQUFVLEVBQUUsU0FBUyxFQUFFLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBRUQsSUFBSSxRQUFRLEdBQUcsa0JBQWtCLENBQUM7SUFDbEMsT0FBTyxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQ3pDLE1BQU0sVUFBVSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNuQyxPQUFPLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFFbEMsT0FBTyxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQzdDLE1BQU0sVUFBVSxDQUFDLEdBQUcsRUFBRSxDQUFDO0lBQ3ZCLE9BQU8sQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsQ0FBQztBQUNuQyxDQUFDLENBQUM7QUFFRixNQUFNLENBQUMsT0FBTyxHQUFHLEVBQUUsT0FBTyxFQUFFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgaW1wb3J0L25vLWV4dHJhbmVvdXMtZGVwZW5kZW5jaWVzXG5pbXBvcnQgeyBTZWNyZXRzTWFuYWdlckNsaWVudCB9IGZyb20gXCJAYXdzLXNkay9jbGllbnQtc2VjcmV0cy1tYW5hZ2VyXCI7XG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgaW1wb3J0L25vLWV4dHJhbmVvdXMtZGVwZW5kZW5jaWVzXG5pbXBvcnQgeyBDb25uZWN0aW9uLCBjcmVhdGVDb25uZWN0aW9uIH0gZnJvbSBcIm15c3FsMi9wcm9taXNlXCI7XG5pbXBvcnQgdHlwZSB7IFVzZXJuYW1lUGFzc3dvcmQsIERhdGFiYXNlQ3JlZGVudGlhbHMgfSBmcm9tIFwiLi90eXBlc1wiO1xuaW1wb3J0IHtcbiAgZmV0Y2hTZWNyZXQsXG4gIGZldGNoQWxsU2VjcmV0cyxcbiAgcGFyc2VKc29uQXJyYXlGcm9tRW52LFxuICByZWFkUmVtb3RlLFxufSBmcm9tIFwiLi91dGlsXCI7XG5cbmNvbnN0IGFkbWluU2VjcmV0QXJuID0gcHJvY2Vzcy5lbnYuQURNSU5fU0VDUkVUX0FSTiE7XG5jb25zdCBkYXRhYmFzZU5hbWUgPSBwcm9jZXNzLmVudi5EQl9OQU1FITtcbmNvbnN0IGNoYXJhY3RlclNldCA9IHByb2Nlc3MuZW52LkRCX0NIQVJBQ1RFUl9TRVQhIHx8IFwidXRmOG1iNFwiO1xuY29uc3QgY29sbGF0aW9uID0gcHJvY2Vzcy5lbnYuREJfQ09MTEFUSU9OO1xuXG5jb25zdCBzZWNyZXRzTWFuYWdlckNsaWVudCA9IG5ldyBTZWNyZXRzTWFuYWdlckNsaWVudCh7fSk7XG5jb25zdCBvd25lclNlY3JldEFybnMgPSBwYXJzZUpzb25BcnJheUZyb21FbnYoXCJPV05FUl9TRUNSRVRTXCIpO1xuY29uc3QgcmVhZGVyU2VjcmV0QXJucyA9IHBhcnNlSnNvbkFycmF5RnJvbUVudihcIlJFQURFUl9TRUNSRVRTXCIpO1xuY29uc3QgdW5wcml2aWxlZ2VkU2VjcmV0QXJucyA9IHBhcnNlSnNvbkFycmF5RnJvbUVudihcIlVOUFJJVklMRUdFRF9TRUNSRVRTXCIpO1xuXG5hc3luYyBmdW5jdGlvbiBjcmVhdGVVc2VyKFxuICBjb25uZWN0aW9uOiBDb25uZWN0aW9uLFxuICB1c2VyOiBzdHJpbmcsXG4gIHBhc3N3b3JkOiBzdHJpbmdcbikge1xuICBjb25zdCBlc2NhcGVkUGFzc3dvcmQgPSBjb25uZWN0aW9uLmVzY2FwZShwYXNzd29yZCk7XG4gIGxldCBjcmVhdGVVc2VyU3FsID0gYENSRUFURSBVU0VSIElGIE5PVCBFWElTVFMgJHt1c2VyfSBJREVOVElGSUVEIEJZICR7ZXNjYXBlZFBhc3N3b3JkfWA7XG4gIGNvbnNvbGUubG9nKGBBYm91dCB0byBjcmVhdGUgdXNlcjogJHt1c2VyfWApO1xuICBhd2FpdCBjb25uZWN0aW9uLmV4ZWN1dGUoY3JlYXRlVXNlclNxbCwgW3Bhc3N3b3JkXSk7XG4gIGNvbnNvbGUubG9nKGBVc2VyIGNyZWF0ZWQ6ICR7dXNlcn1gKTtcbn1cblxuYXN5bmMgZnVuY3Rpb24gYXBwbHlHcmFudChcbiAgY29ubmVjdGlvbjogQ29ubmVjdGlvbixcbiAgdXNlcjogc3RyaW5nLFxuICBwcml2aWxlZ2VzOiBzdHJpbmdcbikge1xuICBsZXQgZ3JhbnRQcml2U3FsID0gYEdSQU5UICR7cHJpdmlsZWdlc30gT04gJHtkYXRhYmFzZU5hbWV9LiogVE8gJHt1c2VyfWA7XG4gIGNvbnNvbGUubG9nKGBBYm91dCB0byBncmFudCBwcml2aWxlZ2VzIHRvIHVzZXI6ICR7dXNlcn1gKTtcbiAgYXdhaXQgY29ubmVjdGlvbi5leGVjdXRlKGdyYW50UHJpdlNxbCk7XG4gIGNvbnNvbGUubG9nKGBQcml2aWxlZ2VzIGdyYW50ZWQ6ICR7cHJpdmlsZWdlc30gT04gJHtkYXRhYmFzZU5hbWV9LipgKTtcbn1cblxuY29uc3QgaGFuZGxlciA9IGFzeW5jICgpID0+IHtcbiAgY29uc3QgYWRtaW5TZWNyZXQ6IERhdGFiYXNlQ3JlZGVudGlhbHMgPSBhd2FpdCBmZXRjaFNlY3JldChcbiAgICBhZG1pblNlY3JldEFybixcbiAgICBzZWNyZXRzTWFuYWdlckNsaWVudFxuICApO1xuXG4gIGNvbnN0IHJlZ2lvbiA9IHByb2Nlc3MuZW52LkFXU19SRUdJT04gfHwgcHJvY2Vzcy5lbnYuQVdTX0RFRkFVTFRfUkVHSU9OO1xuICBjb25zdCBjYUJ1bmRsZSA9IGF3YWl0IHJlYWRSZW1vdGUoXG4gICAgcHJvY2Vzcy5lbnYuQ0FfQ0VSVFNfVVJMIHx8XG4gICAgICBgaHR0cHM6Ly90cnVzdHN0b3JlLnBraS5yZHMuYW1hem9uYXdzLmNvbS8ke3JlZ2lvbn0vJHtyZWdpb259LWJ1bmRsZS5wZW1gXG4gICk7XG5cbiAgY29uc3QgY29ubmVjdGlvbiA9IGF3YWl0IGNyZWF0ZUNvbm5lY3Rpb24oe1xuICAgIGhvc3Q6IGFkbWluU2VjcmV0Lmhvc3QsXG4gICAgdXNlcjogYWRtaW5TZWNyZXQudXNlcm5hbWUsXG4gICAgcGFzc3dvcmQ6IGFkbWluU2VjcmV0LnBhc3N3b3JkLFxuICAgIHBvcnQ6IGFkbWluU2VjcmV0LnBvcnQsXG4gICAgc3NsOiB7IGNhOiBjYUJ1bmRsZSB9LFxuICAgIGNvbm5lY3RUaW1lb3V0OiA0MDAwMCxcbiAgfSk7XG5cbiAgbGV0IGNyZWF0ZURiU3FsID0gYENSRUFURSBEQVRBQkFTRSBJRiBOT1QgRVhJU1RTICR7ZGF0YWJhc2VOYW1lfSBDSEFSQUNURVIgU0VUICR7Y2hhcmFjdGVyU2V0fWA7XG4gIGlmIChjb2xsYXRpb24pIHtcbiAgICBjcmVhdGVEYlNxbCArPSBgIENPTExBVEUgJHtjb2xsYXRpb259YDtcbiAgfVxuICBjb25zb2xlLmxvZyhgQWJvdXQgdG8gY3JlYXRlIGRhdGFiYXNlOiAke2RhdGFiYXNlTmFtZX1gKTtcbiAgYXdhaXQgY29ubmVjdGlvbi5leGVjdXRlKGNyZWF0ZURiU3FsKTtcbiAgY29uc29sZS5sb2coYERhdGFiYXNlIGNyZWF0ZWQ6ICR7ZGF0YWJhc2VOYW1lfWApO1xuXG4gIGNvbnN0IG93bmVyU2VjcmV0czogVXNlcm5hbWVQYXNzd29yZFtdID0gYXdhaXQgZmV0Y2hBbGxTZWNyZXRzKFxuICAgIG93bmVyU2VjcmV0QXJucyxcbiAgICBzZWNyZXRzTWFuYWdlckNsaWVudFxuICApO1xuICBmb3IgKGNvbnN0IHVzZXJTZWNyZXQgb2Ygb3duZXJTZWNyZXRzKSB7XG4gICAgY29uc3QgbXlzcWxVc2VyID0gYCcke3VzZXJTZWNyZXQudXNlcm5hbWV9J0AnJSdgO1xuICAgIGF3YWl0IGNyZWF0ZVVzZXIoY29ubmVjdGlvbiwgbXlzcWxVc2VyLCB1c2VyU2VjcmV0LnBhc3N3b3JkKTtcbiAgICBhd2FpdCBhcHBseUdyYW50KGNvbm5lY3Rpb24sIG15c3FsVXNlciwgXCJBTEwgUFJJVklMRUdFU1wiKTtcbiAgfVxuXG4gIGNvbnN0IHJlYWRlclNlY3JldHM6IFVzZXJuYW1lUGFzc3dvcmRbXSA9IGF3YWl0IGZldGNoQWxsU2VjcmV0cyhcbiAgICByZWFkZXJTZWNyZXRBcm5zLFxuICAgIHNlY3JldHNNYW5hZ2VyQ2xpZW50XG4gICk7XG4gIGZvciAoY29uc3QgdXNlclNlY3JldCBvZiByZWFkZXJTZWNyZXRzKSB7XG4gICAgY29uc3QgbXlzcWxVc2VyID0gYCcke3VzZXJTZWNyZXQudXNlcm5hbWV9J0AnJSdgO1xuICAgIGF3YWl0IGNyZWF0ZVVzZXIoY29ubmVjdGlvbiwgbXlzcWxVc2VyLCB1c2VyU2VjcmV0LnBhc3N3b3JkKTtcbiAgICBhd2FpdCBhcHBseUdyYW50KGNvbm5lY3Rpb24sIG15c3FsVXNlciwgXCJTRUxFQ1QsIFNIT1cgVklFV1wiKTtcbiAgfVxuXG4gIGNvbnN0IHVucHJpdmlsZWdlZFNlY3JldHM6IFVzZXJuYW1lUGFzc3dvcmRbXSA9IGF3YWl0IGZldGNoQWxsU2VjcmV0cyhcbiAgICB1bnByaXZpbGVnZWRTZWNyZXRBcm5zLFxuICAgIHNlY3JldHNNYW5hZ2VyQ2xpZW50XG4gICk7XG4gIGZvciAoY29uc3QgdXNlclNlY3JldCBvZiB1bnByaXZpbGVnZWRTZWNyZXRzKSB7XG4gICAgY29uc3QgbXlzcWxVc2VyID0gYCcke3VzZXJTZWNyZXQudXNlcm5hbWV9J0AnJSdgO1xuICAgIGF3YWl0IGNyZWF0ZVVzZXIoY29ubmVjdGlvbiwgbXlzcWxVc2VyLCB1c2VyU2VjcmV0LnBhc3N3b3JkKTtcbiAgfVxuXG4gIGxldCBmbHVzaFNxbCA9IFwiRkxVU0ggUFJJVklMRUdFU1wiO1xuICBjb25zb2xlLmxvZyhcIkFib3V0IHRvIGZsdXNoIHByaXZpbGVnZXNcIik7XG4gIGF3YWl0IGNvbm5lY3Rpb24uZXhlY3V0ZShmbHVzaFNxbCk7XG4gIGNvbnNvbGUubG9nKFwiUHJpdmlsZWdlcyBmbHVzaGVkXCIpO1xuXG4gIGNvbnNvbGUubG9nKFwiQWJvdXQgdG8gY2xvc2UgdGhlIGNvbm5lY3Rpb25cIik7XG4gIGF3YWl0IGNvbm5lY3Rpb24uZW5kKCk7XG4gIGNvbnNvbGUubG9nKFwiQ29ubmVjdGlvbiBjbG9zZWRcIik7XG59O1xuXG5tb2R1bGUuZXhwb3J0cyA9IHsgaGFuZGxlciB9O1xuIl19