"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// eslint-disable-next-line import/no-extraneous-dependencies
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
// eslint-disable-next-line import/no-extraneous-dependencies
const db_provision_pgsql_1 = require("@libreworks/db-provision-pgsql");
// eslint-disable-next-line import/no-extraneous-dependencies
const pg_1 = require("pg");
const util_1 = require("./util");
const adminSecretArn = process.env.ADMIN_SECRET_ARN;
const ownerSecretArn = process.env.OWNER_SECRET_ARN;
const databaseName = process.env.DB_NAME;
const encoding = process.env.DB_ENCODING || "UTF8";
const locale = process.env.DB_LOCALE;
const schemaName = process.env.SCHEMA_NAME;
const secretsManagerClient = new client_secrets_manager_1.SecretsManagerClient({});
const ownerSecretArns = (0, util_1.parseJsonArrayFromEnv)("OWNER_SECRETS");
const readerSecretArns = (0, util_1.parseJsonArrayFromEnv)("READER_SECRETS");
const unprivilegedSecretArns = (0, util_1.parseJsonArrayFromEnv)("UNPRIVILEGED_SECRETS");
const handler = async () => {
    // Here's the first network request:
    // Load the admin secret, needed to create the catalog and its owner.
    const adminSecret = await (0, util_1.fetchSecret)(adminSecretArn, secretsManagerClient);
    // Here's the second network request:
    const region = process.env.AWS_REGION || process.env.AWS_DEFAULT_REGION;
    const caBundle = await (0, util_1.readRemote)(process.env.CA_CERTS_URL ||
        `https://truststore.pki.rds.amazonaws.com/${region}/${region}-bundle.pem`);
    // First we need to connect to the "postgres" database.
    const clientDefaults = {
        host: adminSecret.host,
        port: adminSecret.port,
        user: adminSecret.username,
        password: adminSecret.password,
        connectionTimeoutMillis: 40000,
        ssl: { ca: caBundle },
    };
    const client = new pg_1.Client({
        ...clientDefaults,
        database: "postgres",
    });
    const serverClient = new db_provision_pgsql_1.ServerClient(client);
    // Here's our second network request.
    console.log("About to open a connection to database: postgres");
    await client.connect();
    const catalog = new db_provision_pgsql_1.Catalog(databaseName, encoding, locale);
    console.log(`About to create database: ${databaseName}`);
    await serverClient.createDatabase(catalog);
    console.log(`Database created: ${databaseName}`);
    // Here are the next set of network requests:
    // Load all other Secrets Manager secrets in one go.
    const [schemaOwnerSecret, ownerSecrets, readerSecrets, unprivilegedSecrets] = await Promise.all([
        (0, util_1.fetchSecret)(ownerSecretArn, secretsManagerClient),
        (0, util_1.fetchAllSecrets)(ownerSecretArns, secretsManagerClient),
        (0, util_1.fetchAllSecrets)(readerSecretArns, secretsManagerClient),
        (0, util_1.fetchAllSecrets)(unprivilegedSecretArns, secretsManagerClient),
    ]);
    // Map the secrets to the Logins for the sake of performance.
    const loginMap = new Map();
    // Create a Login object for each secret we have. Store it in the map.
    const loginSecrets = [
        schemaOwnerSecret,
        ...ownerSecrets,
        ...readerSecrets,
        ...unprivilegedSecrets,
    ];
    for (const secret of loginSecrets) {
        const login = new db_provision_pgsql_1.Login(secret.username, secret.password);
        loginMap.set(secret, login);
    }
    // Here are the next set of network requests:
    // Let's do this serially until we determine that we can use Promise.all.
    for (const login of loginMap.values()) {
        console.log(`About to create user: ${login.username}`);
        await serverClient.createRole(login);
        console.log(`User created: ${login.username}`);
    }
    const adminRole = new db_provision_pgsql_1.Role(`${databaseName}_${schemaName}_adm`);
    const readerRole = new db_provision_pgsql_1.Role(`${databaseName}_${schemaName}_ro`);
    // Here is the next network request:
    // Create roles for our admin users and read-only users.
    for (const role of [adminRole, readerRole]) {
        console.log(`About to create role: ${role.name}`);
        await serverClient.createRole(role);
        console.log(`Role created: ${role.name}`);
    }
    const grants = [];
    // Create a Grant object for the schema owner.
    let owner = loginMap.get(schemaOwnerSecret);
    grants.push(catalog.grant(owner, "CONNECT", "CREATE", "TEMP"));
    // Create a Grant object for schema admins and all readers.
    for (const secret of [...ownerSecrets, ...readerSecrets]) {
        grants.push(catalog.grant(loginMap.get(secret), "CONNECT", "TEMP"));
    }
    // Create a Grant object for the users with no privileges.
    for (const secret of unprivilegedSecrets) {
        grants.push(catalog.grant(loginMap.get(secret), "CONNECT"));
    }
    // Here are the next set of network requests:
    // Let's do this serially until we determine that we can use Promise.all
    for (const grant of grants) {
        console.log(`About to grant privileges: ${grant.toSql()}`);
        await serverClient.createGrant(grant);
        console.log("Privileges granted");
    }
    console.log("About to close the connection");
    await client.end();
    console.log("Connection closed");
    const schemaClient = new pg_1.Client({
        ...clientDefaults,
        user: owner.name,
        password: owner.password,
        database: databaseName,
    });
    const databaseClient = new db_provision_pgsql_1.DatabaseClient(schemaClient);
    // Here is the next network request:
    console.log(`About to open a connection to database: ${databaseName}`);
    await schemaClient.connect();
    // Here is the next network request:
    const schema = catalog.createSchema(schemaName || owner.username, owner);
    console.log(`About to create schema: ${schema.name} (if it does not already exist)`);
    await schemaClient.query(schema.toSql());
    console.log(`Schema created: ${schema.name}`);
    console.log(`About to alter schema ${schema.name} owner to: ${owner.name}`);
    await schemaClient.query(schema.changeOwner(owner).toSql());
    console.log(`Schema altered: ${schema.name}`);
    const admins = ownerSecrets.map((secret) => loginMap.get(secret));
    const readers = readerSecrets.map((secret) => loginMap.get(secret));
    // Here are the next set of network requests:
    // Create the grants and default permissions for the owner secrets.
    console.log(`About to create schema grants for admins`);
    await databaseClient.createAdminGrants(schema, adminRole, admins);
    // Create the grants and default permissions for the reader secrets.
    console.log(`About to create schema grants for readers`);
    await databaseClient.createReaderGrants(schema, readerRole, readers);
    console.log("About to close the connection");
    await schemaClient.end();
    console.log("Connection closed");
};
module.exports = { handler };
//# sourceMappingURL=data:application/json;base64,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