"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const intrinsic_1 = require("./private/intrinsic");
const stack_1 = require("./stack");
/**
 * Given an object, converts all keys to PascalCase given they are currently in camel case.
 * @param obj The object.
 */
function capitalizePropertyNames(construct, obj) {
    const stack = stack_1.Stack.of(construct);
    obj = stack.resolve(obj);
    if (typeof (obj) !== 'object') {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(x => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, o => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof (o) === 'object' && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
/**
 * Returns a copy of `obj` without `undefined` (or `null`) values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x != null).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value == null) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * A Token that applies a function AFTER resolve resolution
 */
class PostResolveToken extends intrinsic_1.Intrinsic {
    constructor(value, processor) {
        super(value);
        this.processor = processor;
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return super.resolve(context);
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
/**
 * @returns the list of stacks that lead from the top-level stack (non-nested) all the way to a nested stack.
 */
function pathToTopLevelStack(s) {
    if (s.nestedStackParent) {
        return [...pathToTopLevelStack(s.nestedStackParent), s];
    }
    else {
        return [s];
    }
}
exports.pathToTopLevelStack = pathToTopLevelStack;
/**
 * @returns true if this stack is a direct or indirect parent of the nested
 * stack `nested`. If `nested` is a top-level stack, returns false.
 */
function isParentOfNestedStack(parent, child) {
    // if "nested" is not a nested stack, then by definition we cannot be its parent
    if (!child.nestedStackParent) {
        return false;
    }
    // if this is the direct parent, then we found it
    if (parent === child.nestedStackParent) {
        return true;
    }
    // traverse up
    return isParentOfNestedStack(parent, child.nestedStackParent);
}
exports.isParentOfNestedStack = isParentOfNestedStack;
/**
 * Given two arrays, returns the last common element or `undefined` if there
 * isn't (arrays are foriegn).
 */
function findLastCommonElement(path1, path2) {
    let i = 0;
    while (i < path1.length && i < path2.length) {
        if (path1[i] !== path2[i]) {
            break;
        }
        i++;
    }
    return path1[i - 1];
}
exports.findLastCommonElement = findLastCommonElement;
//# sourceMappingURL=data:application/json;base64,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