"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function createTree(context) {
    const root = new lib_1.App();
    const highChild = new lib_1.Construct(root, 'HighChild');
    if (context) {
        Object.keys(context).forEach(key => highChild.node.setContext(key, context[key]));
    }
    const child1 = new lib_1.Construct(highChild, 'Child1');
    const child2 = new lib_1.Construct(highChild, 'Child2');
    const child1_1 = new lib_1.Construct(child1, 'Child11');
    const child1_2 = new lib_1.Construct(child1, 'Child12');
    const child1_1_1 = new lib_1.Construct(child1_1, 'Child111');
    const child2_1 = new lib_1.Construct(child2, 'Child21');
    return {
        root, child1, child2, child1_1, child1_2, child1_1_1, child2_1
    };
}
class MyBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
}
/**
 * Returns a string with a tree representation of this construct and it's children.
 */
function toTreeString(node, depth = 0) {
    let out = '';
    for (let i = 0; i < depth; ++i) {
        out += '  ';
    }
    const name = node.node.id || '';
    out += `${node.constructor.name}${name.length > 0 ? ' [' + name + ']' : ''}\n`;
    for (const child of node.node.children) {
        out += toTreeString(child, depth + 1);
    }
    return out;
}
module.exports = {
    'the "Root" construct is a special construct which can be used as the root of the tree'(test) {
        const root = new lib_1.App();
        test.equal(root.node.id, '', 'if not specified, name of a root construct is an empty string');
        test.ok(!root.node.scope, 'no parent');
        test.equal(root.node.children.length, 1);
        test.done();
    },
    'constructs cannot be created with an empty name unless they are root'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.Construct(root, ''));
        test.done();
    },
    'construct.name returns the name of the construct'(test) {
        const t = createTree();
        test.equal(t.child1.node.id, 'Child1');
        test.equal(t.child2.node.id, 'Child2');
        test.equal(t.child1_1.node.id, 'Child11');
        test.equal(t.child1_2.node.id, 'Child12');
        test.equal(t.child1_1_1.node.id, 'Child111');
        test.equal(t.child2_1.node.id, 'Child21');
        test.done();
    },
    'construct id can use any character except the path separator'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'valid');
        new lib_1.Construct(root, 'ValiD');
        new lib_1.Construct(root, 'Va123lid');
        new lib_1.Construct(root, 'v');
        new lib_1.Construct(root, '  invalid');
        new lib_1.Construct(root, 'invalid   ');
        new lib_1.Construct(root, '123invalid');
        new lib_1.Construct(root, 'in valid');
        new lib_1.Construct(root, 'in_Valid');
        new lib_1.Construct(root, 'in-Valid');
        new lib_1.Construct(root, 'in\\Valid');
        new lib_1.Construct(root, 'in.Valid');
        test.done();
    },
    'if construct id contains path seperators, they will be replaced by double-dash'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Boom/Boom/Bam');
        test.deepEqual(c.node.id, 'Boom--Boom--Bam');
        test.done();
    },
    'if "undefined" is forcefully used as an "id", it will be treated as an empty string'(test) {
        const c = new lib_1.Construct(undefined, undefined);
        test.deepEqual(c.node.id, '');
        test.done();
    },
    "dont allow unresolved tokens to be used in construct IDs"(test) {
        // GIVEN
        const root = new lib_1.App();
        const token = lib_1.Lazy.stringValue({ produce: () => 'lazy' });
        // WHEN + THEN
        test.throws(() => new lib_1.Construct(root, `MyID: ${token}`), /Cannot use tokens in construct ID: MyID: \${Token/);
        test.done();
    },
    'construct.uniqueId returns a tree-unique alphanumeric id of this construct'(test) {
        const root = new lib_1.App();
        const child1 = new lib_1.Construct(root, 'This is the first child');
        const child2 = new lib_1.Construct(child1, 'Second level');
        const c1 = new lib_1.Construct(child2, 'My construct');
        const c2 = new lib_1.Construct(child1, 'My construct');
        test.deepEqual(c1.node.path, 'This is the first child/Second level/My construct');
        test.deepEqual(c2.node.path, 'This is the first child/My construct');
        test.deepEqual(c1.node.uniqueId, 'ThisisthefirstchildSecondlevelMyconstruct202131E0');
        test.deepEqual(c2.node.uniqueId, 'ThisisthefirstchildMyconstruct8C288DF9');
        test.done();
    },
    'cannot calculate uniqueId if the construct path is ["Default"]'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Default');
        test.throws(() => c.node.uniqueId, /Unable to calculate a unique id for an empty set of components/);
        test.done();
    },
    'construct.getChildren() returns an array of all children'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Child1');
        new lib_1.Construct(root, 'Child2');
        test.equal(child.node.children.length, 0, 'no children');
        test.equal(root.node.children.length, 3, 'three children are expected');
        test.done();
    },
    'construct.findChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.tryFindChild(child.node.id), child, 'findChild(name) can be used to retrieve the child from a parent');
        test.ok(!root.node.tryFindChild('NotFound'), 'findChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.getChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.findChild(child.node.id), child, 'getChild(name) can be used to retrieve the child from a parent');
        test.throws(() => {
            root.node.findChild('NotFound');
        }, '', 'getChild(name) returns undefined if the child is not found');
        test.done();
    },
    'can remove children from the tree using tryRemoveChild()'(test) {
        const root = new lib_1.App();
        const childrenBeforeAdding = root.node.children.length; // Invariant to adding 'Metadata' resource or not
        // Add & remove
        const child = new lib_1.Construct(root, 'Construct');
        test.equals(true, root.node.tryRemoveChild(child.node.id));
        test.equals(false, root.node.tryRemoveChild(child.node.id)); // Second time does nothing
        test.equals(undefined, root.node.tryFindChild(child.node.id));
        test.equals(childrenBeforeAdding, root.node.children.length);
        test.done();
    },
    'construct.toString() and construct.toTreeString() can be used for diagnostics'(test) {
        const t = createTree();
        test.equal(t.root.toString(), '<root>');
        test.equal(t.child1_1_1.toString(), 'HighChild/Child1/Child11/Child111');
        test.equal(t.child2.toString(), 'HighChild/Child2');
        test.equal(toTreeString(t.root), 'App\n  TreeMetadata [Tree]\n  Construct [HighChild]\n    Construct [Child1]\n      Construct [Child11]\n        Construct [Child111]\n      Construct [Child12]\n    Construct [Child2]\n      Construct [Child21]\n');
        test.done();
    },
    'construct.getContext(key) can be used to read a value from context defined at the root level'(test) {
        const context = {
            ctx1: 12,
            ctx2: 'hello'
        };
        const t = createTree(context);
        test.equal(t.child1_2.node.tryGetContext('ctx1'), 12);
        test.equal(t.child1_1_1.node.tryGetContext('ctx2'), 'hello');
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'construct.setContext(k,v) sets context at some level and construct.getContext(key) will return the lowermost value defined in the stack'(test) {
        const root = new lib_1.App();
        const highChild = new lib_1.Construct(root, 'highChild');
        highChild.node.setContext('c1', 'root');
        highChild.node.setContext('c2', 'root');
        const child1 = new lib_1.Construct(highChild, 'child1');
        child1.node.setContext('c2', 'child1');
        child1.node.setContext('c3', 'child1');
        const child2 = new lib_1.Construct(highChild, 'child2');
        const child3 = new lib_1.Construct(child1, 'child1child1');
        child3.node.setContext('c1', 'child3');
        child3.node.setContext('c4', 'child3');
        test.equal(highChild.node.tryGetContext('c1'), 'root');
        test.equal(highChild.node.tryGetContext('c2'), 'root');
        test.equal(highChild.node.tryGetContext('c3'), undefined);
        test.equal(child1.node.tryGetContext('c1'), 'root');
        test.equal(child1.node.tryGetContext('c2'), 'child1');
        test.equal(child1.node.tryGetContext('c3'), 'child1');
        test.equal(child2.node.tryGetContext('c1'), 'root');
        test.equal(child2.node.tryGetContext('c2'), 'root');
        test.equal(child2.node.tryGetContext('c3'), undefined);
        test.equal(child3.node.tryGetContext('c1'), 'child3');
        test.equal(child3.node.tryGetContext('c2'), 'child1');
        test.equal(child3.node.tryGetContext('c3'), 'child1');
        test.equal(child3.node.tryGetContext('c4'), 'child3');
        test.done();
    },
    'construct.setContext(key, value) can only be called before adding any children'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'child1');
        test.throws(() => root.node.setContext('k', 'v'));
        test.done();
    },
    'fails if context key contains unresolved tokens'(test) {
        const root = new lib_1.App();
        test.throws(() => root.node.setContext(`my-${lib_1.Aws.REGION}`, 'foo'), /Invalid context key/);
        test.throws(() => root.node.tryGetContext(lib_1.Aws.REGION), /Invalid context key/);
        test.done();
    },
    'construct.pathParts returns an array of strings of all names from root to node'(test) {
        const tree = createTree();
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'HighChild/Child1/Child11/Child111');
        test.deepEqual(tree.child2.node.path, 'HighChild/Child2');
        test.done();
    },
    'if a root construct has a name, it should be included in the path'(test) {
        const tree = createTree({});
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'HighChild/Child1/Child11/Child111');
        test.done();
    },
    'construct can not be created with the name of a sibling'(test) {
        const root = new lib_1.App();
        // WHEN
        new lib_1.Construct(root, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(root, 'SameName');
        }, /There is already a Construct with name 'SameName' in App/);
        // WHEN
        const c0 = new lib_1.Construct(root, 'c0');
        new lib_1.Construct(c0, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(c0, 'SameName');
        }, /There is already a Construct with name 'SameName' in Construct \[c0\]/);
        test.done();
    },
    'addMetadata(type, data) can be used to attach metadata to constructs FIND_ME'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        test.deepEqual(con.node.metadata, [], 'starts empty');
        con.node.addMetadata('key', 'value');
        con.node.addMetadata('number', 103);
        con.node.addMetadata('array', [123, 456]);
        test.deepEqual(con.node.metadata[0].type, 'key');
        test.deepEqual(con.node.metadata[0].data, 'value');
        test.deepEqual(con.node.metadata[1].data, 103);
        test.deepEqual(con.node.metadata[2].data, [123, 456]);
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace[0].indexOf('FIND_ME') !== -1, 'First stack line should include this function\s name');
        test.done();
    },
    'addMetadata(type, undefined/null) is ignored'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'Foo');
        con.node.addMetadata('Null', null);
        con.node.addMetadata('Undefined', undefined);
        con.node.addMetadata('True', true);
        con.node.addMetadata('False', false);
        con.node.addMetadata('Empty', '');
        const exists = (key) => con.node.metadata.find(x => x.type === key);
        test.ok(!exists('Null'));
        test.ok(!exists('Undefined'));
        test.ok(exists('True'));
        test.ok(exists('False'));
        test.ok(exists('Empty'));
        test.done();
    },
    'addWarning(message) can be used to add a "WARNING" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addWarning('This construct is deprecated, use the other one instead');
        test.deepEqual(con.node.metadata[0].type, cxapi.WARNING_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'This construct is deprecated, use the other one instead');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addError(message) can be used to add a "ERROR" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addError('Stop!');
        test.deepEqual(con.node.metadata[0].type, cxapi.ERROR_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Stop!');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addInfo(message) can be used to add an "INFO" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addInfo('Hey there, how do you do?');
        test.deepEqual(con.node.metadata[0].type, cxapi.INFO_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Hey there, how do you do?');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'multiple children of the same type, with explicit names are welcome'(test) {
        const root = new lib_1.App();
        new MyBeautifulConstruct(root, 'mbc1');
        new MyBeautifulConstruct(root, 'mbc2');
        new MyBeautifulConstruct(root, 'mbc3');
        new MyBeautifulConstruct(root, 'mbc4');
        test.ok(root.node.children.length >= 4);
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'construct.validate() can be implemented to perform validation, ConstructNode.validate(construct.node) will return all errors from the subtree (DFS)'(test) {
        class MyConstruct extends lib_1.Construct {
            validate() {
                return ['my-error1', 'my-error2'];
            }
        }
        class YourConstruct extends lib_1.Construct {
            validate() {
                return ['your-error1'];
            }
        }
        class TheirConstruct extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new YourConstruct(this, 'YourConstruct');
            }
            validate() {
                return ['their-error'];
            }
        }
        class TestStack extends lib_1.App {
            constructor() {
                super();
                new MyConstruct(this, 'MyConstruct');
                new TheirConstruct(this, 'TheirConstruct');
            }
            validate() {
                return ['stack-error'];
            }
        }
        const stack = new TestStack();
        const errors = lib_1.ConstructNode.validate(stack.node).map((v) => ({ path: v.source.node.path, message: v.message }));
        // validate DFS
        test.deepEqual(errors, [
            { path: 'MyConstruct', message: 'my-error1' },
            { path: 'MyConstruct', message: 'my-error2' },
            { path: 'TheirConstruct/YourConstruct', message: 'your-error1' },
            { path: 'TheirConstruct', message: 'their-error' },
            { path: '', message: 'stack-error' }
        ]);
        test.done();
    },
    'construct.lock() protects against adding children anywhere under this construct (direct or indirect)'(test) {
        class LockableConstruct extends lib_1.Construct {
            lockMe() {
                this.node._lock();
            }
            unlockMe() {
                this.node._unlock();
            }
        }
        const stack = new lib_1.App();
        const c0a = new LockableConstruct(stack, 'c0a');
        const c0b = new lib_1.Construct(stack, 'c0b');
        const c1a = new lib_1.Construct(c0a, 'c1a');
        const c1b = new lib_1.Construct(c0a, 'c1b');
        c0a.lockMe();
        // now we should still be able to add children to c0b, but not to c0a or any its children
        new lib_1.Construct(c0b, 'c1a');
        test.throws(() => new lib_1.Construct(c0a, 'fail1'), /Cannot add children to "c0a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1a, 'fail2'), /Cannot add children to "c0a\/c1a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1b, 'fail3'), /Cannot add children to "c0a\/c1b" during synthesis/);
        c0a.unlockMe();
        new lib_1.Construct(c0a, 'c0aZ');
        new lib_1.Construct(c1a, 'c1aZ');
        new lib_1.Construct(c1b, 'c1bZ');
        test.done();
    },
    'findAll returns a list of all children in either DFS or BFS'(test) {
        // GIVEN
        const c1 = new lib_1.Construct(undefined, '1');
        const c2 = new lib_1.Construct(c1, '2');
        new lib_1.Construct(c1, '3');
        new lib_1.Construct(c2, '4');
        new lib_1.Construct(c2, '5');
        // THEN
        test.deepEqual(c1.node.findAll().map(x => x.node.id), c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id)); // default is PreOrder
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id), ['1', '2', '4', '5', '3']);
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.POSTORDER).map(x => x.node.id), ['4', '5', '2', '3', '1']);
        test.done();
    },
    'ancestors returns a list of parents up to root'(test) {
        const { child1_1_1 } = createTree();
        test.deepEqual(child1_1_1.node.scopes.map(x => x.node.id), ['', 'HighChild', 'Child1', 'Child11', 'Child111']);
        test.done();
    },
    '"root" returns the root construct'(test) {
        const { child1, child2, child1_1_1, root } = createTree();
        test.ok(child1.node.root === root);
        test.ok(child2.node.root === root);
        test.ok(child1_1_1.node.root === root);
        test.done();
    },
    'defaultChild': {
        'returns the child with id "Resource"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Resource');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns the child with id "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'can override defaultChild'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'Resource');
            const defaultChild = new lib_1.Construct(root, 'OtherResource');
            root.node.defaultChild = defaultChild;
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns "undefined" if there is no default'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'child2');
            test.equal(root.node.defaultChild, undefined);
            test.done();
        },
        'fails if there are both "Resource" and "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            new lib_1.Construct(root, 'Resource');
            test.throws(() => root.node.defaultChild, /Cannot determine default child for . There is both a child with id "Resource" and id "Default"/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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