"use strict";
/*
{
  "items": [
    {
      "id": 7,
      "name": "Laubbäume",
      "uuid": "eb3eba32-1d46-4a6a-87bf-328672e2626c",
      "space": {
        "2ff6f540-1676-410e-8067-1b58c3a002ed": [
          [
            4,
            5
          ]
        ],
        "3e9f0c09-a118-4d2c-9cf7-ea52fb1b80ce": [
          1,
          3
        ],
        "43b549b3-911d-4df3-8c9c-9986ed02be45": [
          [
            241,
            45,
            45,
            1
          ],
          [
            [
              76,
              26,
              26,
              1
            ],
            [
              146,
              146,
              146,
              1
            ]
          ]
        ],
        "cee8ddbc-5982-47a8-93a6-c82930acd69d": [
          {
            "taxa": [
              {
                "name_uuid": "151d41f5-5941-4169-b77b-175ab0876ca6",
                "taxon_nuid": "006",
                "taxon_author": null,
                "taxon_source": "taxonomy.sources.col",
                "taxon_latname": "Plantae"
              },
              {
                "name_uuid": "c303f1b7-feec-45cd-b615-f0773ced2107",
                "taxon_nuid": "003",
                "taxon_author": "admin",
                "taxon_source": "taxonomy.sources.custom",
                "taxon_latname": "Plantae"
              }
            ],
            "latname": "Plantae",
            "is_custom": false
          }
        ]
      },
      "taxon": {
        "name_uuid": "bd143912-7a66-4f7f-ad25-c3c16a361273",
        "taxon_nuid": "00600800601d008001",
        "taxon_author": "",
        "taxon_source": "taxonomy.sources.col",
        "taxon_latname": "Acer"
      },
      "image_url": "/media/treesofbavaria/imagestore/1/thumbnails/73c0d5e7970ab4ed9b855dc0a4535dc5_2E9oVUY-e40c54f70e9a62da34083d82850ac09b-400.jpg",
      "node_type": "node",
      "is_visible": true,
      "max_points": 150,
      "meta_node_id": 7,
      "decision_rule": "mit Blättern, im Winter kahl"
    },
    {
      "id": 8,
      "name": "Nadelbäume",
      "uuid": "6fcd3ac4-2e64-4ba4-858b-0acae18fbcfb",
      "space": {
        "0c9f1e62-521e-4c09-b98e-156c3d72a9bc": [
          "<p>Text mit &gt;</p>"
        ],
        "2ff6f540-1676-410e-8067-1b58c3a002ed": [
          [
            1,
            5
          ]
        ],
        "3e9f0c09-a118-4d2c-9cf7-ea52fb1b80ce": [
          3
        ],
        "43b549b3-911d-4df3-8c9c-9986ed02be45": [
          [
            241,
            45,
            45,
            1
          ]
        ],
        "cee8ddbc-5982-47a8-93a6-c82930acd69d": []
      },
      "taxon": null,
      "image_url": "/media/treesofbavaria/imagestore/1/thumbnails/1659ca81b47f6be0b1b1ca8477c5f454_PkstfHA-e40c54f70e9a62da34083d82850ac09b-400.jpg",
      "node_type": "node",
      "is_visible": true,
      "max_points": 200,
      "meta_node_id": 8,
      "decision_rule": "mit Nadeln"
    },
    {
      "id": 157,
      "name": "Nadelbaueme Kopie QV",
      "uuid": "8727707d-aebc-4366-a46d-b7e25f1ae88e",
      "space": {
        "0c9f1e62-521e-4c09-b98e-156c3d72a9bc": [
          "<p>Text mit &gt;</p>"
        ],
        "27c566f8-802a-47fd-8938-2e469aef22b0": [
          "<p>Element 1</p>"
        ],
        "3e9f0c09-a118-4d2c-9cf7-ea52fb1b80ce": [
          2
        ],
        "cee8ddbc-5982-47a8-93a6-c82930acd69d": []
      },
      "taxon": null,
      "image_url": "/media/treesofbavaria/imagestore/1/thumbnails/1659ca81b47f6be0b1b1ca8477c5f454_PkstfHA-e40c54f70e9a62da34083d82850ac09b-400.jpg",
      "node_type": "node",
      "is_visible": true,
      "max_points": 150,
      "meta_node_id": 157,
      "decision_rule": ""
    },
    {
      "id": 59,
      "name": "Unready",
      "uuid": "960d0c2c-2c30-48a0-8ddb-7e542ee1a152",
      "space": {
        "3e9f0c09-a118-4d2c-9cf7-ea52fb1b80ce": [
          1
        ],
        "cee8ddbc-5982-47a8-93a6-c82930acd69d": []
      },
      "taxon": null,
      "image_url": "/static/noimage.png",
      "node_type": "node",
      "is_visible": true,
      "max_points": 50,
      "meta_node_id": 59,
      "decision_rule": ""
    },
    {
      "id": 162,
      "name": "Eintrag mit Synonym",
      "uuid": "4342a9b7-5883-47f2-a22b-bd0ea0af2553",
      "space": {
        "cee8ddbc-5982-47a8-93a6-c82930acd69d": [
          {
            "taxa": [
              {
                "name_uuid": "151d41f5-5941-4169-b77b-175ab0876ca6",
                "taxon_nuid": "006",
                "taxon_author": null,
                "taxon_source": "taxonomy.sources.col",
                "taxon_latname": "Plantae"
              },
              {
                "name_uuid": "c303f1b7-feec-45cd-b615-f0773ced2107",
                "taxon_nuid": "003",
                "taxon_author": "admin",
                "taxon_source": "taxonomy.sources.custom",
                "taxon_latname": "Plantae"
              }
            ],
            "latname": "Plantae",
            "is_custom": false
          }
        ]
      },
      "taxon": {
        "name_uuid": "8dfdae06-8626-46ac-8611-86926973ae72",
        "taxon_nuid": "00600800800b003002009001",
        "taxon_author": "(Bak.) Reed",
        "taxon_source": "taxonomy.sources.col",
        "taxon_latname": "Schizaea malaccana robustior"
      },
      "image_url": "/static/noimage.png",
      "node_type": "result",
      "is_visible": true,
      "max_points": 0,
      "meta_node_id": 162,
      "decision_rule": ""
    }
  ],
  "matrix_filters": {
    "0c9f1e62-521e-4c09-b98e-156c3d72a9bc": {
      "name": "Test text",
      "type": "TextOnlyFilter",
      "weight": 50,
      "allow_multiple_values": false
    },
    "27c566f8-802a-47fd-8938-2e469aef22b0": {
      "name": "Dreifach Abhängig",
      "type": "DescriptiveTextAndImagesFilter",
      "weight": 50,
      "restrictions": {
        "2ff6f540-1676-410e-8067-1b58c3a002ed": [
          [
            2,
            4
          ]
        ]
      },
      "allow_multiple_values": false
    },
    "2ff6f540-1676-410e-8067-1b58c3a002ed": {
      "name": "Doppelt Abhängig",
      "type": "RangeFilter",
      "weight": 50,
      "restrictions": {
        "3e9f0c09-a118-4d2c-9cf7-ea52fb1b80ce": [
          2
        ]
      },
      "allow_multiple_values": false
    },
    "3e9f0c09-a118-4d2c-9cf7-ea52fb1b80ce": {
      "name": "Abhängiges Merkmal - Farbe rot",
      "type": "NumberFilter",
      "weight": 50,
      "restrictions": {
        "43b549b3-911d-4df3-8c9c-9986ed02be45": [
          [
            241,
            45,
            45,
            1
          ]
        ]
      },
      "allow_multiple_values": false
    },
    "43b549b3-911d-4df3-8c9c-9986ed02be45": {
      "name": "Farbe",
      "type": "ColorFilter",
      "weight": 50,
      "allow_multiple_values": false
    },
    "cee8ddbc-5982-47a8-93a6-c82930acd69d": {
      "name": "Taxonomic filter",
      "type": "TaxonFilter",
      "weight": 50,
      "allow_multiple_values": false
    }
  }
}
*/

var MATRIX_FILTERS = {};

var MATRIX_ITEMS = {};

const MODE_STRICT = "strict"; // sort items out immediately
const MODE_FLUID = "fluid"; // do not sort out, sort by points in descending order

var IDENTIFICATION_MODE = MODE_STRICT;

const STATUS_ACTIVE = "active";
const STATUS_INACTIVE = "inactive";


class MatrixFilterSpace {

  constructor(matrix_filter, space_identifier) {

    this.DEBUG = true;

    this.matrix_filter = matrix_filter;

    this.space_identifier = space_identifier;

    // if a MatrixFilter is restricted by this MatrixfilterSpace, the MatrixFilter is listed in restricts{}
    //{matrix_filter_uuid:matrix_filter}
    this.restricts = {};

    this.input_element = null;

    this.matching_matrix_items = {};
    this.mismatching_matrix_items = {};


    // (STRICT MODE) store currently active MatrixItems for this MatrixFilterSpace
    // if this is empty, this MatrixfilterSpace becomes impossible
    this.active_matching_matrix_items = {};

    // selected by the user or unselected
    this.status = STATUS_INACTIVE;

    // Strict mode: this filter is within the current of all remaining  MatrixItems
    this.is_possible = true;

  }

  // register a MatrixFilter restricted by this space
  register_restricted_matrix_filter(restricted_matrix_filter) {
    this.restricts[restricted_matrix_filter.uuid] = restricted_matrix_filter;
  }

  register_matching_matrix_item(matrix_item) {
    this.matching_matrix_items[matrix_item.uuid] = matrix_item;
    // initially all MatrixItems are active
    this.active_matching_matrix_items[matrix_item.uuid] = matrix_item;
  }

  register_mismatching_matrix_item(matrix_item) {
    this.mismatching_matrix_items[matrix_item.uuid] = matrix_item;
  }

  // DOM BINDINGS
  bind_input_element(input_element) {

    if (this.input_element != null) {
      throw new Error("[MatrixFilterSpace] " + this.space_identifier + " of " + this.matrix_filter.matrix_filter_type + " is already bound to an input element");
    }

    var self = this;
    this.input_element = input_element;

    input_element.addEventListener("change", function (event) {
      if (self.DEBUG == true) {
        console.log("[MatrixFilterSpace] new state of input # " + event.currentTarget.id + " : " + event.currentTarget.checked);
      }
      self.on_change(event);
    });

  }

  get_input_element () {
    return this.input_element;
  }

  on_change(event) {

    let input = event.currentTarget;
    let checked = input.checked;
    if (checked == true) {
      this.activate();
    }
    else {
      this.deactivate();
    }
  }

  get_event_data () {
    
    const event_data = {
      detail : {
        matrix_filter : {
          uuid : this.matrix_filter.uuid
        },
        space_identifier: this.space_identifier
      }
    };

    return event_data;
  }

  // if the user clicked on a input element to select the filter,  MatrixFilterSpace becomes active
  // depending on IDENTIFICATION_MODE, this has different consequences
  // MODE_STRICT:
  //  - this space is added to MatrixItem.mismatching_spaces[matrix_filter_uuid][space_identifier]
  //  - all matrix items in this.mismatching_matrix_items have to be informed
  // MODE_FLUID:
  //  - all matrix items in this.matching_matrix_items have to be informed
  // BOTH:
  //  - all MatrixFilters listed in this.restricts are informed
  activate() {

    if (this.DEBUG == true){
      console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " is being activated.");
    }

    this.status = STATUS_ACTIVE; 

    if (this.input_element.checked == false){
      this.input_element.checked = true;
    }

    this.input_element.setAttribute("checked", true);

    // work restrictions
    this.signal_restricted_matrix_filters(true);

    // notify matrix items
    this.signal_matrix_items();

    // fire event
    const turned_on_event = new Event("turnedOn");
		this.input_element.dispatchEvent(turned_on_event);

    // deactivate all other spaces if only one is allowed
    if (this.matrix_filter.allow_multiple_values == false){

      if (this.DEBUG == true){
        console.log("[MatrixFilterSpace] no multiple values allowed, deactivating all other spaces for " + this.matrix_filter.matrix_filter_type);
      }

      for (let space_identifier in this.matrix_filter.matrix_filter_spaces){
        if (space_identifier != this.space_identifier){
          let matrix_filter_space = this.matrix_filter.matrix_filter_spaces[space_identifier];

          if (matrix_filter_space.status == STATUS_ACTIVE){
            matrix_filter_space.deactivate();
          }
        }
      }
    }

    this.post_activation();
  }

  // hook
  post_activation () {}

  // if the user clicked on a input element to UNselect the filter,  MatrixFilterSpace becomes INactive 
  deactivate() {

    if (this.status == STATUS_ACTIVE){

      if (this.DEBUG == true){
        console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " is being deactivated.");
      }
      
      this.status = STATUS_INACTIVE;

      if (this.input_element.checked == true){
        this.input_element.checked = false;
      }

      this.input_element.removeAttribute("checked");

      // work restrictions
      this.signal_restricted_matrix_filters(false);

      // notify matrix items
      this.signal_matrix_items();

      const turned_off_event = new Event("turnedOff");
      this.input_element.dispatchEvent(turned_off_event);
    }

  }

  // signal all matrix_filters which are restricted by this space_identifier of this matrix_filter
  signal_restricted_matrix_filters(is_turned_on) {

    if (this.DEBUG == true){
      console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " " + this.space_identifier + " restricts: ");
      console.log(this.restricts);
    }

    for (let matrix_filter_uuid in this.restricts) {

      let restricted_matrix_filter = this.restricts[matrix_filter_uuid];

      if (this.DEBUG == true) {
        console.log("[MatrixFilterSpace] " + this.space_identifier + " of " + this.matrix_filter.matrix_filter_type + " is signaling MatrixFilter " + restricted_matrix_filter.matrix_filter_type + " " + restricted_matrix_filter.uuid);
      }

      restricted_matrix_filter.receive_restriction_update(this, is_turned_on);
    }

  }

  signal_matrix_items (){
    // always signal matching matrix items
    this.signal_matching_matrix_items();

    if (IDENTIFICATION_MODE == MODE_STRICT){
      this.signal_mismatching_matrix_items();
    }
  }

  signal_matching_matrix_items (){
    if (this.DEBUG == true){
			console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " " + this.space_identifier + " is signaling matching_matrix_items");
		}

    for (let matrix_item_uuid in this.matching_matrix_items){
      let matrix_item = this.matching_matrix_items[matrix_item_uuid];

      if (this.status == STATUS_ACTIVE) {
        matrix_item.add_matching_matrix_filter_space(this);
      }
      else {
        matrix_item.remove_matching_matrix_filter_space(this);
      }
    }
  }

  signal_mismatching_matrix_items (){
    if (this.DEBUG == true){
			console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " " + this.space_identifier + " is signaling mismatching_matrix_items");
		}

    for (let matrix_item_uuid in this.mismatching_matrix_items){
      let matrix_item = this.mismatching_matrix_items[matrix_item_uuid];
      if (this.status == STATUS_ACTIVE) {
        matrix_item.add_mismatching_matrix_filter_space(this);
      }
      else {
        matrix_item.remove_mismatching_matrix_filter_space(this);
      }
    }
  }

  // manage visibility of this space
  get_is_possible() {
    let is_possible = Object.keys(this.active_matching_matrix_items).length == 0 ? false : true;
    return is_possible;
  }

  activate_matching_matrix_item(matrix_item) {
    
    if (this.DEBUG == true){
			console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " " + this.space_identifier + " now adds MatrixItem " + matrix_item.name + " to active_matching_matrix_items");
		}

    let is_initially_possible = this.get_is_possible();

    if (!this.active_matching_matrix_items.hasOwnProperty(matrix_item.uuid)){
      this.active_matching_matrix_items[matrix_item.uuid] = matrix_item;
    }

    this.is_possible = true;

    if (is_initially_possible == false){
      this.send_possible_event();
    }

  }

  deactivate_matching_matrix_item(matrix_item) {
    if (this.DEBUG == true){
			console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " " + this.space_identifier + " now removes MatrixItem " + matrix_item.name + " from active_matching_matrix_items");
		}

    let is_initially_possible = this.get_is_possible();

    if (this.active_matching_matrix_items.hasOwnProperty(matrix_item.uuid)){
      delete this.active_matching_matrix_items[matrix_item.uuid];
    }

    let is_possible = this.get_is_possible();

    if (is_possible == false){
      this.is_possible = false;
      if (is_initially_possible == true){
        this.send_impossible_event();
      }
    }
  }

  // send an event if this.is_possible changes from false to true
  send_possible_event() {
    if (this.DEBUG == true){
			console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " " + this.space_identifier + " became possible");
		}

    const event_data = this.get_event_data();
    const possible_event = new CustomEvent("possible", event_data);

    let input_element = this.get_input_element();
    input_element.dispatchEvent(possible_event);
  }

  // send an event if this.is_possible changes from true to false
  send_impossible_event() {
    if (this.DEBUG == true){
			console.log("[MatrixFilterSpace] " + this.matrix_filter.matrix_filter_type + " " + this.space_identifier + " became impossible");
		}

    const event_data = this.get_event_data();
    const impossible_event = new CustomEvent("impossible", event_data);

    let input_element = this.get_input_element();
    input_element.dispatchEvent(impossible_event);
  }

}

class DescriptiveTextAndImagesSpace extends MatrixFilterSpace {
}

class TextOnlySpace extends MatrixFilterSpace {

}

class ColorSpace extends MatrixFilterSpace {

}

class NumberSpace extends MatrixFilterSpace {

}

class RangeSpace extends MatrixFilterSpace {

  // event listener is bound to Rangefilter
  bind_input_element(input_element) {
    this.input_element = input_element;
  }

  on_change(event) {

    throw new Error("[RangeSpace] method on_change can only be called from RangeFilter");
  }

  get_input_element () {
    return this.matrix_filter.input_element;
  }

  // called from RangeFilter if the selected value is within this range
  activate() {    
    if (this.status == STATUS_INACTIVE){

      if (this.DEBUG == true){
        let space_str = this.matrix_filter.get_space_str_from_space_identifier(this.space_identifier);
        console.log("[RangeSpace] " + space_str + " is now active");
      }

      this.status = STATUS_ACTIVE;
      // work restrictions
      this.signal_restricted_matrix_filters(true);

      // notify matrix items
      this.signal_matrix_items();

      this.post_activation();

    }
  }

  // called from RangeFilter if the selected value is NOT within this range
  // and on reset
  deactivate() {

    if (this.status == STATUS_ACTIVE){

      if (this.DEBUG == true){
        let space_str = this.matrix_filter.get_space_str_from_space_identifier(this.space_identifier);
        console.log("[RangeSpace] " + space_str + " is now inactive");
      }

      this.status = STATUS_INACTIVE;

      // work restrictions
      this.signal_restricted_matrix_filters(false);

      // notify matrix items
      this.signal_matrix_items();

    }
  }

}

class TaxonFilterSpace extends MatrixFilterSpace {

}


class MatrixFilter {

  constructor(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition) {
    this.DEBUG = true;

    // for firing events
    this.form_element = form_element;

    this.uuid = uuid;
    this.matrix_filter_type = matrix_filter_type;

    this.name = name;

    this.allow_multiple_values = allow_multiple_values;

    this.definition = definition

    // {space_identifier : MatrixFilterSpace}
    // contains the spaces of this filter
    this.matrix_filter_spaces = {};

    // if this filter is restricted by another filter, the "other" filter is listed in restricted_by{}
    // {matrix_filter_uuid : { space_identifier : false } }
    // if all space_identifiers are set to true, show the filter
    this.restricted_by = {};

    // active: unrestricted filter, or all restrictions are met
    this.status = STATUS_ACTIVE;

    this.weight = weight;

    // map space_identifiers to matrix_items
    // {spaceIdentifier : [matrixItemUuid, matrixItemUuid]}
    this.matrix_item_registry = {};

    // for strict mode only
    // {spaceIdentifier: [matrixItemUuid, matrixItemUuid]}
    // if a space has no active items, it will be disabled in strict mode
    this.active_matrix_items = {}
  }

  //
  // MATRIX FILTER SPACE MANAGEMENT
  //

  // return b64 encoded space_identifier
  // space is the parsed space from JSON.parse
  get_space_identifier_from_parsed(space) {
    throw new Error("[MatrixFilter] subclasses require a get_space_identifier_from_parsed method");
  }

  // return b64 encoded space_identifier
  get_space_identifier_from_str(space_str) {
    throw new Error("[MatrixFilter] subclasses require a get_space_identifier_from_str method");
  }

  parse_space_str(space_str) {
    throw new Error("[MatrixFilter] subclasses require a parse_matrix_filter_space_str method");
  }

  // space_identifier is the b64 encoded space_str
  get_space_from_space_identifier(space_identifier) {
    throw new Error("[MatrixFilter] subclasses require a get_space_from_space_identifier method");
  }

  get_space_str_from_space_identifier(space_identifier) {
    throw new Error("[MatrixFilter] subclasses require a get_space_str_from_space_identifier method");
  }

  // space is the parsed space from JSON.parse()
  add_matrix_filter_space(space) {

    if (this.DEBUG == true){
      console.log("[MatrixFilter] " + this.matrix_filter_type + " adding space: ");
      console.log(space);
    }

    var space_identifier = this.get_space_identifier_from_parsed(space);
    var matrix_filter_space = null

    // only register every space_identifier once
    if (!this.matrix_filter_spaces.hasOwnProperty(space_identifier)) {
      matrix_filter_space = new this.MatrixFilterSpaceClass(this, space_identifier);
      this.matrix_filter_spaces[space_identifier] = matrix_filter_space;
    }
    else {
      matrix_filter_space = this.matrix_filter_spaces[space_identifier];
    }

    return matrix_filter_space;
  }

  //
  // RESTRICTION MANAGEMENT
  //
  // if this MatrixFilter is restricted by the space of another filter, register with the restrictive space,
  // AND register with this.restricted_by[matrix_filter_uuid][space_identifier] = BOOLEAN
  apply_restrictions(restrictions) {

    for (let restrictive_matrix_filter_uuid in restrictions) {

      let restrictive_matrix_filter = MATRIX_FILTERS[restrictive_matrix_filter_uuid];

      if (this.DEBUG == true){
        console.log("[MatrixFilter] applying restrictions. " + this.matrix_filter_type + " is restricted by " + restrictive_matrix_filter.matrix_filter_type);
      }

      let restriction_spaces = restrictions[restrictive_matrix_filter_uuid];

      this.restricted_by[restrictive_matrix_filter_uuid] = {};

      for (let v = 0; v < restriction_spaces.length; v++) {

        let parsed_restriction_space = restriction_spaces[v];

        let space_identifier = restrictive_matrix_filter.get_space_identifier_from_parsed(parsed_restriction_space);

        console.log("[MatrixFilter] applying restriction to " + this.matrix_filter_type + ": " + restrictive_matrix_filter.matrix_filter_type + " " + parsed_restriction_space + " " + space_identifier);

        this.restricted_by[restrictive_matrix_filter_uuid][space_identifier] = false;

        // e.g. RangeFilter might not have that space added yet. The RangeSpace can be part of the restriction,
        // but nor part of any MatrixItem
        if (!restrictive_matrix_filter.matrix_filter_spaces.hasOwnProperty(space_identifier)){
          if (this.DEBUG == true){
            console.log("[MatrixFilter] trying to restrict with yet unknown space_identifier: " + space_identifier);
          }
          let matrix_filter_space = restrictive_matrix_filter.add_matrix_filter_space(parsed_restriction_space);
        }

        let restrictive_matrix_filter_space = restrictive_matrix_filter.matrix_filter_spaces[space_identifier];
        restrictive_matrix_filter_space.register_restricted_matrix_filter(this);
      }
    }

    if (Object.keys(this.restricted_by).length > 0) {
      this.deactivate();
    }
  }

  receive_restriction_update(matrix_filter_space, is_turned_on) {

    let restricting_matrix_filter = matrix_filter_space.matrix_filter;
    let space_identifier = matrix_filter_space.space_identifier;

    if (this.restricted_by.hasOwnProperty(restricting_matrix_filter.uuid)) {
      let restriction = this.restricted_by[restricting_matrix_filter.uuid];

      if (restriction.hasOwnProperty(space_identifier)) {

        if (this.DEBUG == true) {
          console.log("[MatrixFilter] " + this.matrix_filter_type + " " + this.uuid + " received restriction update from " + restricting_matrix_filter.matrix_filter_type + " " + space_identifier + " , is_turned_on : " + is_turned_on);
        }

        this.restricted_by[restricting_matrix_filter.uuid][space_identifier] = is_turned_on;
      }

      else {
        throw new Error("[MatrixFilter] " + this.matrix_filter_type + " error in receive_restriction_update. Restrictive filter not found.");
      }
    }

    this.check_restrictions();
  }

  // if the visibilty is false, turn off all selected spaces, then call .update()
  check_restrictions() {

    let is_active = true;

    if (this.DEBUG == true) {
      console.log(this.restricted_by);
    }

    for (let matrix_filter_uuid in this.restricted_by) {

      let restriction = this.restricted_by[matrix_filter_uuid];

      // mf_1.space_1 OR mf_1.space_2 
      let all_spaces_of_this_filter_are_inactive = true;

      for (let space_identifier in restriction) {
        let space_is_active = restriction[space_identifier];
        if (space_is_active == true) {
          all_spaces_of_this_filter_are_inactive = false;
          break;
        }
      }

      if (all_spaces_of_this_filter_are_inactive == true){
        is_active = false;
      }

      if (is_active == false) {
        break;
      }
    }

    if (this.DEBUG == true) {
      console.log("[MatrixFilter] " + this.matrix_filter_type + " " + this.uuid + " worked restrictions, is_active:" + is_active);
    }

    if (is_active == true) {
      this.activate();
    }
    else {
      this.reset();
      this.deactivate();
    }
  }

  // hide or show the matrix_filter as a whole if its visibilty depends on a filter selection
  // use EVENTS, no dom manipulation here
  get_event_data() {

    const event_data = {
      detail: {
        "matrix_filter": {
          "uuid": this.uuid,
          "matrix_filter_type": this.matrix_filter_type,
          "status": this.status
        }
      }
    };

    return event_data;
  }

  // if all restrictions are met, this filter becomes usable by the user (STATUS_ACTIVE)
  activate() {

    this.status = STATUS_ACTIVE;

    const event_data = this.get_event_data();

    const activation_event = new CustomEvent("activate-matrix-filter", event_data);

    this.form_element.dispatchEvent(activation_event);

  }

  // if there is an active restriction, this filter becomes unusable by the user (STATUS_INACTIVE)
  deactivate() {

    if (this.status == STATUS_ACTIVE){

      this.status = STATUS_INACTIVE;

      // deactivate all MatrixFilterSpaces on MatrixFilter deactivation
      this.reset();

      const event_data = this.get_event_data();

      const deactivation_event = new CustomEvent("deactivate-matrix-filter", event_data);

      this.form_element.dispatchEvent(deactivation_event);
    }

  }

  // reset deactivates all MatrixFilterSpaces of this filter
  reset () {
    for (let space_identifier in this.matrix_filter_spaces){
      let matrix_filter_space = this.matrix_filter_spaces[space_identifier];
      matrix_filter_space.deactivate();
    }
  }

  // MATRIX ITEM MANAGEMENT


  // DOM BINDINGS
  // get the space_identifier from the input

  bind_input_element(input_element) {
    let space_identifier = this.get_space_identifier_from_input_element(input_element);

    // it is possible, that a space_identifier does not exist on any matrix item. In this case, the input_element stays unbound
    if (this.matrix_filter_spaces.hasOwnProperty(space_identifier)){
      let matrix_filter_space = this.matrix_filter_spaces[space_identifier];
      matrix_filter_space.bind_input_element(input_element);
    }
    else {
      console.warn("[MatrixFilter] " + this.matrix_filter_type + " space_identifier " + space_identifier + " not found in matrix_filter_spaces.");
      console.warn(this.matrix_filter_spaces);

      // send impossible event
      const event_data = {
        detail : {
          matrix_filter : {
            uuid : this.uuid
          },
          space_identifier: space_identifier
        }
      };
      const impossible_event = new CustomEvent("impossible", event_data);
      input_element.dispatchEvent(impossible_event);
    }
  }

  get_space_identifier_from_input_element(input_element) {
    let space_str = input_element.value;
    let space_identifier = this.get_space_identifier_from_str(space_str);
    return space_identifier;
  }

}

class StringBasedMatrixFilter extends MatrixFilter {

  // space is the parsed space from JSON.parse
  get_space_identifier_from_parsed(space) {
    if (this.DEBUG == true) {
      console.log("[StringBasedMatrixFilter] " + this.matrix_filter_type + "] stringifying parsed space: " + space);
    }

    // base64 encoded space
    let space_identifier = btoa(space);

    if (this.DEBUG == true) {
      console.log(space_identifier);
    }

    return space_identifier;
  }

  get_space_identifier_from_str(space_str) {
    return btoa(space_str);
  }

  parse_space_str(space_str) {
    return space_str;
  }

  get_space_from_space_identifier(space_identifier) {
    return atob(space_identifier);
  }

  get_space_str_from_space_identifier(space_identifier) {
    return atob(space_identifier);
  }

}

class ObjectBasedMatrixFilter extends MatrixFilter {

  get_space_identifier_from_parsed(space) {
    if (this.DEBUG == true) {
      console.log("[ObjectBasedMatrixFilter] " + this.matrix_filter_type + " stringifying parsed space: " + space);
    }

    // b64 encoded space identifier
    let space_identifier = btoa(JSON.stringify(space));

    if (this.DEBUG == true) {
      console.log(space_identifier);
    }

    return space_identifier;
  }

  // str, not b64encoded
  get_space_identifier_from_str(space_str) {

    if (this.DEBUG == true) {
      console.log("[ObjectBasedMatrixFilter] trying to parse: " + space_str);
    }

    // first parse to ensure the identifier is equal to from_parsed
    return btoa(JSON.stringify(JSON.parse(space_str)));
  }

  parse_space_str(space_str) {
    return JSON.parse(space_str);
  }

  // return thr parsed space
  get_space_from_space_identifier(space_identifier) {
    return JSON.parse(atob(space_identifier));
  }

  get_space_str_from_space_identifier(space_identifier) {
    return atob(space_identifier);
  }
}


class DescriptiveTextAndImagesFilter extends StringBasedMatrixFilter {

  constructor(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition) {

    super(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition);

    this.MatrixFilterSpaceClass = DescriptiveTextAndImagesSpace;
  }

}

class TextOnlyFilter extends StringBasedMatrixFilter {

  constructor(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition) {

    super(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition);

    this.MatrixFilterSpaceClass = TextOnlySpace;
  }

}


class ColorFilter extends ObjectBasedMatrixFilter {

  constructor(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition) {

    super(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition);

    this.MatrixFilterSpaceClass = ColorSpace;
  }

}

class NumberFilter extends ObjectBasedMatrixFilter {

  constructor(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition) {

    super(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition);

    this.MatrixFilterSpaceClass = NumberSpace;
  }

}

/*
* RangeFilter differs from other filters. There is only one range slider. You can think of multiple ranges being defined on one slider input element
*/
class RangeFilter extends ObjectBasedMatrixFilter {

  constructor(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition) {

    super(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition);

    this.MatrixFilterSpaceClass = RangeSpace;

    this.tolerance = 0;
    if ('tolerance' in definition && typeof definition.tolerance === 'number') {
      this.tolerance = definition.tolerance;
    }

    // all RangeFilteSpace instances share the same input_element which is bound to RangeFilter
    this.input_element = null;
  }

  // return the value as a string
  get_value_from_input_element (input_element) {

		let is_null = input_element.getAttribute("is-null", "true");

		if (is_null == "true" || is_null == true){
			return "";
		}
		else {
			return input_element.value;
		}
	}

  // only one range slider input element
  // how to handle MatrixFilterSpaces ? there are ranges defined on MatrixItems and on MatrixFilters(restrictions)
  bind_input_element(input_element) {

    var self = this;

    this.input_element = input_element;

    input_element.addEventListener("change", function(event){
        self.on_change(event);
    });

    input_element.addEventListener("clear", function(event){
      self.on_clear(event);
    });

  }

  /**
   * a specific value is selected
   * if the value is within a given range of a space, activate this space
   */
  on_change(event){

    let value_str = this.get_value_from_input_element(event.currentTarget);

    if (this.DEBUG == true){
      console.log("[RangeFilter] user selected " + value_str);
    }

    for (let space_identifier in this.matrix_filter_spaces){
      let matrix_filter_space = this.matrix_filter_spaces[space_identifier];
      let specific_range = this.get_space_from_space_identifier(space_identifier);
      
      let value_is_within_range = false;
      if (value_str.length > 0){
        let value = parseFloat(value_str);
        let upper_limit = specific_range[1];
        let lower_limit = specific_range[0];
        if (this.tolerance) {
          upper_limit = upper_limit + (upper_limit * ( this.tolerance / 100));
          lower_limit = lower_limit - (lower_limit * ( this.tolerance / 100));
        }
        if (value >= lower_limit && value <= upper_limit){
          value_is_within_range = true;
        }
      }

      if (value_is_within_range == true){

        if (this.DEBUG == true){
          console.log("[RangeFilter] " + value_str + " is within " + specific_range);
        }
        matrix_filter_space.activate();
      }
      else {
        if (this.DEBUG == true){
          console.log("[RangeFilter] " + value_str + " is not within " + specific_range);
        }
        matrix_filter_space.deactivate();
      }

    }
  }

  on_clear (event) {
    this.reset();
  }

  send_turned_off_event() {
    const turned_off_event = new Event("turnedOff");
    this.input_element.dispatchEvent(turned_off_event);
  }

  reset () {
    super.reset()

    this.input_element.value = "";
    this.send_turned_off_event();
  }

}

class TaxonFilter extends ObjectBasedMatrixFilter {

  constructor(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition) {

    super(form_element, uuid, matrix_filter_type, name, weight, allow_multiple_values, definition);

    this.MatrixFilterSpaceClass = TaxonFilterSpace;
  }

  // input_element.value is the b64 encoded space
  get_space_identifier_from_input_element(input_element) {
    let space_identifier = input_element.value;
    return space_identifier;
  }

}

const MATRIX_FILTER_CLASSES = {
  "DescriptiveTextAndImagesFilter": DescriptiveTextAndImagesFilter,
  "TextOnlyFilter": TextOnlyFilter,
  "NumberFilter": NumberFilter,
  "RangeFilter": RangeFilter,
  "TaxonFilter": TaxonFilter,
  "ColorFilter": ColorFilter
};


class MatrixItem {

  constructor(uuid, name, taxon, max_points, form_element) {

    this.DEBUG = true;

    this.uuid = uuid;
    this.name = name;
    this.taxon = taxon;

    // for firing events
    this.form_element = form_element;

    this.points = 0;
    this.points_percentage =0;
    this.max_points = max_points;

    // matrix_filter_spaces associated with this MatrixItem
    // { matrix_filter_uuid : { space_identifier : MatrixfilterSpace } }
    this.matrix_filter_spaces = {}; // constant

    this.active_matching_matrix_filter_spaces = {}; // non-RangeSpace
    this.active_mismatching_matrix_filter_spaces = {}; // non-RangeSpace

    this.active_matching_range_spaces = {};
    this.active_mismatching_range_spaces = {};

    this.status = STATUS_ACTIVE;

  }

  add_matrix_filter_space (matrix_filter_space) {
    let matrix_filter = matrix_filter_space.matrix_filter;

    if (!this.matrix_filter_spaces.hasOwnProperty(matrix_filter.uuid)){
      this.matrix_filter_spaces[matrix_filter.uuid] = {};
    }

    this.matrix_filter_spaces[matrix_filter.uuid][matrix_filter_space.space_identifier] = matrix_filter_space;

  }

  // ALL MODES
  calculate_points () {
    let points = 0;

    const registries = [this.active_matching_matrix_filter_spaces, this.active_matching_range_spaces];

    registries.forEach((registry) => {
      for (let matrix_filter_uuid in registry) {

        for (let space_identifier in registry[matrix_filter_uuid]){
          let weight = registry[matrix_filter_uuid][space_identifier];
          points = points + weight;
        }
      }
    });

		if (this.DEBUG == true){
			console.log("[MatrixItem] " + this.name + " total points: "  + points);
		}

		return points;
  }

  // if a matching matrix filter space is added points are calculated
  add_matching_matrix_filter_space(matrix_filter_space) {

    let matrix_filter = matrix_filter_space.matrix_filter;

    let registry = this.active_matching_matrix_filter_spaces;
    if (matrix_filter.matrix_filter_type === 'RangeFilter'){
      registry = this.active_matching_range_spaces;
    }

    if (this.DEBUG == true){
      console.log("[MatrixItem] " + this.name + " is adding space " + matrix_filter_space.space_identifier + " of " + matrix_filter.matrix_filter_type);
    }

    if (!registry.hasOwnProperty(matrix_filter.uuid)) {
      registry[matrix_filter.uuid] = {};
    }

    registry[matrix_filter.uuid][matrix_filter_space.space_identifier] = matrix_filter.weight;

    // it might be necessary to active the item if it was a range filter
    if (matrix_filter.matrix_filter_type === 'RangeFilter') {
      const range_filter_is_deactivating = this.range_filter_is_deactivating(matrix_filter);
      if (range_filter_is_deactivating === false && Object.keys(this.active_mismatching_matrix_filter_spaces).length === 0 && this.status === STATUS_INACTIVE) {
        this.activate();
      }
    }

    this.update();

  }

  range_filter_is_deactivating(range_filter) {

    if (range_filter.uuid in this.active_matching_range_spaces || Object.keys(this.active_mismatching_range_spaces).length === 0) {
      return false;
    } else if (range_filter.uuid in this.active_mismatching_range_spaces) {
      return true;
    }

    return false;

  }

  remove_matching_matrix_filter_space(matrix_filter_space) {

    let matrix_filter = matrix_filter_space.matrix_filter;

    let registry = this.active_matching_matrix_filter_spaces;
    if (matrix_filter.matrix_filter_type === 'RangeFilter'){
      registry = this.active_matching_range_spaces;
    }

    if (this.DEBUG == true){
      console.log("[MatrixItem] " + this.name + " is removing space " + matrix_filter_space.space_identifier + " of " + matrix_filter.matrix_filter_type);
    }

    if (registry.hasOwnProperty(matrix_filter.uuid)) {
      let matrix_item_space = registry[matrix_filter.uuid];

      if (matrix_item_space.hasOwnProperty(matrix_filter_space.space_identifier)){
        delete registry[matrix_filter.uuid][matrix_filter_space.space_identifier];
      }

      if (Object.keys(registry[matrix_filter.uuid]).length == 0){
        delete registry[matrix_filter.uuid];
      }
    }

    if (matrix_filter.matrix_filter_type === 'RangeFilter') {
      if (Object.keys(this.active_mismatching_range_spaces).length > 0 && !(matrix_filter.uuid in this.active_matching_range_spaces) && this.status === STATUS_ACTIVE) {
        //this.deactivate();
      }
    }

    this.update();
  }

  update() {

    if (this.DEBUG == true){
      console.log("[MatrixItem] " + this.name + " updating points");
    }

    this.points = this.calculate_points();
    this.points_percentage = this.points / this.max_points;
    this.send_matrix_item_update_event();

    if (this.points_percentage == 1){
      this.send_100_percent_event();
    }
  }

  // events
  get_event_data () {
    const event_data = {
			detail : {
				"matrix_item" : {
					"uuid" : this.uuid,
					"points" : this.points,
          "max_points" : this.max_points,
          "points_percentage" : this.points_percentage
				}
			}
		};

    return event_data;

  }

	send_matrix_item_update_event () {

		if (this.DEBUG == true){
			console.log("[MatrixItem] " + this.name +  " sending update-matrix-item event");
		}

		const event_data = this.get_event_data();

		const points_update_event = new CustomEvent("update-matrix-item", event_data);

		this.form_element.dispatchEvent(points_update_event);
	}

  // send an event when a matrix item reaches 100%
  send_100_percent_event () {
    if (this.DEBUG == true){
			console.log("[MatrixItem] " + this.name +  " sending matrix-item-100-percent");
		}

		const event_data = this.get_event_data();

		const matrix_item_100_percent_event = new CustomEvent("matrix-item-100-percent", event_data);

		this.form_element.dispatchEvent(matrix_item_100_percent_event);
  }

  reset() {

    this.points = 0;

    this.active_matching_matrix_filter_spaces = {};
    this.active_mismatching_matrix_filter_spaces = {};
    this.active_matching_range_spaces = {};
    this.active_mismatching_range_spaces = {};
    
    this.activate();
  }

  //
  // STRICT MODE ONLY
  //
  // a mismatch will deactivate a matrix item. RangeSpace mismatches require a special treatment
  add_mismatching_matrix_filter_space(matrix_filter_space) {

    let matrix_filter = matrix_filter_space.matrix_filter;

    let registry = this.active_mismatching_matrix_filter_spaces;
    if (matrix_filter.matrix_filter_type === 'RangeFilter') {
      registry = this.active_mismatching_range_spaces;
    }

    if (this.DEBUG == true){
      console.log("[MatrixItem] " + this.name + " received an added mismatch from " + matrix_filter.matrix_filter_type + " : " + matrix_filter_space.space_identifier);
    }
    
    let matrix_item_has_initial_mismatches = Object.keys(this.active_mismatching_matrix_filter_spaces).length == 0 ? false : true;

    if(!registry.hasOwnProperty(matrix_filter.uuid)){
      registry[matrix_filter.uuid] = {}; 
    }

    registry[matrix_filter.uuid][matrix_filter_space.space_identifier] = matrix_filter_space;

    let deactivate_self = true;
    // evaluate RangeFilter situation only if there are no other mismatches
    if (Object.keys(this.active_mismatching_matrix_filter_spaces).length == 0) {
      // check if the range filter situation should trigger a deactivation
      // range filters can overlap and are item specific
      // deactivate if the RangeFilter (uuid) has only mismatches and no matches 
      if (matrix_filter.uuid in this.active_matching_range_spaces) {
        deactivate_self = false;
      }
    }

    //if (matrix_item_has_initial_mismatches == false){
    if (deactivate_self === true && this.status === STATUS_ACTIVE) {
      this.deactivate();
    }
    
  }

  remove_mismatching_matrix_filter_space(matrix_filter_space) {

    let matrix_filter = matrix_filter_space.matrix_filter;

    let registry = this.active_mismatching_matrix_filter_spaces;
    if (matrix_filter.matrix_filter_type === 'RangeFilter') {
      registry = this.active_mismatching_range_spaces;
    }

    if (this.DEBUG == true){
      console.log("[MatrixItem] " + this.name + " received a removed mismatch from " + matrix_filter.matrix_filter_type + " : " + matrix_filter_space.space_identifier);
    }

    let matrix_item_is_initially_visible = Object.keys(this.active_mismatching_matrix_filter_spaces).length == 0 ? false : true;

    if (registry.hasOwnProperty(matrix_filter.uuid)) {
      let matrix_item_space = registry[matrix_filter.uuid];

      if (matrix_item_space.hasOwnProperty(matrix_filter_space.space_identifier)){
        delete registry[matrix_filter.uuid][matrix_filter_space.space_identifier];
      }

      if (Object.keys(registry[matrix_filter.uuid]).length == 0){
        delete registry[matrix_filter.uuid];
      }
    }

    let activate_self = false;
    if (Object.keys(this.active_mismatching_matrix_filter_spaces).length === 0) {
      activate_self = true;

      const range_filter_is_deactivating = this.range_filter_is_deactivating(matrix_filter);

      if (range_filter_is_deactivating === true) {
        activate_self = false;
      }      
    }

    // if (Object.keys(this.active_mismatching_matrix_filter_spaces).length == 0 && matrix_item_is_initially_visible == true) {
    if (activate_self === true && this.status == STATUS_INACTIVE) {
      this.activate();
    }
  }

  // the matrix_item had mismatches and the last mismatch has been removed. the matrix item transforms from inactive to active
  activate() {

    this.status = STATUS_ACTIVE;

    // when a MatrixItem gets activated, MatrixFilterSpaces can become possible
    this.signal_matrix_filter_spaces(true);

		if (this.DEBUG == true){
			console.log("[MatrixItem] " + this.name +  " sending activate-matrix-item event");
		}

		const event_data = this.get_event_data();

		const activation_event = new CustomEvent("activate-matrix-item", event_data);

		this.form_element.dispatchEvent(activation_event);
  }

  // the matrix_item had no mismatches and the the first mismatch has been added. the matrix item transforms from active to inactive
  deactivate() {

    this.status = STATUS_INACTIVE;

    // when a MatrixItem gets deactivated, MatrixFilterSpaces can become impossible
    this.signal_matrix_filter_spaces(false);

    if (this.DEBUG == true){
			console.log("[MatrixItem] " + this.name +  " sending deactivate-matrix-item event");
		}

		const event_data = this.get_event_data();

		const deactivation_event = new CustomEvent("deactivate-matrix-item", event_data);

		this.form_element.dispatchEvent(deactivation_event);
  }

  signal_matrix_filter_spaces(matrix_item_is_active) {
    for (let matrix_filter_uuid in this.matrix_filter_spaces){
      let item_spaces = this.matrix_filter_spaces[matrix_filter_uuid];
      for (let space_identifier in item_spaces){
        let matrix_filter_space = item_spaces[space_identifier];

        if (matrix_item_is_active == true){
          matrix_filter_space.activate_matching_matrix_item(this);
        }
        else {
          matrix_filter_space.deactivate_matching_matrix_item(this);
        }
      }
    }
  }
}


class IdentificationMatrix {

  constructor(filter_form_id, get_matrix_filters_and_items, options) {

    this.DEBUG = true;

    var options = options || {};

    if (options.hasOwnProperty("mode")) {
      IDENTIFICATION_MODE = MODE_STRICT//options["mode"];
    }

    this.get_matrix_filters_and_items = get_matrix_filters_and_items;

    this.data = [];

    this.form_element = document.getElementById(filter_form_id);

    this.init();

  }

  init(onFinished) {

    if (this.DEBUG == true){
      console.log("[IdentificationMatrix] is initializing.");
    }

    var self = this;

    this.get_matrix_filters_and_items(function (matrix_filters_and_items_json) {

      if (self.DEBUG == true) {
        console.log(matrix_filters_and_items_json);
      }


      self.data = matrix_filters_and_items_json;

      // instantiate filters
      for (let matrix_filter_uuid in matrix_filters_and_items_json["matrix_filters"]) {
        let matrix_filter_data = matrix_filters_and_items_json["matrix_filters"][matrix_filter_uuid];
        let MatrixFilterClass = MATRIX_FILTER_CLASSES[matrix_filter_data.type];

        let matrix_filter = new MatrixFilterClass(self.form_element, matrix_filter_uuid, matrix_filter_data.type, matrix_filter_data.name, matrix_filter_data.weight, matrix_filter_data.allow_multiple_values, matrix_filter_data.definition);

        MATRIX_FILTERS[matrix_filter_uuid] = matrix_filter;
      }

      let items = matrix_filters_and_items_json["items"];

      // instantiate matrix items (tree nodes)
      for (let i = 0; i < items.length; i++) {
        let item = items[i];
        let matrix_item = new MatrixItem(item.uuid, item.name, item.taxon, item.max_points, self.form_element);
        MATRIX_ITEMS[matrix_item.uuid] = matrix_item;

        // add MatrixFilterSpaces to matrix filters and to matrix items
        for (let matrix_filter_uuid in item["space"]) {

          let spaces = item["space"][matrix_filter_uuid];
          let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];

          for (let s = 0; s < spaces.length; s++) {
            let parsed_space = spaces[s];
            let matrix_filter_space = matrix_filter.add_matrix_filter_space(parsed_space);
            matrix_item.add_matrix_filter_space(matrix_filter_space);

            // bind evaluation to MatrixFilterSpace.post_change
            matrix_filter_space.post_activation = function(){
              self.evaluate_identification();
            }
          }
        }

      }

      self.bind_dom_elements_to_matrix_filters();

      // apply restrictions - AFTER instantiating MatrixItems. MatrixItems fill
      for (let matrix_filter_uuid in matrix_filters_and_items_json["matrix_filters"]) {
        let data = matrix_filters_and_items_json["matrix_filters"][matrix_filter_uuid];
        let restrictions = data["restrictions"] || {};

        let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];
        matrix_filter.apply_restrictions(restrictions);
      }

      // add matching and mismatching matrix_items to MatrixFilterSpace instances
      for (let matrix_filter_uuid in MATRIX_FILTERS){
        let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];

        for (let space_identifier in matrix_filter.matrix_filter_spaces){
          let matrix_filter_space = matrix_filter.matrix_filter_spaces[space_identifier];

          for (let matrix_item_uuid in MATRIX_ITEMS){
            let matrix_item = MATRIX_ITEMS[matrix_item_uuid];
            
            let space_contained_in_item = false;

            if (matrix_item.matrix_filter_spaces.hasOwnProperty(matrix_filter.uuid)){
              if (matrix_item.matrix_filter_spaces[matrix_filter.uuid].hasOwnProperty(space_identifier)){
                space_contained_in_item = true;
              }
            }

            if (space_contained_in_item == true){
              matrix_filter_space.register_matching_matrix_item(matrix_item);
            }
            else {
              matrix_filter_space.register_mismatching_matrix_item(matrix_item);
            }

          }

        }
      }

      // if the user comes from using the back button, activate all checked MatrixFilterSpaces
      // this has to be done at the end ofinitialization, so the MatrixFilterSpaces can signal/notify MatrixItems, which leads to the MatrixItems updateing their points
      for (let matrix_filter_uuid in MATRIX_FILTERS){
        let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];

        for (let space_identifier in matrix_filter.matrix_filter_spaces){
          let matrix_filter_space = matrix_filter.matrix_filter_spaces[space_identifier];

          if (matrix_filter_space.input_element != null && matrix_filter_space.input_element.checked == true){
            // the input might be checked on initialization, e.g. when the page is rendered from history using the back button
    
            matrix_filter_space.activate();
    
          }
        }
      }

      if (typeof onFinished == "function"){
        onFinished();
      }

    });
  }

  bind_dom_elements_to_matrix_filters() {

    // horizontal sliders
    // checkboxes are multiselect, ranges are single select
    var inputs = this.form_element.querySelectorAll("input[type=radio], input[type=checkbox], input[type=range]");

    for (let i = 0; i < inputs.length; i++) {
      let input = inputs[i];

      let matrix_filter_uuid = input.name;
      let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];
      matrix_filter.bind_input_element(input);
    }

  }

  deactivate_all_matrix_filters() {
    for (let matrix_filter_uuid in MATRIX_FILTERS){
      let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];
      matrix_filter.reset();
    }
  }

  set_mode (mode){
    console.log("[IdentificationMatrix] changing mode to " + mode);
    IDENTIFICATION_MODE = MODE_STRICT//mode;
    this.reset();
  }

  reset() {
    this.deactivate_all_matrix_filters();
    this.reset_all_matrix_items();
    this.form_element.reset();
    this.send_reset_event();
  }

  reset_all_matrix_filters() {
    for (let matrix_filter_uuid in MATRIX_FILTERS) {
      let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];
      matrix_filter.reset();
    }
  }

  reset_all_matrix_items() {
    for (let matrix_item_uuid in MATRIX_ITEMS) {
      let matrix_item = MATRIX_ITEMS[matrix_item_uuid];
      matrix_item.reset();
    }
  }

  send_reset_event() {
    const reset_event = new Event("reset");

    this.form_element.dispatchEvent(reset_event);
  }

  // reset all items, reload from data
  reload () {

    if (this.DEBUG == true){
      console.log("[IdentificationMatrix] reloading data");
    }

    var self = this;

    MATRIX_FILTERS = {};
    MATRIX_ITEMS = {};
    this.init(function(){

      // make all MatrixFilterSpaces possible
      for (let matrix_filter_uuid in MATRIX_FILTERS){
        let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];
        for (let space_identifier in matrix_filter.matrix_filter_spaces){
          let matrix_filter_space = matrix_filter.matrix_filter_spaces[space_identifier];
          matrix_filter_space.send_possible_event();
        }
      }

      self.reset();
    });

  }

  destroy () {
    MATRIX_FILTERS = {};
    MATRIX_ITEMS = {};
  }

  // events when identification has finished
  evaluate_identification () {

    if (this.DEBUG == true){
      console.log("[IdentificationMatrix] evalutating identification");
    }

    let all_filters_have_selection = true;

    for (let matrix_filter_uuid in MATRIX_FILTERS){
      let matrix_filter = MATRIX_FILTERS[matrix_filter_uuid];

      let matrix_filter_has_selection = false;

      if (matrix_filter.status == STATUS_INACTIVE){
        matrix_filter_has_selection = true;
        continue;
      }
      else {

        // this does not work with ranges. the user can select a value that is not within any given range.
        // simply check if the range slider has been set
        
        if (matrix_filter.matrix_filter_type == "RangeFilter"){
          if (matrix_filter.input_element.value.length != 0){
            matrix_filter_has_selection = true;
          }
        }
        else {
          for (let space_identifier in matrix_filter.matrix_filter_spaces){
            let matrix_filter_space = matrix_filter.matrix_filter_spaces[space_identifier];

            if (matrix_filter_space.status == STATUS_ACTIVE){
              matrix_filter_has_selection = true;
              break;
            }
          }
        }
      }

      if (matrix_filter_has_selection == false){
        all_filters_have_selection = false;
        if (this.DEBUG == true){
          console.log("[IdentificationMatrix] " + matrix_filter.name + " has no selection yet.");
        }
        break;
      }
    }

    if (all_filters_have_selection == true){
      this.send_identification_finished_event();
    }

  }
  // notify the user to select the next identification step
  send_identification_finished_event (){
    // strict mode always ends with 100%
    // send event when every VISIBLE matrix filter has one selection
    // use MatrixFilter.active_matrix_filter_spaces{}
    if (this.DEBUG == true){
      console.log("[IdentificationMatrix] sending identification-finished event");
    }
    const event_data = {};
    const finished_event = new CustomEvent("identification-finished", event_data);
    
    this.form_element.dispatchEvent(finished_event);
  }

  get_nature_guide_item (matrix_item_uuid){
    var matched_nature_guide_item = null;

    for (let i=0; i<this.data.items.length; i++){
      let nature_guide_item = this.data.items[i];
      if (nature_guide_item.uuid == matrix_item_uuid){
        matched_nature_guide_item = nature_guide_item;
        break;
      }
    }

    return matched_nature_guide_item;
  }

}
