"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
class EncryptionKeyBase extends cdk_1.Construct {
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias) {
        return new alias_1.EncryptionKeyAlias(this, 'Alias', { alias, key: this });
    }
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        if (!this.policy) {
            if (allowNoOp) {
                return;
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(this.node.resolve(this.keyArn))}`);
        }
        this.policy.addStatement(statement);
    }
}
exports.EncryptionKeyBase = EncryptionKeyBase;
/**
 * Defines a KMS key.
 */
class EncryptionKey extends EncryptionKeyBase {
    /**
     * Defines an imported encryption key.
     *
     * `ref` can be obtained either via a call to `key.export()` or using
     * literals.
     *
     * For example:
     *
     *   const keyAttr = key.export();
     *   const keyRef1 = EncryptionKey.import(this, 'MyImportedKey1', keyAttr);
     *   const keyRef2 = EncryptionKey.import(this, 'MyImportedKey2', {
     *     keyArn: new KeyArn('arn:aws:kms:...')
     *   });
     *
     * @param scope The parent construct.
     * @param id The name of the construct.
     * @param props The key reference.
     */
    static import(scope, id, props) {
        return new ImportedEncryptionKey(scope, id, props);
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.policy) {
            this.policy = props.policy;
        }
        else {
            this.policy = new aws_iam_1.PolicyDocument();
            this.allowAccountToAdmin();
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
        });
        this.keyArn = resource.keyArn;
        resource.options.deletionPolicy = props.retain === false
            ? cdk_1.DeletionPolicy.Delete
            : cdk_1.DeletionPolicy.Retain;
    }
    /**
     * Exports this key from the current stack.
     * @returns a key ref which can be used in a call to `EncryptionKey.import(ref)`.
     */
    export() {
        return {
            keyArn: new cdk_1.CfnOutput(this, 'KeyArn', { value: this.keyArn }).makeImportValue().toString()
        };
    }
    /**
     * Let users from this account admin this key.
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    allowAccountToAdmin() {
        const actions = [
            "kms:Create*",
            "kms:Describe*",
            "kms:Enable*",
            "kms:List*",
            "kms:Put*",
            "kms:Update*",
            "kms:Revoke*",
            "kms:Disable*",
            "kms:Get*",
            "kms:Delete*",
            "kms:ScheduleKeyDeletion",
            "kms:CancelKeyDeletion"
        ];
        this.addToResourcePolicy(new aws_iam_1.PolicyStatement()
            .addAllResources()
            .addActions(...actions)
            .addAccountRootPrincipal());
    }
}
exports.EncryptionKey = EncryptionKey;
class ImportedEncryptionKey extends EncryptionKeyBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.policy = undefined; // no policy associated with an imported key
        this.keyArn = props.keyArn;
    }
    export() {
        return this.props;
    }
}
//# sourceMappingURL=data:application/json;base64,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