"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SizeRoundingBehavior = exports.Size = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const token_1 = require("./token");
/**
 * Represents the amount of digital storage.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 *
 * @stability stable
 */
class Size {
    constructor(amount, unit) {
        if (!token_1.Token.isUnresolved(amount) && amount < 0) {
            throw new Error(`Storage amounts cannot be negative. Received: ${amount}`);
        }
        this.amount = amount;
        this.unit = unit;
    }
    /**
     * Create a Storage representing an amount kibibytes.
     *
     * 1 KiB = 1024 bytes
     *
     * @param amount the amount of kibibytes to be represented.
     * @returns a new `Size` instance
     * @stability stable
     */
    static kibibytes(amount) {
        return new Size(amount, StorageUnit.Kibibytes);
    }
    /**
     * Create a Storage representing an amount mebibytes.
     *
     * 1 MiB = 1024 KiB
     *
     * @param amount the amount of mebibytes to be represented.
     * @returns a new `Size` instance
     * @stability stable
     */
    static mebibytes(amount) {
        return new Size(amount, StorageUnit.Mebibytes);
    }
    /**
     * Create a Storage representing an amount gibibytes.
     *
     * 1 GiB = 1024 MiB
     *
     * @param amount the amount of gibibytes to be represented.
     * @returns a new `Size` instance
     * @stability stable
     */
    static gibibytes(amount) {
        return new Size(amount, StorageUnit.Gibibytes);
    }
    /**
     * Create a Storage representing an amount tebibytes.
     *
     * 1 TiB = 1024 GiB
     *
     * @param amount the amount of tebibytes to be represented.
     * @returns a new `Size` instance
     * @stability stable
     */
    static tebibytes(amount) {
        return new Size(amount, StorageUnit.Tebibytes);
    }
    /**
     * (deprecated) Create a Storage representing an amount pebibytes.
     *
     * 1 PiB = 1024 TiB
     *
     * @deprecated use `pebibytes` instead
     */
    static pebibyte(amount) {
        return Size.pebibytes(amount);
    }
    /**
     * Create a Storage representing an amount pebibytes.
     *
     * 1 PiB = 1024 TiB
     *
     * @param amount the amount of pebibytes to be represented.
     * @returns a new `Size` instance
     * @stability stable
     */
    static pebibytes(amount) {
        return new Size(amount, StorageUnit.Pebibytes);
    }
    /**
     * Return this storage as a total number of kibibytes.
     *
     * @param opts the conversion options.
     * @returns the quantity of bytes expressed in kibibytes
     * @stability stable
     */
    toKibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Kibibytes, opts);
    }
    /**
     * Return this storage as a total number of mebibytes.
     *
     * @param opts the conversion options.
     * @returns the quantity of bytes expressed in mebibytes
     * @stability stable
     */
    toMebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Mebibytes, opts);
    }
    /**
     * Return this storage as a total number of gibibytes.
     *
     * @param opts the conversion options.
     * @returns the quantity of bytes expressed in gibibytes
     * @stability stable
     */
    toGibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Gibibytes, opts);
    }
    /**
     * Return this storage as a total number of tebibytes.
     *
     * @param opts the conversion options.
     * @returns the quantity of bytes expressed in tebibytes
     * @stability stable
     */
    toTebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Tebibytes, opts);
    }
    /**
     * Return this storage as a total number of pebibytes.
     *
     * @param opts the conversion options.
     * @returns the quantity of bytes expressed in pebibytes
     * @stability stable
     */
    toPebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Pebibytes, opts);
    }
}
exports.Size = Size;
_a = JSII_RTTI_SYMBOL_1;
Size[_a] = { fqn: "@aws-cdk/core.Size", version: "1.110.1" };
/**
 * Rounding behaviour when converting between units of `Size`.
 *
 * @stability stable
 */
var SizeRoundingBehavior;
(function (SizeRoundingBehavior) {
    SizeRoundingBehavior[SizeRoundingBehavior["FAIL"] = 0] = "FAIL";
    SizeRoundingBehavior[SizeRoundingBehavior["FLOOR"] = 1] = "FLOOR";
    SizeRoundingBehavior[SizeRoundingBehavior["NONE"] = 2] = "NONE";
})(SizeRoundingBehavior = exports.SizeRoundingBehavior || (exports.SizeRoundingBehavior = {}));
class StorageUnit {
    constructor(label, inKibiBytes) {
        this.label = label;
        this.inKibiBytes = inKibiBytes;
        // MAX_SAFE_INTEGER is 2^53, so by representing storage in kibibytes,
        // the highest storage we can represent is 8 exbibytes.
    }
    toString() {
        return this.label;
    }
}
StorageUnit.Kibibytes = new StorageUnit('kibibytes', 1);
StorageUnit.Mebibytes = new StorageUnit('mebibytes', 1024);
StorageUnit.Gibibytes = new StorageUnit('gibibytes', 1024 * 1024);
StorageUnit.Tebibytes = new StorageUnit('tebibytes', 1024 * 1024 * 1024);
StorageUnit.Pebibytes = new StorageUnit('pebibytes', 1024 * 1024 * 1024 * 1024);
function convert(amount, fromUnit, toUnit, options = {}) {
    var _b;
    const rounding = (_b = options.rounding) !== null && _b !== void 0 ? _b : SizeRoundingBehavior.FAIL;
    if (fromUnit.inKibiBytes === toUnit.inKibiBytes) {
        return amount;
    }
    if (token_1.Token.isUnresolved(amount)) {
        throw new Error(`Unable to perform time unit conversion on un-resolved token ${amount}.`);
    }
    const multiplier = fromUnit.inKibiBytes / toUnit.inKibiBytes;
    const value = amount * multiplier;
    switch (rounding) {
        case SizeRoundingBehavior.NONE:
            return value;
        case SizeRoundingBehavior.FLOOR:
            return Math.floor(value);
        default:
        case SizeRoundingBehavior.FAIL:
            if (!Number.isInteger(value)) {
                throw new Error(`'${amount} ${fromUnit}' cannot be converted into a whole number of ${toUnit}.`);
            }
            return value;
    }
}
//# sourceMappingURL=data:application/json;base64,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