"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const bundlers_1 = require("./bundlers");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        // Find project root
        const projectRoot = (_d = (_c = (_b = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`)) !== null && _b !== void 0 ? _b : util_1.findUp(bundlers_1.LockFile.YARN)) !== null && _c !== void 0 ? _c : util_1.findUp(bundlers_1.LockFile.NPM)) !== null && _d !== void 0 ? _d : util_1.findUp('package.json');
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const relativeEntryPath = path.relative(projectRoot, path.resolve(options.entry));
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_e = options.externalModules) !== null && _e !== void 0 ? _e : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_f = options.nodeModules) !== null && _f !== void 0 ? _f : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        let installer = bundlers_1.Installer.NPM;
        let lockFile;
        if (dependencies) {
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.YARN))) {
                installer = bundlers_1.Installer.YARN;
                lockFile = bundlers_1.LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.NPM))) {
                lockFile = bundlers_1.LockFile.NPM;
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_g = options.sourceMaps) !== null && _g !== void 0 ? _g : false,
                    minify: (_h = options.minify) !== null && _h !== void 0 ? _h : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Local
        let localBundler;
        if (!options.forceDockerBundling) {
            localBundler = new bundlers_1.LocalBundler({
                projectRoot,
                relativeEntryPath,
                cacheDir: options.cacheDir,
                environment: options.parcelEnvironment,
                dependencies,
                installer,
                lockFile,
            });
        }
        // Docker
        const dockerBundler = new bundlers_1.DockerBundler({
            runtime: options.runtime,
            relativeEntryPath,
            cacheDir: options.cacheDir,
            environment: options.parcelEnvironment,
            bundlingDockerImage: options.bundlingDockerImage,
            buildImage: !bundlers_1.LocalBundler.runsLocally(projectRoot) || options.forceDockerBundling,
            buildArgs: options.buildArgs,
            parcelVersion: options.parcelVersion,
            dependencies,
            installer,
            lockFile,
        });
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                local: localBundler,
                ...dockerBundler.bundlingOptions,
            },
        });
    }
}
exports.Bundling = Bundling;
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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