"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VERSION_LOCKED = exports.trimFromStart = exports.calculateFunctionHash = void 0;
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const cx_api_1 = require("@aws-cdk/cx-api");
const function_1 = require("./function");
function calculateFunctionHash(fn) {
    const stack = core_1.Stack.of(fn);
    const functionResource = fn.node.defaultChild;
    // render the cloudformation resource from this function
    const config = stack.resolve(functionResource._toCloudFormation());
    // config is of the shape: { Resources: { LogicalId: { Type: 'Function', Properties: { ... } }}}
    const resources = config.Resources;
    const resourceKeys = Object.keys(resources);
    if (resourceKeys.length !== 1) {
        throw new Error(`Expected one rendered CloudFormation resource but found ${resourceKeys.length}`);
    }
    const logicalId = resourceKeys[0];
    const properties = resources[logicalId].Properties;
    let stringifiedConfig;
    if (core_1.FeatureFlags.of(fn).isEnabled(cx_api_1.LAMBDA_RECOGNIZE_VERSION_PROPS)) {
        const updatedProps = sortProperties(filterUsefulKeys(properties));
        stringifiedConfig = JSON.stringify(updatedProps);
    }
    else {
        const sorted = sortProperties(properties);
        config.Resources[logicalId].Properties = sorted;
        stringifiedConfig = JSON.stringify(config);
    }
    if (core_1.FeatureFlags.of(fn).isEnabled(cx_api_1.LAMBDA_RECOGNIZE_LAYER_VERSION)) {
        stringifiedConfig = stringifiedConfig + calculateLayersHash(fn._layers);
    }
    const hash = crypto.createHash('md5');
    hash.update(stringifiedConfig);
    return hash.digest('hex');
}
exports.calculateFunctionHash = calculateFunctionHash;
function trimFromStart(s, maxLength) {
    const desiredLength = Math.min(maxLength, s.length);
    const newStart = s.length - desiredLength;
    return s.substring(newStart);
}
exports.trimFromStart = trimFromStart;
/*
 * The list of properties found in CfnFunction (or AWS::Lambda::Function).
 * They are classified as "locked" to a Function Version or not.
 * When a property is locked, any change to that property will not take effect on previously created Versions.
 * Instead, a new Version must be generated for the change to take effect.
 * Similarly, if a property that's not locked to a Version is modified, a new Version
 * must not be generated.
 *
 * Adding a new property to this list - If the property is part of the UpdateFunctionConfiguration
 * API or UpdateFunctionCode API, then it must be classified as true, otherwise false.
 * See https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html and
 * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html
 */
exports.VERSION_LOCKED = {
    // locked to the version
    Architectures: true,
    Code: true,
    DeadLetterConfig: true,
    Description: true,
    Environment: true,
    EphemeralStorage: true,
    FileSystemConfigs: true,
    FunctionName: true,
    Handler: true,
    ImageConfig: true,
    KmsKeyArn: true,
    Layers: true,
    MemorySize: true,
    PackageType: true,
    Role: true,
    Runtime: true,
    Timeout: true,
    TracingConfig: true,
    VpcConfig: true,
    // not locked to the version
    CodeSigningConfigArn: false,
    ReservedConcurrentExecutions: false,
    Tags: false,
};
function filterUsefulKeys(properties) {
    const versionProps = { ...exports.VERSION_LOCKED, ...function_1.Function._VER_PROPS };
    const unclassified = Object.entries(properties)
        .filter(([k, v]) => v != null && !Object.keys(versionProps).includes(k))
        .map(([k, _]) => k);
    if (unclassified.length > 0) {
        throw new Error(`The following properties are not recognized as version properties: [${unclassified}].`
            + ' See the README of the aws-lambda module to learn more about this and to fix it.');
    }
    const notLocked = Object.entries(versionProps).filter(([_, v]) => !v).map(([k, _]) => k);
    notLocked.forEach(p => delete properties[p]);
    const ret = {};
    Object.entries(properties).filter(([k, _]) => versionProps[k]).forEach(([k, v]) => ret[k] = v);
    return ret;
}
function sortProperties(properties) {
    const ret = {};
    // We take all required properties in the order that they were historically,
    // to make sure the hash we calculate is stable.
    // There cannot be more required properties added in the future,
    // as that would be a backwards-incompatible change.
    const requiredProperties = ['Code', 'Handler', 'Role', 'Runtime'];
    for (const requiredProperty of requiredProperties) {
        ret[requiredProperty] = properties[requiredProperty];
    }
    // then, add all of the non-required properties,
    // in the original order
    for (const property of Object.keys(properties)) {
        if (requiredProperties.indexOf(property) === -1) {
            ret[property] = properties[property];
        }
    }
    return ret;
}
function calculateLayersHash(layers) {
    const layerConfig = {};
    for (const layer of layers) {
        const stack = core_1.Stack.of(layer);
        const layerResource = layer.node.defaultChild;
        // if there is no layer resource, then the layer was imported
        // and we will include the layer arn and runtimes in the hash
        if (layerResource === undefined) {
            layerConfig[layer.layerVersionArn] = layer.compatibleRuntimes;
            continue;
        }
        const config = stack.resolve(layerResource._toCloudFormation());
        const resources = config.Resources;
        const resourceKeys = Object.keys(resources);
        if (resourceKeys.length !== 1) {
            throw new Error(`Expected one rendered CloudFormation resource but found ${resourceKeys.length}`);
        }
        const logicalId = resourceKeys[0];
        const properties = resources[logicalId].Properties;
        // all properties require replacement, so they are all version locked.
        layerConfig[layer.node.id] = properties;
    }
    const hash = crypto.createHash('md5');
    hash.update(JSON.stringify(layerConfig));
    return hash.digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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