# Sometimes, 2 payloads are sent in a single message.
# By adding the " at the end we more or less guarantee that
# the delimiter is not withing a quoted string
import asyncio
import re
from email.utils import parsedate_to_datetime
from json import JSONDecodeError
from time import time
from typing import Any, Optional

from workflowai.core.domain.errors import BaseError, WorkflowAIError

delimiter = re.compile(r'\}\n\ndata: \{"')


def split_chunks(chunk: bytes):
    start = 0
    chunk_str = chunk.removeprefix(b"data: ").removesuffix(b"\n\n").decode()
    for match in delimiter.finditer(chunk_str):
        yield chunk_str[start : match.start() + 1]
        start = match.end() - 2
    yield chunk_str[start:]


def retry_after_to_delay_seconds(retry_after: Any) -> Optional[float]:
    if retry_after is None:
        return None

    try:
        return float(retry_after)
    except ValueError:
        pass
    try:
        retry_after_date = parsedate_to_datetime(retry_after)
        current_time = time()
        return retry_after_date.timestamp() - current_time
    except (TypeError, ValueError, OverflowError):
        return None


# Returns two functions:
# - _should_retry: returns True if we should retry
# - _wait_for_exception: waits after an exception only if we should retry, otherwise raises
# This is a bit convoluted and would be better in a function wrapper, but since we are dealing
# with both Awaitable and AsyncGenerator, a wrapper would just be too complex
def build_retryable_wait(
    max_retry_delay: float = 60,
    max_retry_count: float = 1,
):
    now = time()
    retry_count = 0

    def _leftover_delay():
        # Time remaining before we hit the max retry delay
        return max_retry_delay - (time() - now)

    def _should_retry():
        return retry_count < max_retry_count and _leftover_delay() >= 0

    async def _wait_for_exception(e: WorkflowAIError):
        if not e.response:
            raise e

        nonlocal retry_count
        retry_after = retry_after_to_delay_seconds(e.response.headers.get("Retry-After"))
        leftover_delay = _leftover_delay()
        if not retry_after or leftover_delay < 0 or retry_count >= max_retry_count:
            # Convert error to WorkflowAIError
            try:
                response_json = e.response.json()
                r_err = response_json.get("error", {})
                error_message = response_json.get("detail", {}) or r_err.get("message", "Unknown Error")
                details = r_err.get("details", {})
                error_code = r_err.get("code", "unknown_error")
                status_code = r_err.get("status_code", e.response.status_code)
            except JSONDecodeError:
                error_message = "Unknown error"
                details = {"raw": e.response.content.decode()}
                error_code = "unknown_error"
                status_code = e.response.status_code

            raise WorkflowAIError(
                error=BaseError(
                    message=error_message,
                    details=details,
                    status_code=status_code,
                    code=error_code,
                ),
                response=e.response,
            ) from None

        await asyncio.sleep(retry_after)
        retry_count += 1

    return _should_retry, _wait_for_exception
