
use rand::rngs::OsRng;
use rand::RngCore;
use std::str;
use hex;

use pyo3::prelude::*;

const S_BOX: [[u8; 16]; 16] = [
    [0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76],
    [0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0],
    [0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15],
    [0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75],
    [0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84],
    [0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf],
    [0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8],
    [0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2],
    [0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73],
    [0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb],
    [0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79],
    [0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08],
    [0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a],
    [0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e],
    [0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf],
    [0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16],
];

const INV_S_BOX: [[u8; 16]; 16] = [
    [0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb],
    [0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb],
    [0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e],
    [0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25],
    [0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92],
    [0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84],
    [0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06],
    [0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b],
    [0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73],
    [0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e],
    [0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b],
    [0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4],
    [0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f],
    [0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef],
    [0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61],
    [0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d]];

const MIX_COLUMNS_MUL_2: [u8; 256] = [
    0x00, 0x02, 0x04, 0x06, 0x08, 0x0A, 0x0C, 0x0E, 0x10, 0x12, 0x14, 0x16, 0x18, 0x1A, 0x1C, 0x1E,
    0x20, 0x22, 0x24, 0x26, 0x28, 0x2A, 0x2C, 0x2E, 0x30, 0x32, 0x34, 0x36, 0x38, 0x3A, 0x3C, 0x3E,
    0x40, 0x42, 0x44, 0x46, 0x48, 0x4A, 0x4C, 0x4E, 0x50, 0x52, 0x54, 0x56, 0x58, 0x5A, 0x5C, 0x5E,
    0x60, 0x62, 0x64, 0x66, 0x68, 0x6A, 0x6C, 0x6E, 0x70, 0x72, 0x74, 0x76, 0x78, 0x7A, 0x7C, 0x7E,
    0x80, 0x82, 0x84, 0x86, 0x88, 0x8A, 0x8C, 0x8E, 0x90, 0x92, 0x94, 0x96, 0x98, 0x9A, 0x9C, 0x9E,
    0xA0, 0xA2, 0xA4, 0xA6, 0xA8, 0xAA, 0xAC, 0xAE, 0xB0, 0xB2, 0xB4, 0xB6, 0xB8, 0xBA, 0xBC, 0xBE,
    0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE, 0xD0, 0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE,
    0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE, 0xF0, 0xF2, 0xF4, 0xF6, 0xF8, 0xFA, 0xFC, 0xFE,
    0x1B, 0x19, 0x1F, 0x1D, 0x13, 0x11, 0x17, 0x15, 0x0B, 0x09, 0x0F, 0x0D, 0x03, 0x01, 0x07, 0x05,
    0x3B, 0x39, 0x3F, 0x3D, 0x33, 0x31, 0x37, 0x35, 0x2B, 0x29, 0x2F, 0x2D, 0x23, 0x21, 0x27, 0x25,
    0x5B, 0x59, 0x5F, 0x5D, 0x53, 0x51, 0x57, 0x55, 0x4B, 0x49, 0x4F, 0x4D, 0x43, 0x41, 0x47, 0x45,
    0x7B, 0x79, 0x7F, 0x7D, 0x73, 0x71, 0x77, 0x75, 0x6B, 0x69, 0x6F, 0x6D, 0x63, 0x61, 0x67, 0x65,
    0x9B, 0x99, 0x9F, 0x9D, 0x93, 0x91, 0x97, 0x95, 0x8B, 0x89, 0x8F, 0x8D, 0x83, 0x81, 0x87, 0x85,
    0xBB, 0xB9, 0xBF, 0xBD, 0xB3, 0xB1, 0xB7, 0xB5, 0xAB, 0xA9, 0xAF, 0xAD, 0xA3, 0xA1, 0xA7, 0xA5,
    0xDB, 0xD9, 0xDF, 0xDD, 0xD3, 0xD1, 0xD7, 0xD5, 0xCB, 0xC9, 0xCF, 0xCD, 0xC3, 0xC1, 0xC7, 0xC5,
    0xFB, 0xF9, 0xFF, 0xFD, 0xF3, 0xF1, 0xF7, 0xF5, 0xEB, 0xE9, 0xEF, 0xED, 0xE3, 0xE1, 0xE7, 0xE5
];

const MIX_COLUMNS_MUL_3: [u8; 256] = [
    0x00,0x03,0x06,0x05,0x0c,0x0f,0x0a,0x09,0x18,0x1b,0x1e,0x1d,0x14,0x17,0x12,0x11,
    0x30,0x33,0x36,0x35,0x3c,0x3f,0x3a,0x39,0x28,0x2b,0x2e,0x2d,0x24,0x27,0x22,0x21,
    0x60,0x63,0x66,0x65,0x6c,0x6f,0x6a,0x69,0x78,0x7b,0x7e,0x7d,0x74,0x77,0x72,0x71,
    0x50,0x53,0x56,0x55,0x5c,0x5f,0x5a,0x59,0x48,0x4b,0x4e,0x4d,0x44,0x47,0x42,0x41,
    0xc0,0xc3,0xc6,0xc5,0xcc,0xcf,0xca,0xc9,0xd8,0xdb,0xde,0xdd,0xd4,0xd7,0xd2,0xd1,
    0xf0,0xf3,0xf6,0xf5,0xfc,0xff,0xfa,0xf9,0xe8,0xeb,0xee,0xed,0xe4,0xe7,0xe2,0xe1,
    0xa0,0xa3,0xa6,0xa5,0xac,0xaf,0xaa,0xa9,0xb8,0xbb,0xbe,0xbd,0xb4,0xb7,0xb2,0xb1,
    0x90,0x93,0x96,0x95,0x9c,0x9f,0x9a,0x99,0x88,0x8b,0x8e,0x8d,0x84,0x87,0x82,0x81,
    0x9b,0x98,0x9d,0x9e,0x97,0x94,0x91,0x92,0x83,0x80,0x85,0x86,0x8f,0x8c,0x89,0x8a,
    0xab,0xa8,0xad,0xae,0xa7,0xa4,0xa1,0xa2,0xb3,0xb0,0xb5,0xb6,0xbf,0xbc,0xb9,0xba,
    0xfb,0xf8,0xfd,0xfe,0xf7,0xf4,0xf1,0xf2,0xe3,0xe0,0xe5,0xe6,0xef,0xec,0xe9,0xea,
    0xcb,0xc8,0xcd,0xce,0xc7,0xc4,0xc1,0xc2,0xd3,0xd0,0xd5,0xd6,0xdf,0xdc,0xd9,0xda,
    0x5b,0x58,0x5d,0x5e,0x57,0x54,0x51,0x52,0x43,0x40,0x45,0x46,0x4f,0x4c,0x49,0x4a,
    0x6b,0x68,0x6d,0x6e,0x67,0x64,0x61,0x62,0x73,0x70,0x75,0x76,0x7f,0x7c,0x79,0x7a,
    0x3b,0x38,0x3d,0x3e,0x37,0x34,0x31,0x32,0x23,0x20,0x25,0x26,0x2f,0x2c,0x29,0x2a,
    0x0b,0x08,0x0d,0x0e,0x07,0x04,0x01,0x02,0x13,0x10,0x15,0x16,0x1f,0x1c,0x19,0x1a
];

const MIX_COLUMNS_MUL_9: [u8; 256] = [
    0x00,0x09,0x12,0x1b,0x24,0x2d,0x36,0x3f,0x48,0x41,0x5a,0x53,0x6c,0x65,0x7e,0x77,
    0x90,0x99,0x82,0x8b,0xb4,0xbd,0xa6,0xaf,0xd8,0xd1,0xca,0xc3,0xfc,0xf5,0xee,0xe7,
    0x3b,0x32,0x29,0x20,0x1f,0x16,0x0d,0x04,0x73,0x7a,0x61,0x68,0x57,0x5e,0x45,0x4c,
    0xab,0xa2,0xb9,0xb0,0x8f,0x86,0x9d,0x94,0xe3,0xea,0xf1,0xf8,0xc7,0xce,0xd5,0xdc,
    0x76,0x7f,0x64,0x6d,0x52,0x5b,0x40,0x49,0x3e,0x37,0x2c,0x25,0x1a,0x13,0x08,0x01,
    0xe6,0xef,0xf4,0xfd,0xc2,0xcb,0xd0,0xd9,0xae,0xa7,0xbc,0xb5,0x8a,0x83,0x98,0x91,
    0x4d,0x44,0x5f,0x56,0x69,0x60,0x7b,0x72,0x05,0x0c,0x17,0x1e,0x21,0x28,0x33,0x3a,
    0xdd,0xd4,0xcf,0xc6,0xf9,0xf0,0xeb,0xe2,0x95,0x9c,0x87,0x8e,0xb1,0xb8,0xa3,0xaa,
    0xec,0xe5,0xfe,0xf7,0xc8,0xc1,0xda,0xd3,0xa4,0xad,0xb6,0xbf,0x80,0x89,0x92,0x9b,
    0x7c,0x75,0x6e,0x67,0x58,0x51,0x4a,0x43,0x34,0x3d,0x26,0x2f,0x10,0x19,0x02,0x0b,
    0xd7,0xde,0xc5,0xcc,0xf3,0xfa,0xe1,0xe8,0x9f,0x96,0x8d,0x84,0xbb,0xb2,0xa9,0xa0,
    0x47,0x4e,0x55,0x5c,0x63,0x6a,0x71,0x78,0x0f,0x06,0x1d,0x14,0x2b,0x22,0x39,0x30,
    0x9a,0x93,0x88,0x81,0xbe,0xb7,0xac,0xa5,0xd2,0xdb,0xc0,0xc9,0xf6,0xff,0xe4,0xed,
    0x0a,0x03,0x18,0x11,0x2e,0x27,0x3c,0x35,0x42,0x4b,0x50,0x59,0x66,0x6f,0x74,0x7d,
    0xa1,0xa8,0xb3,0xba,0x85,0x8c,0x97,0x9e,0xe9,0xe0,0xfb,0xf2,0xcd,0xc4,0xdf,0xd6,
    0x31,0x38,0x23,0x2a,0x15,0x1c,0x07,0x0e,0x79,0x70,0x6b,0x62,0x5d,0x54,0x4f,0x46
];

const MIX_COLUMNS_MUL_11: [u8; 256] = [
    0x00,0x0b,0x16,0x1d,0x2c,0x27,0x3a,0x31,0x58,0x53,0x4e,0x45,0x74,0x7f,0x62,0x69,
    0xb0,0xbb,0xa6,0xad,0x9c,0x97,0x8a,0x81,0xe8,0xe3,0xfe,0xf5,0xc4,0xcf,0xd2,0xd9,
    0x7b,0x70,0x6d,0x66,0x57,0x5c,0x41,0x4a,0x23,0x28,0x35,0x3e,0x0f,0x04,0x19,0x12,
    0xcb,0xc0,0xdd,0xd6,0xe7,0xec,0xf1,0xfa,0x93,0x98,0x85,0x8e,0xbf,0xb4,0xa9,0xa2,
    0xf6,0xfd,0xe0,0xeb,0xda,0xd1,0xcc,0xc7,0xae,0xa5,0xb8,0xb3,0x82,0x89,0x94,0x9f,
    0x46,0x4d,0x50,0x5b,0x6a,0x61,0x7c,0x77,0x1e,0x15,0x08,0x03,0x32,0x39,0x24,0x2f,
    0x8d,0x86,0x9b,0x90,0xa1,0xaa,0xb7,0xbc,0xd5,0xde,0xc3,0xc8,0xf9,0xf2,0xef,0xe4,
    0x3d,0x36,0x2b,0x20,0x11,0x1a,0x07,0x0c,0x65,0x6e,0x73,0x78,0x49,0x42,0x5f,0x54,
    0xf7,0xfc,0xe1,0xea,0xdb,0xd0,0xcd,0xc6,0xaf,0xa4,0xb9,0xb2,0x83,0x88,0x95,0x9e,
    0x47,0x4c,0x51,0x5a,0x6b,0x60,0x7d,0x76,0x1f,0x14,0x09,0x02,0x33,0x38,0x25,0x2e,
    0x8c,0x87,0x9a,0x91,0xa0,0xab,0xb6,0xbd,0xd4,0xdf,0xc2,0xc9,0xf8,0xf3,0xee,0xe5,
    0x3c,0x37,0x2a,0x21,0x10,0x1b,0x06,0x0d,0x64,0x6f,0x72,0x79,0x48,0x43,0x5e,0x55,
    0x01,0x0a,0x17,0x1c,0x2d,0x26,0x3b,0x30,0x59,0x52,0x4f,0x44,0x75,0x7e,0x63,0x68,
    0xb1,0xba,0xa7,0xac,0x9d,0x96,0x8b,0x80,0xe9,0xe2,0xff,0xf4,0xc5,0xce,0xd3,0xd8,
    0x7a,0x71,0x6c,0x67,0x56,0x5d,0x40,0x4b,0x22,0x29,0x34,0x3f,0x0e,0x05,0x18,0x13,
    0xca,0xc1,0xdc,0xd7,0xe6,0xed,0xf0,0xfb,0x92,0x99,0x84,0x8f,0xbe,0xb5,0xa8,0xa3
];

const MIX_COLUMNS_MUL_13: [u8; 256] = [
    0x00,0x0d,0x1a,0x17,0x34,0x39,0x2e,0x23,0x68,0x65,0x72,0x7f,0x5c,0x51,0x46,0x4b,
    0xd0,0xdd,0xca,0xc7,0xe4,0xe9,0xfe,0xf3,0xb8,0xb5,0xa2,0xaf,0x8c,0x81,0x96,0x9b,
    0xbb,0xb6,0xa1,0xac,0x8f,0x82,0x95,0x98,0xd3,0xde,0xc9,0xc4,0xe7,0xea,0xfd,0xf0,
    0x6b,0x66,0x71,0x7c,0x5f,0x52,0x45,0x48,0x03,0x0e,0x19,0x14,0x37,0x3a,0x2d,0x20,
    0x6d,0x60,0x77,0x7a,0x59,0x54,0x43,0x4e,0x05,0x08,0x1f,0x12,0x31,0x3c,0x2b,0x26,
    0xbd,0xb0,0xa7,0xaa,0x89,0x84,0x93,0x9e,0xd5,0xd8,0xcf,0xc2,0xe1,0xec,0xfb,0xf6,
    0xd6,0xdb,0xcc,0xc1,0xe2,0xef,0xf8,0xf5,0xbe,0xb3,0xa4,0xa9,0x8a,0x87,0x90,0x9d,
    0x06,0x0b,0x1c,0x11,0x32,0x3f,0x28,0x25,0x6e,0x63,0x74,0x79,0x5a,0x57,0x40,0x4d,
    0xda,0xd7,0xc0,0xcd,0xee,0xe3,0xf4,0xf9,0xb2,0xbf,0xa8,0xa5,0x86,0x8b,0x9c,0x91,
    0x0a,0x07,0x10,0x1d,0x3e,0x33,0x24,0x29,0x62,0x6f,0x78,0x75,0x56,0x5b,0x4c,0x41,
    0x61,0x6c,0x7b,0x76,0x55,0x58,0x4f,0x42,0x09,0x04,0x13,0x1e,0x3d,0x30,0x27,0x2a,
    0xb1,0xbc,0xab,0xa6,0x85,0x88,0x9f,0x92,0xd9,0xd4,0xc3,0xce,0xed,0xe0,0xf7,0xfa,
    0xb7,0xba,0xad,0xa0,0x83,0x8e,0x99,0x94,0xdf,0xd2,0xc5,0xc8,0xeb,0xe6,0xf1,0xfc,
    0x67,0x6a,0x7d,0x70,0x53,0x5e,0x49,0x44,0x0f,0x02,0x15,0x18,0x3b,0x36,0x21,0x2c,
    0x0c,0x01,0x16,0x1b,0x38,0x35,0x22,0x2f,0x64,0x69,0x7e,0x73,0x50,0x5d,0x4a,0x47,
    0xdc,0xd1,0xc6,0xcb,0xe8,0xe5,0xf2,0xff,0xb4,0xb9,0xae,0xa3,0x80,0x8d,0x9a,0x97
];

const MIX_COLUMNS_MUL_14: [u8; 256] = [
    0x00,0x0e,0x1c,0x12,0x38,0x36,0x24,0x2a,0x70,0x7e,0x6c,0x62,0x48,0x46,0x54,0x5a,
    0xe0,0xee,0xfc,0xf2,0xd8,0xd6,0xc4,0xca,0x90,0x9e,0x8c,0x82,0xa8,0xa6,0xb4,0xba,
    0xdb,0xd5,0xc7,0xc9,0xe3,0xed,0xff,0xf1,0xab,0xa5,0xb7,0xb9,0x93,0x9d,0x8f,0x81,
    0x3b,0x35,0x27,0x29,0x03,0x0d,0x1f,0x11,0x4b,0x45,0x57,0x59,0x73,0x7d,0x6f,0x61,
    0xad,0xa3,0xb1,0xbf,0x95,0x9b,0x89,0x87,0xdd,0xd3,0xc1,0xcf,0xe5,0xeb,0xf9,0xf7,
    0x4d,0x43,0x51,0x5f,0x75,0x7b,0x69,0x67,0x3d,0x33,0x21,0x2f,0x05,0x0b,0x19,0x17,
    0x76,0x78,0x6a,0x64,0x4e,0x40,0x52,0x5c,0x06,0x08,0x1a,0x14,0x3e,0x30,0x22,0x2c,
    0x96,0x98,0x8a,0x84,0xae,0xa0,0xb2,0xbc,0xe6,0xe8,0xfa,0xf4,0xde,0xd0,0xc2,0xcc,
    0x41,0x4f,0x5d,0x53,0x79,0x77,0x65,0x6b,0x31,0x3f,0x2d,0x23,0x09,0x07,0x15,0x1b,
    0xa1,0xaf,0xbd,0xb3,0x99,0x97,0x85,0x8b,0xd1,0xdf,0xcd,0xc3,0xe9,0xe7,0xf5,0xfb,
    0x9a,0x94,0x86,0x88,0xa2,0xac,0xbe,0xb0,0xea,0xe4,0xf6,0xf8,0xd2,0xdc,0xce,0xc0,
    0x7a,0x74,0x66,0x68,0x42,0x4c,0x5e,0x50,0x0a,0x04,0x16,0x18,0x32,0x3c,0x2e,0x20,
    0xec,0xe2,0xf0,0xfe,0xd4,0xda,0xc8,0xc6,0x9c,0x92,0x80,0x8e,0xa4,0xaa,0xb8,0xb6,
    0x0c,0x02,0x10,0x1e,0x34,0x3a,0x28,0x26,0x7c,0x72,0x60,0x6e,0x44,0x4a,0x58,0x56,
    0x37,0x39,0x2b,0x25,0x0f,0x01,0x13,0x1d,0x47,0x49,0x5b,0x55,0x7f,0x71,0x63,0x6d,
    0xd7,0xd9,0xcb,0xc5,0xef,0xe1,0xf3,0xfd,0xa7,0xa9,0xbb,0xb5,0x9f,0x91,0x83,0x8d
];

const RCON: [u8; 14] = [0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1B, 0x36, 0x6c, 0xd8, 0xab, 0x4d];

#[pyfunction]
pub fn aes256_ctr_encrypt(key: String, message: String) -> PyResult<String> {
    Ok(encrypt_ctr(key.as_str(), message.as_str()))
}

#[pyfunction]
pub fn aes256_ctr_decrypt(key: String, message: String) -> PyResult<String> {
    Ok(decrypt_ctr(key.as_str(), message))
}


fn generate_aes_key() -> [u8; 32] {
    let mut key = [0u8; 32];
    let mut rng = OsRng;
    rng.fill_bytes(&mut key);
    key
}

fn rot_word(word: [u8; 4]) -> [u8; 4] {
    return [word[1], word[2], word[3], word[0]];
}

fn sub_word(mut word: [u8; 4]) -> [u8; 4] {
    for byte in word.iter_mut() {
        *byte = find_s_box_sub(*byte)
    }
    return word;
}

fn xor_words(a: [u8; 4], b: [u8; 4]) -> [u8; 4] {
    [a[0] ^ b[0], a[1] ^ b[1], a[2] ^ b[2], a[3] ^ b[3]]
}

fn key_expansion(key: [u8; 32]) -> [[[u8; 4]; 4]; 15] {
    let mut round_keys = [[0u8; 4]; 60];

    for i in 0..8 {
        round_keys[i] = [key[4 * i], key[4 * i + 1], key[4 * i + 2], key[4 * i + 3]];
    }

    for i in 8..60 {
        let mut temp = round_keys[i];

        if i % 8 == 0 {
            let mut rcon = [0u8; 4];
            rcon[0] = RCON[(i / 8) - 1];
            temp = xor_words(round_keys[i - 8], sub_word(rot_word(round_keys[i - 1])));
            temp[0] = temp[0] ^ RCON[(i / 8) - 1];
            round_keys[i] = temp;
        } else if i % 8 == 4 {
            round_keys[i] = xor_words(sub_word(round_keys[i - 1]), round_keys[i - 8]);
        } else {
            round_keys[i] = xor_words(round_keys[i - 8], round_keys[i - 1]);
        }
    }

    let mut expanded_states = [[[0u8; 4]; 4]; 15];

    expanded_states[0] = [round_keys[0], round_keys[1], round_keys[2], round_keys[3]];

    for i in 1..15 {
        expanded_states[i] = [round_keys[i * 4], round_keys[i * 4 + 1], round_keys[i * 4 + 2], round_keys[i * 4 + 3]];
    }

    return expanded_states;
}


fn find_s_box_sub(byte: u8) -> u8 {
    let row = ((byte >> 4) & 0xF) as usize;
    let col = (byte & 0xF) as usize;
    return S_BOX[row][col];
}

fn find_inv_s_box_sub(byte: u8) -> u8 {
    let row = ((byte >> 4) & 0xF) as usize;
    let col = (byte & 0xF) as usize;
    return INV_S_BOX[row][col];
}

fn sub_bytes(mut state: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    for row in state.iter_mut() {
        for byte in row.iter_mut() {
            *byte = find_s_box_sub(*byte);
        }
    }
    return state;
}

fn inv_sub_bytes(mut state: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    for row in state.iter_mut() {
        for byte in row.iter_mut() {
            *byte = find_inv_s_box_sub(*byte);
        }
    }
    return state;
}

fn rotate_row(mut row: [u8; 4], n: usize) -> [u8; 4] {
    row.rotate_left(n);
    return row;
}

fn _shift_row(mut state: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    for i in 1..4 {
        state[i] = rotate_row(state[i], i);
    }
    return state;
}

fn shift_row(mut state: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    let temp_state = state.clone();

    state[0][1] = temp_state[1][1];
    state[1][1] = temp_state[2][1];
    state[2][1] = temp_state[3][1];
    state[3][1] = temp_state[0][1];

    state[0][2] = temp_state[2][2];
    state[1][2] = temp_state[3][2];
    state[2][2] = temp_state[0][2];
    state[3][2] = temp_state[1][2];

    state[0][3] = temp_state[3][3];
    state[1][3] = temp_state[0][3];
    state[2][3] = temp_state[1][3];
    state[3][3] = temp_state[2][3];

    return state;
}

fn inv_shift_row(mut state: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    let temp_state = state.clone();

    state[0][1] = temp_state[3][1];
    state[1][1] = temp_state[0][1];
    state[2][1] = temp_state[1][1];
    state[3][1] = temp_state[2][1];

    state[0][2] = temp_state[2][2];
    state[1][2] = temp_state[3][2];
    state[2][2] = temp_state[0][2];
    state[3][2] = temp_state[1][2];

    state[0][3] = temp_state[1][3];
    state[1][3] = temp_state[2][3];
    state[2][3] = temp_state[3][3];
    state[3][3] = temp_state[0][3];

    return state;
}


fn print_state(state: [[u8; 4]; 4]) {
    for row in state.iter() {
        println!("{:?}", row);
    }
    println!()
}

fn print_state_hex(state: [[u8; 4]; 4]) {
    for row in state.iter() {
        for &value in row.iter() {
            print!("{:02x} ", value);
        }
        println!();
    }
    println!();
}


fn double_in_galois_field(mut num: u8) -> u8 {
    let msb = num & 0b1000_0000;
    num <<= 1;
    if msb != 0 {
        num = num ^ 0x1B;
    }
    return num;
}


fn mix_column(state: &mut [[u8; 4]; 4]) {
    for i in 0..4 {
        let s0 = state[0][i];
        let s1 = state[1][i];
        let s2 = state[2][i];
        let s3 = state[3][i];

        state[0][i] = MIX_COLUMNS_MUL_2[s0 as usize] ^ MIX_COLUMNS_MUL_3[s1 as usize] ^ s2 ^ s3;
        state[1][i] = s0 ^ MIX_COLUMNS_MUL_2[s1 as usize] ^ MIX_COLUMNS_MUL_3[s2 as usize] ^ s3;
        state[2][i] = s0 ^ s1 ^ MIX_COLUMNS_MUL_2[s2 as usize] ^ MIX_COLUMNS_MUL_3[s3 as usize];
        state[3][i] = MIX_COLUMNS_MUL_3[s0 as usize] ^ s1 ^ s2 ^ MIX_COLUMNS_MUL_2[s3 as usize];
    }
}

fn inv_mix_column(state: &mut [[u8; 4]; 4]) {
    for i in 0..4 {
        let s0 = state[0][i];
        let s1 = state[1][i];
        let s2 = state[2][i];
        let s3 = state[3][i];

        state[0][i] = MIX_COLUMNS_MUL_14[s0 as usize] ^ MIX_COLUMNS_MUL_11[s1 as usize] ^ MIX_COLUMNS_MUL_13[s2 as usize] ^ MIX_COLUMNS_MUL_9[s3 as usize];
        state[1][i] = MIX_COLUMNS_MUL_9[s0 as usize] ^ MIX_COLUMNS_MUL_14[s1 as usize] ^ MIX_COLUMNS_MUL_11[s2 as usize] ^ MIX_COLUMNS_MUL_13[s3 as usize];
        state[2][i] = MIX_COLUMNS_MUL_13[s0 as usize] ^ MIX_COLUMNS_MUL_9[s1 as usize] ^ MIX_COLUMNS_MUL_14[s2 as usize] ^ MIX_COLUMNS_MUL_11[s3 as usize];
        state[3][i] = MIX_COLUMNS_MUL_11[s0 as usize] ^ MIX_COLUMNS_MUL_13[s1 as usize] ^ MIX_COLUMNS_MUL_9[s2 as usize] ^ MIX_COLUMNS_MUL_14[s3 as usize];
    }
}


fn add_round_key(mut state: [[u8; 4]; 4], key: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    for i in 0..4 {
        for j in 0..4 {
            state[i][j] = state[i][j] ^ key[i][j];
        }
    }
    return state;
}

fn decrypt_block(key: [u8; 32], mut state: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    let round_keys: [[[u8; 4]; 4]; 15] = key_expansion(key);

    state = add_round_key(state, round_keys[14]);
    state = inv_shift_row(state);
    state = inv_sub_bytes(state);

    for i in (1..14).rev() {
        state = add_round_key(state, round_keys[i]);
        inv_mix_column(&mut state);
        state = inv_shift_row(state);
        state = inv_sub_bytes(state);
    }

    state = add_round_key(state, round_keys[0]);

    return state;
}

fn encrypt_block(key: [u8; 32], mut state: [[u8; 4]; 4]) -> [[u8; 4]; 4] {
    let round_keys: [[[u8; 4]; 4]; 15] = key_expansion(key);

    state = add_round_key(state, round_keys[0]);

    for i in 1..14 {
        state = sub_bytes(state);
        state = shift_row(state);
        mix_column(&mut state);
        state = add_round_key(state, round_keys[i]);
    }

    state = sub_bytes(state);
    state = shift_row(state);
    state = add_round_key(state, round_keys[14]);

    return state;
}


fn hex_to_vec_matrix(message: String) -> Vec<[[u8; 4]; 4]> {
    let mut message_in_matrix = vec![];
    let num_of_block = message.len() / 32;

    for i in 0..num_of_block {
        let block = &message[i * 32..(i + 1) * 32];
        let mut tmp_matrix = [[0u8; 4]; 4];

        for (j, byte) in block.as_bytes().chunks(2).enumerate() {
            let hex_str = std::str::from_utf8(byte).unwrap_or("00");
            let value = u8::from_str_radix(hex_str, 16).unwrap_or(0);

            tmp_matrix[j / 4][j % 4] = value;

        }

        message_in_matrix.push(tmp_matrix);
    }

    return message_in_matrix;
}

fn encrypt_ctr(key: &str, mut message: &str) -> String {
    let message = pad(message);

    //let message = message.to_string();

    let mut iv = [0u8; 16];
    OsRng.fill_bytes(&mut iv);

    let mut counter: u64 = 0;

    let mut vec_of_state = hex_to_vec_matrix(message);

    let key = hex_key_to_u8_array(key).unwrap();
    let mut ciphertext = vec![];

    let hex_iv: String = iv.iter().map(|b| format!("{:02x}", b)).collect();
    ciphertext.push(hex_iv);

    let mut iv_and_count = [[0u8; 4]; 4];

    for i in 0..2 {
        for j in 0..4 {
            iv_and_count[i][j] = iv[i * 4 + j];
        }
    }

    for block in vec_of_state.iter() {
        let counter_bytes = counter.to_be_bytes();

        for i in 0..2 {
            for j in 0..4 {
                iv_and_count[i + 2][j] = counter_bytes[i * 4 + j];
            }
        }

        let encrypted_block = add_round_key(encrypt_block(key, iv_and_count), *block);

        ciphertext.push(array_to_hex_string(encrypted_block));
        counter += 1;

    }

    return ciphertext.join("");
}

fn encrypt_ecb(key: &str, message: String) -> Vec<String> {
    let mut vec_of_state = hex_to_vec_matrix(message);

    let key = hex_key_to_u8_array(key).unwrap();
    let mut ciphertext = vec![];

    for block in vec_of_state.iter() {
        ciphertext.push(array_to_hex_string(encrypt_block(key, *block)));
    }

    return ciphertext;
}

fn decrypt_ctr(key: &str, ciphertext: String) -> String {

    let iv = &hex_to_bytes_vec(&ciphertext)[0..16];

    let message = ciphertext[32..].to_string();

    let mut counter: u64 = 0;
    let mut plaintext = vec![];
    let key = hex_key_to_u8_array(key).unwrap();
    let mut vec_of_state = hex_to_vec_matrix(message);
    let mut iv_and_count = [[0u8; 4]; 4];

    for i in 0..2 {
        for j in 0..4 {
            iv_and_count[i][j] = iv[i * 4 + j];
        }
    }

    for block in vec_of_state.iter() {
        let counter_bytes = counter.to_be_bytes();

        for i in 0..2 {
            for j in 0..4 {
                iv_and_count[i + 2][j] = counter_bytes[i * 4 + j];
            }
        }

        let encrypted_block = add_round_key(encrypt_block(key, iv_and_count), *block);

        plaintext.push(array_to_hex_string(encrypted_block));
        counter += 1;

    }

    return unpad(plaintext.join("").as_str());
}


fn array_to_hex_string(array: [[u8; 4]; 4]) -> String {
    array
        .iter()
        .flat_map(|row| row.iter())
        .map(|byte| format!("{:02x}", byte))
        .collect()
}


fn utf8_to_u8_array(input: &str) -> Result<[[u8; 4]; 4], String> {
    let bytes = input.as_bytes();

    if bytes.len() > 16 {
        return Err("The UTF-8 string contains more than 16 bytes.".to_string());
    }

    let mut array = [[0u8; 4]; 4];

    for (i, &byte) in bytes.iter().enumerate() {
        array[i / 4][i % 4] = byte;
    }

    Ok(array)
}

fn hex_key_to_u8_array(hex: &str) -> Result<[u8; 32], String> {
    if hex.len() != 64 {
        return Err("the string must be 32 characters.".to_string());
    }

    let mut array = [0u8; 32];
    for (i, chunk) in hex.as_bytes().chunks(2).enumerate() {
        let hex_str = std::str::from_utf8(chunk).map_err(|_| "Non-UTF-8 valid string.".to_string())?;
        array[i] = u8::from_str_radix(hex_str, 16).map_err(|_| format!("Unable to convert '{}' in u8.", hex_str))?;
    }

    Ok(array)
}


fn pad(mut message: &str)-> String {

    let mut message = message.to_string();

    if message.len() % 32 == 0 {
        let num_to_add = 16;
        let pad_value = format!("{:02x}", num_to_add);
        message.push_str(pad_value.repeat(num_to_add).as_str());
    }

    else if message.len() % 32 != 0 {
        let num_to_add = (32 - message.len() % 32) / 2;
        let pad_value = format!("{:02x}", num_to_add);
        message.push_str(pad_value.repeat(num_to_add).as_str());
    }

    return message;
}

fn unpad(message: &str) -> String {
    let message_vec = message;
    let pad_num: String = message_vec.chars().rev().take(2).collect::<Vec<_>>().into_iter().rev().collect();
    let mut pad_num = hex::decode(pad_num).unwrap();
    let pad_num = pad_num[0]  as usize;
    message_vec.chars().take(message_vec.len() - pad_num*2).collect()
}

fn hex_to_bytes_vec(hex: &str) -> Vec<u8> {
    (0..hex.len())
        .step_by(2)
        .map(|i| u8::from_str_radix(&hex[i..i + 2], 16).expect("Hex invalid"))
        .collect()
}

