"use strict";
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lib_2 = require("../lib");
module.exports = {
    'the Permission class is a programming model for iam'(test) {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addActions('sqs:SendMessage');
        p.addActions('dynamodb:CreateTable', 'dynamodb:DeleteTable');
        p.addResources('myQueue');
        p.addResources('yourQueue');
        p.addAllResources();
        p.addAwsAccountPrincipal(`my${core_1.Token.asString({ account: 'account' })}name`);
        p.addAccountCondition('12221121221');
        test.deepEqual(stack.resolve(p.toStatementJson()), { Action: ['sqs:SendMessage',
                'dynamodb:CreateTable',
                'dynamodb:DeleteTable'],
            Resource: ['myQueue', 'yourQueue', '*'],
            Effect: 'Allow',
            Principal: { AWS: { 'Fn::Join': ['',
                        ['arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::my',
                            { account: 'account' },
                            'name:root']] } },
            Condition: { StringEquals: { 'sts:ExternalId': '12221121221' } } });
        test.done();
    },
    'the PolicyDocument class is a dom for iam policy documents'(test) {
        const stack = new core_1.Stack();
        const doc = new lib_1.PolicyDocument();
        const p1 = new lib_1.PolicyStatement();
        p1.addActions('sqs:SendMessage');
        p1.addNotResources('arn:aws:sqs:us-east-1:123456789012:forbidden_queue');
        const p2 = new lib_1.PolicyStatement();
        p2.effect = lib_1.Effect.DENY;
        p2.addActions('cloudformation:CreateStack');
        const p3 = new lib_1.PolicyStatement();
        p3.effect = lib_1.Effect.ALLOW;
        p3.addNotActions('cloudformation:UpdateTerminationProtection');
        doc.addStatements(p1);
        doc.addStatements(p2);
        doc.addStatements(p3);
        test.deepEqual(stack.resolve(doc), {
            Version: '2012-10-17',
            Statement: [{ Effect: 'Allow', Action: 'sqs:SendMessage', NotResource: 'arn:aws:sqs:us-east-1:123456789012:forbidden_queue' },
                { Effect: 'Deny', Action: 'cloudformation:CreateStack' },
                { Effect: 'Allow', NotAction: 'cloudformation:UpdateTerminationProtection' }]
        });
        test.done();
    },
    'Cannot combine Actions and NotActions'(test) {
        test.throws(() => {
            new lib_1.PolicyStatement({
                actions: ['abc'],
                notActions: ['def'],
            });
        }, /Cannot add 'NotActions' to policy statement if 'Actions' have been added/);
        test.done();
    },
    'Cannot combine Resources and NotResources'(test) {
        test.throws(() => {
            new lib_1.PolicyStatement({
                resources: ['abc'],
                notResources: ['def'],
            });
        }, /Cannot add 'NotResources' to policy statement if 'Resources' have been added/);
        test.done();
    },
    'Permission allows specifying multiple actions upon construction'(test) {
        const stack = new core_1.Stack();
        const perm = new lib_1.PolicyStatement();
        perm.addResources('MyResource');
        perm.addActions('Action1', 'Action2', 'Action3');
        test.deepEqual(stack.resolve(perm.toStatementJson()), {
            Effect: 'Allow',
            Action: ['Action1', 'Action2', 'Action3'],
            Resource: 'MyResource'
        });
        test.done();
    },
    'PolicyDoc resolves to undefined if there are no permissions'(test) {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        test.deepEqual(stack.resolve(p), undefined);
        test.done();
    },
    'canonicalUserPrincipal adds a principal to a policy with the passed canonical user id'(test) {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        const canoncialUser = "averysuperduperlongstringfor";
        p.addPrincipals(new lib_1.CanonicalUserPrincipal(canoncialUser));
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: "Allow",
            Principal: {
                CanonicalUser: canoncialUser
            }
        });
        test.done();
    },
    'addAccountRootPrincipal adds a principal with the current account root'(test) {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAccountRootPrincipal();
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: "Allow",
            Principal: {
                AWS: {
                    "Fn::Join": [
                        "",
                        [
                            "arn:",
                            { Ref: "AWS::Partition" },
                            ":iam::",
                            { Ref: "AWS::AccountId" },
                            ":root"
                        ]
                    ]
                }
            }
        });
        test.done();
    },
    'addFederatedPrincipal adds a Federated principal with the passed value'(test) {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addFederatedPrincipal("com.amazon.cognito", { StringEquals: { key: 'value' } });
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: "Allow",
            Principal: {
                Federated: "com.amazon.cognito"
            },
            Condition: {
                StringEquals: { key: 'value' }
            }
        });
        test.done();
    },
    'addAwsAccountPrincipal can be used multiple times'(test) {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAwsAccountPrincipal('1234');
        p.addAwsAccountPrincipal('5678');
        test.deepEqual(stack.resolve(p.toStatementJson()), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::1234:root']] },
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::5678:root']] }
                ]
            }
        });
        test.done();
    },
    'hasResource': {
        'false if there are no resources'(test) {
            test.equal(new lib_1.PolicyStatement().hasResource, false, 'hasResource should be false for an empty permission');
            test.done();
        },
        'true if there is one resource'(test) {
            test.equal(new lib_1.PolicyStatement({ resources: ['one-resource'] }).hasResource, true, 'hasResource is true when there is one resource');
            test.done();
        },
        'true for multiple resources'(test) {
            const p = new lib_1.PolicyStatement();
            p.addResources('r1');
            p.addResources('r2');
            test.equal(p.hasResource, true, 'hasResource is true when there are multiple resource');
            test.done();
        },
    },
    'hasPrincipal': {
        'false if there is no principal'(test) {
            test.equal(new lib_1.PolicyStatement().hasPrincipal, false);
            test.done();
        },
        'true if there is a principal'(test) {
            const p = new lib_1.PolicyStatement();
            p.addArnPrincipal('bla');
            test.equal(p.hasPrincipal, true);
            test.done();
        }
    },
    'statementCount returns the number of statement in the policy document'(test) {
        const p = new lib_1.PolicyDocument();
        test.equal(p.statementCount, 0);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['action1'] }));
        test.equal(p.statementCount, 1);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['action2'] }));
        test.equal(p.statementCount, 2);
        test.done();
    },
    '{ AWS: "*" } principal': {
        'is represented as `Anyone`'(test) {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.Anyone()] }));
            test.deepEqual(stack.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        },
        'is represented as `AnyPrincipal`'(test) {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.AnyPrincipal()] }));
            test.deepEqual(stack.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        },
        'is represented as `addAnyPrincipal`'(test) {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const s = new lib_1.PolicyStatement();
            s.addAnyPrincipal();
            p.addStatements(s);
            test.deepEqual(stack.resolve(p), {
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
            test.done();
        }
    },
    'addResources() will not break a list-encoded Token'(test) {
        const stack = new core_1.Stack();
        const statement = new lib_1.PolicyStatement();
        statement.addActions(...core_1.Lazy.listValue({ produce: () => ['a', 'b', 'c'] }));
        statement.addResources(...core_1.Lazy.listValue({ produce: () => ['x', 'y', 'z'] }));
        test.deepEqual(stack.resolve(statement.toStatementJson()), {
            Effect: 'Allow',
            Action: ['a', 'b', 'c'],
            Resource: ['x', 'y', 'z'],
        });
        test.done();
    },
    'addCanonicalUserPrincipal can be used to add cannonical user principals'(test) {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        const s1 = new lib_1.PolicyStatement();
        s1.addCanonicalUserPrincipal('cannonical-user-1');
        const s2 = new lib_1.PolicyStatement();
        s2.addPrincipals(new lib_1.CanonicalUserPrincipal('cannonical-user-2'));
        p.addStatements(s1);
        p.addStatements(s2);
        test.deepEqual(stack.resolve(p), {
            Statement: [
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-1' } },
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-2' } }
            ],
            Version: '2012-10-17'
        });
        test.done();
    },
    'addPrincipal correctly merges array in'(test) {
        const stack = new core_1.Stack();
        const arrayPrincipal = {
            get grantPrincipal() { return this; },
            assumeRoleAction: 'sts:AssumeRole',
            policyFragment: new lib_2.PrincipalPolicyFragment({ AWS: ['foo', 'bar'] }),
            addToPolicy() { return false; }
        };
        const s = new lib_1.PolicyStatement();
        s.addAccountRootPrincipal();
        s.addPrincipals(arrayPrincipal);
        test.deepEqual(stack.resolve(s.toStatementJson()), {
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] },
                    'foo', 'bar'
                ]
            }
        });
        test.done();
    },
    // https://github.com/aws/aws-cdk/issues/1201
    'policy statements with multiple principal types can be created using multiple addPrincipal calls'(test) {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addArnPrincipal('349494949494');
        s.addServicePrincipal('test.service');
        s.addResources('resource');
        s.addActions('action');
        test.deepEqual(stack.resolve(s.toStatementJson()), {
            Action: 'action',
            Effect: 'Allow',
            Principal: { AWS: '349494949494', Service: 'test.service' },
            Resource: 'resource'
        });
        test.done();
    },
    'Service principals': {
        'regional service principals resolve appropriately'(test) {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com');
            test.deepEqual(stack.resolve(s.toStatementJson()), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'regional service principals resolve appropriately (with user-set region)'(test) {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-northeast-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com', { region: 'cn-north-1' });
            test.deepEqual(stack.resolve(s.toStatementJson()), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
            test.done();
        },
        'obscure service principals resolve to the user-provided value'(test) {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('test.service-principal.dev');
            test.deepEqual(stack.resolve(s.toStatementJson()), {
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'test.service-principal.dev' }
            });
            test.done();
        },
    },
    'CompositePrincipal can be used to represent a principal that has multiple types': {
        'with a single principal'(test) {
            const stack = new core_1.Stack();
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            test.deepEqual(stack.resolve(statement.toStatementJson()), { Effect: 'Allow', Principal: { AWS: 'i:am:an:arn' } });
            test.done();
        },
        'conditions are not allowed on individual principals of a composite'(test) {
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am'));
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('federated', { condition: 1 })), /Components of a CompositePrincipal must not have conditions/);
            test.done();
        },
        'principals and conditions are a big nice merge'(test) {
            const stack = new core_1.Stack();
            // add via ctor
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'), new lib_2.ServicePrincipal('amazon.com'));
            // add via `addPrincipals` (with condition)
            p.addPrincipals(new lib_1.Anyone(), new lib_2.ServicePrincipal('another.service'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            // add via policy statement
            statement.addArnPrincipal('aws-principal-3');
            statement.addCondition('cond2', { boom: 123 });
            test.deepEqual(stack.resolve(statement.toStatementJson()), {
                Condition: {
                    cond2: { boom: 123 }
                },
                Effect: 'Allow',
                Principal: {
                    AWS: ['i:am:an:arn', '*', 'aws-principal-3'],
                    Service: ['amazon.com', 'another.service'],
                }
            });
            test.done();
        },
        'cannot mix types of assumeRoleAction in a single composite'(test) {
            // GIVEN
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('arn')); // assumeRoleAction is "sts:AssumeRule"
            // THEN
            test.throws(() => p.addPrincipals(new lib_2.FederatedPrincipal('fed', {}, 'sts:Boom')), /Cannot add multiple principals with different "assumeRoleAction". Expecting "sts:AssumeRole", got "sts:Boom"/);
            test.done();
        }
    },
    'duplicate statements': {
        'without tokens'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement = new lib_1.PolicyStatement();
            statement.addResources('resource1', 'resource2');
            statement.addActions('action1', 'action2');
            statement.addServicePrincipal('service');
            statement.addConditions({
                a: {
                    b: 'c'
                },
                d: {
                    e: 'f'
                }
            });
            // WHEN
            p.addStatements(statement);
            p.addStatements(statement);
            p.addStatements(statement);
            // THEN
            test.equal(stack.resolve(p).Statement.length, 1);
            test.done();
        },
        'with tokens'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement1 = new lib_1.PolicyStatement();
            statement1.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement1.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            const statement2 = new lib_1.PolicyStatement();
            statement2.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement2.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            // WHEN
            p.addStatements(statement1);
            p.addStatements(statement2);
            // THEN
            test.equal(stack.resolve(p).Statement.length, 1);
            test.done();
        },
    },
    'autoAssignSids enables auto-assignment of a unique SID for each statement'(test) {
        // GIVEN
        const doc = new lib_1.PolicyDocument({
            assignSids: true
        });
        // WHEN
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['action2'], resources: ['resource2'] }));
        // THEN
        const stack = new core_1.Stack();
        test.deepEqual(stack.resolve(doc), {
            Version: '2012-10-17',
            Statement: [
                { Action: 'action1', Effect: 'Allow', Resource: 'resource1', Sid: '0' },
                { Action: 'action2', Effect: 'Allow', Resource: 'resource2', Sid: '1' }
            ],
        });
        test.done();
    },
    'constructor args are equivalent to mutating in-place'(test) {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addActions('action1', 'action2');
        s.addAllResources();
        s.addArnPrincipal('arn');
        s.addCondition('key', { equals: 'value' });
        const doc1 = new lib_1.PolicyDocument();
        doc1.addStatements(s);
        const doc2 = new lib_1.PolicyDocument();
        doc2.addStatements(new lib_1.PolicyStatement({
            actions: ['action1', 'action2'],
            resources: ['*'],
            principals: [new lib_2.ArnPrincipal('arn')],
            conditions: {
                key: { equals: 'value' }
            }
        }));
        test.deepEqual(stack.resolve(doc1), stack.resolve(doc2));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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