"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
function createPolicyWithLogicalId(stack, logicalId) {
    const policy = new lib_1.Policy(stack, logicalId);
    const cfnPolicy = policy.node.defaultChild;
    cfnPolicy.overrideLogicalId(logicalId); // force a particular logical ID
    // add statements & principal to satisfy validation
    policy.addStatements(new lib_1.PolicyStatement({
        actions: ['*'],
        resources: ['*'],
    }));
    policy.attachToRole(new lib_1.Role(stack, 'Role', { assumedBy: new lib_1.AnyPrincipal() }));
}
module.exports = {
    'fails when policy is empty'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synth(), /Policy is empty/);
        test.done();
    },
    'policy with statements'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy', { policyName: 'MyPolicyName' });
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const group = new lib_1.Group(stack, 'MyGroup');
        group.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicyName' } },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' } } });
        test.done();
    },
    'policy name can be omitted, in which case the logical id will be used'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['sqs:SendMessage'] }));
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['arn'], actions: ['sns:Subscribe'] }));
        const user = new lib_1.User(stack, 'MyUser');
        user.attachInlinePolicy(policy);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: 'sqs:SendMessage', Effect: 'Allow', Resource: '*' },
                                { Action: 'sns:Subscribe', Effect: 'Allow', Resource: 'arn' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'policy can be attached users, groups and roles and added permissions via props'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const user1 = new lib_1.User(stack, 'User1');
        const group1 = new lib_1.Group(stack, 'Group1');
        const role1 = new lib_1.Role(stack, 'Role1', {
            assumedBy: new lib_1.ServicePrincipal('test.service')
        });
        new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
            users: [user1],
            groups: [group1],
            roles: [role1],
            statements: [new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:PutItem'] })],
        });
        assert_1.expect(stack).toMatch({ Resources: { User1E278A736: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } },
                MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:PutItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }] } } } });
        test.done();
    },
    'idempotent if a principal (user/group/role) is attached twice'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyPolicy');
        p.addStatements(new lib_1.PolicyStatement({ actions: ['*'], resources: ['*'] }));
        const user = new lib_1.User(stack, 'MyUser');
        p.attachToUser(user);
        p.attachToUser(user);
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' } } });
        test.done();
    },
    'users, groups, roles and permissions can be added using methods'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const p = new lib_1.Policy(stack, 'MyTestPolicy', {
            policyName: 'Foo',
        });
        p.attachToUser(new lib_1.User(stack, 'User1'));
        p.attachToUser(new lib_1.User(stack, 'User2'));
        p.attachToGroup(new lib_1.Group(stack, 'Group1'));
        p.attachToRole(new lib_1.Role(stack, 'Role1', { assumedBy: new lib_1.ServicePrincipal('test.service') }));
        p.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['dynamodb:GetItem'] }));
        assert_1.expect(stack).toMatch({ Resources: { MyTestPolicy316BDB50: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'Group1BEBD4686' }],
                        PolicyDocument: { Statement: [{ Action: 'dynamodb:GetItem', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'Foo',
                        Roles: [{ Ref: 'Role13A5C70C1' }],
                        Users: [{ Ref: 'User1E278A736' }, { Ref: 'User21F1486D1' }] } },
                User1E278A736: { Type: 'AWS::IAM::User' },
                User21F1486D1: { Type: 'AWS::IAM::User' },
                Group1BEBD4686: { Type: 'AWS::IAM::Group' },
                Role13A5C70C1: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'policy can be attached to users, groups or role via methods on the principal'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        const policy = new lib_1.Policy(stack, 'MyPolicy');
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('test.service') });
        user.attachInlinePolicy(policy);
        group.attachInlinePolicy(policy);
        role.attachInlinePolicy(policy);
        policy.addStatements(new lib_1.PolicyStatement({ resources: ['*'], actions: ['*'] }));
        assert_1.expect(stack).toMatch({ Resources: { MyPolicy39D66CF6: { Type: 'AWS::IAM::Policy',
                    Properties: { Groups: [{ Ref: 'MyGroupCBA54B1B' }],
                        PolicyDocument: { Statement: [{ Action: '*', Effect: 'Allow', Resource: '*' }],
                            Version: '2012-10-17' },
                        PolicyName: 'MyPolicy39D66CF6',
                        Roles: [{ Ref: 'MyRoleF48FFE04' }],
                        Users: [{ Ref: 'MyUserDC45028B' }] } },
                MyUserDC45028B: { Type: 'AWS::IAM::User' },
                MyGroupCBA54B1B: { Type: 'AWS::IAM::Group' },
                MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' } } } } });
        test.done();
    },
    'fails if policy name is not unique within a user/group/role'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        // create two policies named Foo and attach them both to the same user/group/role
        const p1 = new lib_1.Policy(stack, 'P1', { policyName: 'Foo' });
        const p2 = new lib_1.Policy(stack, 'P2', { policyName: 'Foo' });
        const p3 = new lib_1.Policy(stack, 'P3'); // uses logicalID as name
        const user = new lib_1.User(stack, 'MyUser');
        const group = new lib_1.Group(stack, 'MyGroup');
        const role = new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        p1.attachToUser(user);
        p1.attachToGroup(group);
        p1.attachToRole(role);
        // try to attach p2 to all of these and expect to fail
        test.throws(() => p2.attachToUser(user), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToGroup(group), /A policy named "Foo" is already attached/);
        test.throws(() => p2.attachToRole(role), /A policy named "Foo" is already attached/);
        p3.attachToUser(user);
        p3.attachToGroup(group);
        p3.attachToRole(role);
        test.done();
    },
    'fails if policy is not attached to a principal'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'MyStack');
        new lib_1.Policy(stack, 'MyPolicy');
        test.throws(() => app.synth(), /Policy must be attached to at least one principal: user, group or role/);
        test.done();
    },
    "generated policy name is the same as the logical id if it's shorter than 128 characters"(test) {
        const stack = new core_1.Stack();
        createPolicyWithLogicalId(stack, 'Foo');
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            "PolicyName": "Foo",
        }));
        test.done();
    },
    'generated policy name only uses the last 128 characters of the logical id'(test) {
        const stack = new core_1.Stack();
        const logicalId128 = 'a' + dup(128 - 2) + 'a';
        const logicalIdOver128 = 'PREFIX' + logicalId128;
        createPolicyWithLogicalId(stack, logicalIdOver128);
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            "PolicyName": logicalId128,
        }));
        test.done();
        function dup(count) {
            let r = '';
            for (let i = 0; i < count; ++i) {
                r += 'x';
            }
            return r;
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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