# pylint: disable=line-too-long, unused-wildcard-import
"""
Application model |dam_v002| is a simplification of |dam_v001|.  While most
functionalities are identical, |dam_v002| does not calculate |RequiredRemoteRelease| on
its own but picks this information from the simulation results of another model.

The following explanations focus on this difference.  For further information on using
|dam_v002|, please read the documentation on model |dam_v001|.

Integration tests
=================

.. how_to_understand_integration_tests::

Each of the following examples repeats one example demonstrating a specific
functionality of application model |dam_v001|.  We define identical parameter values,
initial conditions, and input time series to achieve comparability.   The sequence
|RequiredRemoteRelease| requires special care.  |dam_v001| calculates its values based
on other information, but |dam_v002| expects externally calculated values for it.
Hence, we use the tabulated results of the selected |dam_v001| examples as the input
data of the node object `demand`, which passes this information to |dam_v002| during
simulation.  The limited precision of the copy-pasted |RequiredRemoteRelease| values
causes tiny deviations between the results of both models.

The following time- and space-related setup is identical to the one of |dam_v001|,
except we do not need to add other models to construct meaningful examples:

>>> from hydpy import pub
>>> pub.timegrids = "01.01.2000", "21.01.2000",  "1d"
>>> from hydpy import Node
>>> inflow = Node("inflow")
>>> outflow = Node("outflow")
>>> demand = Node("demand", variable="D")
>>> from hydpy import Element
>>> dam = Element("dam", inlets=inflow, outlets=outflow, receivers=demand)
>>> from hydpy.models.dam_v002 import *
>>> parameterstep("1d")
>>> dam.model = model

We prepare an identical |IntegrationTest| object:

>>> from hydpy import IntegrationTest
>>> test = IntegrationTest(dam)
>>> test.dateformat = "%d.%m."
>>> test.plotting_options.axis1 = fluxes.inflow, fluxes.outflow
>>> test.plotting_options.axis2 = states.watervolume

As initial conditions, |dam_v002| requires logged values for the required remote
release instead of logged values for the total remote discharge and outflow.  Following
the above reasoning, we copy-paste the first value of the "requiredremoterelease"
column that is identical for all drought-related calculations performed by |dam_v001|:

>>> test.inits=((states.watervolume, 0.0),
...             (logs.loggedadjustedevaporation, 0.0),
...             (logs.loggedrequiredremoterelease, 0.005))

For a start, ihe inflow is constant again:

>>> inflow.sequences.sim.series = 1.0

|dam_v002| implements fewer parameters than |dam_v001|.  Besides that, all parameter
settings are identical:

>>> watervolume2waterlevel(PPoly.from_data(xs=[0.0, 1.0], ys=[0.0, 0.25]))
>>> waterlevel2flooddischarge(PPoly.from_data(xs=[0.0], ys=[0.0]))
>>> catchmentarea(86.4)
>>> neardischargeminimumthreshold(0.2)
>>> neardischargeminimumtolerance(0.2)
>>> waterlevelminimumthreshold(0.0)
>>> waterlevelminimumtolerance(0.0)
>>> restricttargetedrelease(True)
>>> surfacearea(1.44)
>>> correctionprecipitation(1.2)
>>> correctionevaporation(1.2)
>>> weightevaporation(0.8)
>>> thresholdevaporation(0.0)
>>> toleranceevaporation(0.001)

.. _dam_v002_smooth_near_minimum:

smooth near minimum
___________________

This example repeats the :ref:`dam_v001_smooth_near_minimum` example of application
model |dam_v001|.  We use the values of |RequiredRemoteRelease| calculated by
|dam_v001|, as explained above:

>>> demand.sequences.sim.series = [
...     0.008588, 0.010053, 0.013858, 0.027322, 0.064075, 0.235523, 0.470414,
...     0.735001, 0.891263, 0.696325, 0.349797, 0.105231, 0.111928, 0.240436,
...     0.229369, 0.058622, 0.016958, 0.008447, 0.004155, 0.0]

Note that the first tabulated value (0.005 m³/s) serves as an initial condition, and we
have to assign the following nineteen values to the time series of the `demand` node.
The last value of the node's time series is of no importance.  We arbitrarily set it to
0.0 m³/s.

The test results confirm that both models behave identically under low flow conditions
for a "near" and a "remote" need for water supply:

.. integration-test::

    >>> test("dam_v002_smooth_near_minimum")
    |   date | waterlevel | precipitation | adjustedprecipitation | potentialevaporation | adjustedevaporation | actualevaporation | inflow | requiredremoterelease | requiredrelease | targetedrelease | actualrelease | flooddischarge |  outflow | watervolume |   demand | inflow |  outflow |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |   0.017055 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |                 0.005 |        0.210526 |        0.210526 |      0.210438 |            0.0 | 0.210438 |    0.068218 | 0.008588 |    1.0 | 0.210438 |
    | 02.01. |   0.034099 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.008588 |         0.21092 |         0.21092 |       0.21092 |            0.0 |  0.21092 |    0.136395 | 0.010053 |    1.0 |  0.21092 |
    | 03.01. |   0.051139 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.010053 |        0.211084 |        0.211084 |      0.211084 |            0.0 | 0.211084 |    0.204557 | 0.013858 |    1.0 | 0.211084 |
    | 04.01. |    0.06817 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.013858 |        0.211523 |        0.211523 |      0.211523 |            0.0 | 0.211523 |    0.272681 | 0.027322 |    1.0 | 0.211523 |
    | 05.01. |   0.085165 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.027322 |        0.213209 |        0.213209 |      0.213209 |            0.0 | 0.213209 |     0.34066 | 0.064075 |    1.0 | 0.213209 |
    | 06.01. |   0.102034 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.064075 |        0.219043 |        0.219043 |      0.219043 |            0.0 | 0.219043 |    0.408135 | 0.235523 |    1.0 | 0.219043 |
    | 07.01. |   0.117512 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.235523 |        0.283419 |        0.283419 |      0.283419 |            0.0 | 0.283419 |    0.470047 | 0.470414 |    1.0 | 0.283419 |
    | 08.01. |   0.128847 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.470414 |        0.475211 |        0.475211 |      0.475211 |            0.0 | 0.475211 |    0.515389 | 0.735001 |    1.0 | 0.475211 |
    | 09.01. |   0.134565 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.735001 |         0.73528 |         0.73528 |       0.73528 |            0.0 |  0.73528 |    0.538261 | 0.891263 |    1.0 |  0.73528 |
    | 10.01. |   0.136913 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.891263 |        0.891314 |        0.891314 |      0.891314 |            0.0 | 0.891314 |    0.547651 | 0.696325 |    1.0 | 0.891314 |
    | 11.01. |   0.143463 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.696325 |         0.69675 |         0.69675 |       0.69675 |            0.0 |  0.69675 |    0.573852 | 0.349797 |    1.0 |  0.69675 |
    | 12.01. |   0.157149 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.349797 |        0.366407 |        0.366407 |      0.366407 |            0.0 | 0.366407 |    0.628595 | 0.105231 |    1.0 | 0.366407 |
    | 13.01. |   0.173819 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.105231 |        0.228241 |        0.228241 |      0.228241 |            0.0 | 0.228241 |    0.695275 | 0.111928 |    1.0 | 0.228241 |
    | 14.01. |   0.190449 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.111928 |        0.230054 |        0.230054 |      0.230054 |            0.0 | 0.230054 |    0.761798 | 0.240436 |    1.0 | 0.230054 |
    | 15.01. |   0.205864 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.240436 |        0.286374 |        0.286374 |      0.286374 |            0.0 | 0.286374 |    0.823455 | 0.229369 |    1.0 | 0.286374 |
    | 16.01. |    0.22142 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.229369 |        0.279807 |        0.279807 |      0.279807 |            0.0 | 0.279807 |     0.88568 | 0.058622 |    1.0 | 0.279807 |
    | 17.01. |    0.23831 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.058622 |         0.21805 |         0.21805 |       0.21805 |            0.0 |  0.21805 |     0.95324 | 0.016958 |    1.0 |  0.21805 |
    | 18.01. |   0.255333 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.016958 |        0.211893 |        0.211893 |      0.211893 |            0.0 | 0.211893 |    1.021333 | 0.008447 |    1.0 | 0.211893 |
    | 19.01. |   0.272378 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.008447 |        0.210904 |        0.210904 |      0.210904 |            0.0 | 0.210904 |    1.089511 | 0.004155 |    1.0 | 0.210904 |
    | 20.01. |   0.289432 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.004155 |        0.210435 |        0.210435 |      0.210435 |            0.0 | 0.210435 |    1.157729 |      0.0 |    1.0 | 0.210435 |

.. _dam_v002_restriction_enabled:

restriction enabled
___________________

This example repeats the :ref:`dam_v001_restriction_enabled` example of application
model |dam_v001|.  We update the time series of the inflow and the required remote
release accordingly:

>>> inflow.sequences.sim.series[10:] = 0.1
>>> demand.sequences.sim.series = [
...     0.008746, 0.010632, 0.015099, 0.03006, 0.068641, 0.242578, 0.474285,
...     0.784512, 0.95036, 0.35, 0.034564, 0.299482, 0.585979, 0.557422,
...     0.229369, 0.142578, 0.068641, 0.029844, 0.012348, 0.0]
>>> neardischargeminimumtolerance(0.0)

The recalculation confirms that the restriction on releasing water when there is little
inflow works as explained for model |dam_v001|:

.. integration-test::

    >>> test("dam_v002_restriction_enabled")
    |   date | waterlevel | precipitation | adjustedprecipitation | potentialevaporation | adjustedevaporation | actualevaporation | inflow | requiredremoterelease | requiredrelease | targetedrelease | actualrelease | flooddischarge |  outflow | watervolume |   demand | inflow |  outflow |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |   0.017282 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |                 0.005 |             0.2 |             0.2 |      0.199917 |            0.0 | 0.199917 |    0.069127 | 0.008746 |    1.0 | 0.199917 |
    | 02.01. |   0.034562 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.008746 |             0.2 |             0.2 |           0.2 |            0.0 |      0.2 |    0.138247 | 0.010632 |    1.0 |      0.2 |
    | 03.01. |   0.051842 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.010632 |             0.2 |             0.2 |           0.2 |            0.0 |      0.2 |    0.207367 | 0.015099 |    1.0 |      0.2 |
    | 04.01. |   0.069122 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.015099 |             0.2 |             0.2 |           0.2 |            0.0 |      0.2 |    0.276487 |  0.03006 |    1.0 |      0.2 |
    | 05.01. |   0.086402 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |               0.03006 |             0.2 |             0.2 |           0.2 |            0.0 |      0.2 |    0.345607 | 0.068641 |    1.0 |      0.2 |
    | 06.01. |   0.103682 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.068641 |             0.2 |             0.2 |           0.2 |            0.0 |      0.2 |    0.414727 | 0.242578 |    1.0 |      0.2 |
    | 07.01. |   0.120042 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.242578 |        0.242578 |        0.242578 |      0.242578 |            0.0 | 0.242578 |    0.480168 | 0.474285 |    1.0 | 0.242578 |
    | 08.01. |   0.131398 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.474285 |        0.474285 |        0.474285 |      0.474285 |            0.0 | 0.474285 |     0.52559 | 0.784512 |    1.0 | 0.474285 |
    | 09.01. |   0.136052 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |              0.784512 |        0.784512 |        0.784512 |      0.784512 |            0.0 | 0.784512 |    0.544208 |  0.95036 |    1.0 | 0.784512 |
    | 10.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |               0.95036 |         0.95036 |         0.95036 |       0.95036 |            0.0 |  0.95036 |    0.548497 |     0.35 |    1.0 |  0.95036 |
    | 11.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |                  0.35 |            0.35 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.034564 |    0.1 |      0.1 |
    | 12.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.034564 |             0.2 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.299482 |    0.1 |      0.1 |
    | 13.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.299482 |        0.299482 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.585979 |    0.1 |      0.1 |
    | 14.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.585979 |        0.585979 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.557422 |    0.1 |      0.1 |
    | 15.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.557422 |        0.557422 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.229369 |    0.1 |      0.1 |
    | 16.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.229369 |        0.229369 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.142578 |    0.1 |      0.1 |
    | 17.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.142578 |             0.2 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.068641 |    0.1 |      0.1 |
    | 18.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.068641 |             0.2 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.029844 |    0.1 |      0.1 |
    | 19.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.029844 |             0.2 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 | 0.012348 |    0.1 |      0.1 |
    | 20.01. |   0.137124 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.1 |              0.012348 |             0.2 |             0.1 |           0.1 |            0.0 |      0.1 |    0.548497 |      0.0 |    0.1 |      0.1 |

.. _dam_v002_smooth_stage_minimum:

smooth stage minimum
____________________

This example repeats the :ref:`dam_v001_smooth_stage_minimum` example of application
model |dam_v001|.  We update parameters |NearDischargeMinimumThreshold|,
|WaterLevelMinimumThreshold|, and |WaterLevelMinimumTolerance|, as well as the time
series of the inflow and the required remote release, accordingly:

>>> inflow.sequences.sim.series = numpy.linspace(0.2, 0.0, 20)
>>> neardischargeminimumthreshold(0.0)
>>> waterlevelminimumthreshold(0.005)
>>> waterlevelminimumtolerance(0.01)
>>> demand.sequences.sim.series = [
...     0.01232, 0.029323, 0.064084, 0.120198, 0.247367, 0.45567, 0.608464,
...     0.537314, 0.629775, 0.744091, 0.82219, 0.841916, 0.701812, 0.533258,
...     0.351863, 0.185207, 0.107697, 0.055458, 0.025948, 0.0]

|dam_v002| deals with limited water availability, as already known from |dam_v001|:

.. integration-test::

    >>> test("dam_v002_smooth_stage_minimum")
    |   date | waterlevel | precipitation | adjustedprecipitation | potentialevaporation | adjustedevaporation | actualevaporation |   inflow | requiredremoterelease | requiredrelease | targetedrelease | actualrelease | flooddischarge |  outflow | watervolume |   demand |   inflow |  outflow |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |   0.004295 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |      0.2 |                 0.005 |           0.005 |           0.005 |      0.001137 |            0.0 | 0.001137 |    0.017182 |  0.01232 |      0.2 | 0.001137 |
    | 02.01. |    0.00822 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.189474 |               0.01232 |         0.01232 |         0.01232 |      0.007802 |            0.0 | 0.007802 |    0.032878 | 0.029323 | 0.189474 | 0.007802 |
    | 03.01. |   0.011517 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.178947 |              0.029323 |        0.029323 |        0.029323 |      0.026306 |            0.0 | 0.026306 |    0.046066 | 0.064084 | 0.178947 | 0.026306 |
    | 04.01. |   0.013812 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.168421 |              0.064084 |        0.064084 |        0.064084 |      0.062169 |            0.0 | 0.062169 |    0.055247 | 0.120198 | 0.168421 | 0.062169 |
    | 05.01. |   0.014663 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.157895 |              0.120198 |        0.120198 |        0.120198 |      0.118489 |            0.0 | 0.118489 |    0.058651 | 0.247367 | 0.157895 | 0.118489 |
    | 06.01. |   0.012606 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.147368 |              0.247367 |        0.247367 |        0.247367 |      0.242594 |            0.0 | 0.242594 |    0.050424 |  0.45567 | 0.147368 | 0.242594 |
    | 07.01. |   0.006991 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.136842 |               0.45567 |         0.45567 |         0.45567 |      0.396785 |            0.0 | 0.396785 |    0.027965 | 0.608464 | 0.136842 | 0.396785 |
    | 08.01. |   0.003444 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.126316 |              0.608464 |        0.608464 |        0.608464 |      0.290539 |            0.0 | 0.290539 |    0.013776 | 0.537314 | 0.126316 | 0.290539 |
    | 09.01. |   0.002654 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.115789 |              0.537314 |        0.537314 |        0.537314 |      0.152381 |            0.0 | 0.152381 |    0.010614 | 0.629775 | 0.115789 | 0.152381 |
    | 10.01. |   0.001941 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.105263 |              0.629775 |        0.629775 |        0.629775 |      0.138257 |            0.0 | 0.138257 |    0.007764 | 0.744091 | 0.105263 | 0.138257 |
    | 11.01. |   0.001258 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.094737 |              0.744091 |        0.744091 |        0.744091 |      0.126361 |            0.0 | 0.126361 |    0.005031 |  0.82219 | 0.094737 | 0.126361 |
    | 12.01. |   0.000698 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.084211 |               0.82219 |         0.82219 |         0.82219 |      0.110141 |            0.0 | 0.110141 |    0.002791 | 0.841916 | 0.084211 | 0.110141 |
    | 13.01. |   0.000279 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.073684 |              0.841916 |        0.841916 |        0.841916 |      0.093088 |            0.0 | 0.093088 |    0.001114 | 0.701812 | 0.073684 | 0.093088 |
    | 14.01. |   0.000138 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.063158 |              0.701812 |        0.701812 |        0.701812 |      0.069675 |            0.0 | 0.069675 |    0.000551 | 0.533258 | 0.063158 | 0.069675 |
    | 15.01. |   0.000156 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.052632 |              0.533258 |        0.533258 |        0.533258 |      0.051783 |            0.0 | 0.051783 |    0.000625 | 0.351863 | 0.052632 | 0.051783 |
    | 16.01. |   0.000301 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.042105 |              0.351863 |        0.351863 |        0.351863 |      0.035397 |            0.0 | 0.035397 |    0.001204 | 0.185207 | 0.042105 | 0.035397 |
    | 17.01. |   0.000547 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.031579 |              0.185207 |        0.185207 |        0.185207 |       0.02019 |            0.0 |  0.02019 |    0.002188 | 0.107697 | 0.031579 |  0.02019 |
    | 18.01. |   0.000726 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.021053 |              0.107697 |        0.107697 |        0.107697 |       0.01279 |            0.0 |  0.01279 |    0.002902 | 0.055458 | 0.021053 |  0.01279 |
    | 19.01. |   0.000803 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 | 0.010526 |              0.055458 |        0.055458 |        0.055458 |      0.006931 |            0.0 | 0.006931 |    0.003213 | 0.025948 | 0.010526 | 0.006931 |
    | 20.01. |   0.000768 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |      0.0 |              0.025948 |        0.025948 |        0.012974 |      0.001635 |            0.0 | 0.001635 |    0.003072 |      0.0 |      0.0 | 0.001635 |

.. _dam_v002_evaporation:

evaporation
___________

This example repeats the :ref:`dam_v001_evaporation` example of application model
|dam_v001|.  We add an |evap_ret_io| submodel and update the time series of the
required remote release time series accordingly:

>>> with model.add_pemodel_v1("evap_ret_io") as pemodel:
...     evapotranspirationfactor(1.0)
>>> pemodel.prepare_inputseries()
>>> pemodel.sequences.inputs.referenceevapotranspiration.series = 10 * [1.0] + 10 * [5.0]
>>> demand.sequences.sim.series = [
...     0.012321, 0.029352, 0.064305, 0.120897, 0.248435, 0.453671, 0.585089,
...     0.550583, 0.694398, 0.784979, 0.81852, 0.840207, 0.72592, 0.575373,
...     0.386003, 0.198088, 0.113577, 0.05798, 0.026921, 0.0]

|dam_v002| uses the evaporation values discussed for |dam_v001|:

.. integration-test::

    >>> test("dam_v002_evaporation")
    |   date | waterlevel | precipitation | adjustedprecipitation | potentialevaporation | adjustedevaporation | actualevaporation |   inflow | requiredremoterelease | requiredrelease | targetedrelease | actualrelease | flooddischarge |  outflow | watervolume |   demand |   inflow |  outflow |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |   0.003964 |           0.0 |                   0.0 |                  1.0 |               0.016 |          0.015403 |      0.2 |                 0.005 |           0.005 |           0.005 |      0.001069 |            0.0 | 0.001069 |    0.015857 | 0.012321 |      0.2 | 0.001069 |
    | 02.01. |   0.007488 |           0.0 |                   0.0 |                  1.0 |              0.0192 |            0.0192 | 0.189474 |              0.012321 |        0.012321 |        0.012321 |      0.007143 |            0.0 | 0.007143 |    0.029951 | 0.029352 | 0.189474 | 0.007143 |
    | 03.01. |   0.010384 |           0.0 |                   0.0 |                  1.0 |             0.01984 |           0.01984 | 0.178947 |              0.029352 |        0.029352 |        0.029352 |      0.025047 |            0.0 | 0.025047 |    0.041534 | 0.064305 | 0.178947 | 0.025047 |
    | 04.01. |   0.012275 |           0.0 |                   0.0 |                  1.0 |            0.019968 |          0.019968 | 0.168421 |              0.064305 |        0.064305 |        0.064305 |      0.060896 |            0.0 | 0.060896 |    0.049099 | 0.120897 | 0.168421 | 0.060896 |
    | 05.01. |   0.012723 |           0.0 |                   0.0 |                  1.0 |            0.019994 |          0.019994 | 0.157895 |              0.120897 |        0.120897 |        0.120897 |      0.117158 |            0.0 | 0.117158 |    0.050891 | 0.248435 | 0.157895 | 0.117158 |
    | 06.01. |   0.010373 |           0.0 |                   0.0 |                  1.0 |            0.019999 |          0.019999 | 0.147368 |              0.248435 |        0.248435 |        0.248435 |      0.236137 |            0.0 | 0.236137 |    0.041494 | 0.453671 | 0.147368 | 0.236137 |
    | 07.01. |   0.005529 |           0.0 |                   0.0 |                  1.0 |                0.02 |              0.02 | 0.136842 |              0.453671 |        0.453671 |        0.453671 |      0.341126 |            0.0 | 0.341126 |    0.022116 | 0.585089 | 0.136842 | 0.341126 |
    | 08.01. |   0.002955 |           0.0 |                   0.0 |                  1.0 |                0.02 |              0.02 | 0.126316 |              0.585089 |        0.585089 |        0.585089 |      0.225476 |            0.0 | 0.225476 |     0.01182 | 0.550583 | 0.126316 | 0.225476 |
    | 09.01. |   0.002162 |           0.0 |                   0.0 |                  1.0 |                0.02 |              0.02 | 0.115789 |              0.550583 |        0.550583 |        0.550583 |      0.132505 |            0.0 | 0.132505 |    0.008648 | 0.694398 | 0.115789 | 0.132505 |
    | 10.01. |   0.001323 |           0.0 |                   0.0 |                  1.0 |                0.02 |          0.019985 | 0.105263 |              0.694398 |        0.694398 |        0.694398 |      0.124104 |            0.0 | 0.124104 |    0.005294 | 0.784979 | 0.105263 | 0.124104 |
    | 11.01. |  -0.000008 |           0.0 |                   0.0 |                  5.0 |               0.084 |          0.067874 | 0.094737 |              0.784979 |        0.784979 |        0.784979 |      0.088493 |            0.0 | 0.088493 |   -0.000031 |  0.81852 | 0.094737 | 0.088493 |
    | 12.01. |  -0.000276 |           0.0 |                   0.0 |                  5.0 |              0.0968 |          0.027904 | 0.084211 |               0.81852 |         0.81852 |         0.81852 |      0.068717 |            0.0 | 0.068717 |   -0.001103 | 0.840207 | 0.084211 | 0.068717 |
    | 13.01. |  -0.000428 |           0.0 |                   0.0 |                  5.0 |             0.09936 |          0.015203 | 0.073684 |              0.840207 |        0.840207 |        0.840207 |      0.065534 |            0.0 | 0.065534 |   -0.001713 |  0.72592 | 0.073684 | 0.065534 |
    | 14.01. |  -0.000479 |           0.0 |                   0.0 |                  5.0 |            0.099872 |           0.01084 | 0.063158 |               0.72592 |         0.72592 |         0.72592 |      0.054647 |            0.0 | 0.054647 |   -0.001914 | 0.575373 | 0.063158 | 0.054647 |
    | 15.01. |  -0.000483 |           0.0 |                   0.0 |                  5.0 |            0.099974 |          0.009906 | 0.052632 |              0.575373 |        0.575373 |        0.575373 |      0.042914 |            0.0 | 0.042914 |    -0.00193 | 0.386003 | 0.052632 | 0.042914 |
    | 16.01. |  -0.000438 |           0.0 |                   0.0 |                  5.0 |            0.099995 |          0.010953 | 0.042105 |              0.386003 |        0.386003 |        0.386003 |      0.029087 |            0.0 | 0.029087 |   -0.001752 | 0.198088 | 0.042105 | 0.029087 |
    | 17.01. |   -0.00038 |           0.0 |                   0.0 |                  5.0 |            0.099999 |          0.013614 | 0.031579 |              0.198088 |        0.198088 |        0.198088 |      0.015271 |            0.0 | 0.015271 |   -0.001519 | 0.113577 | 0.031579 | 0.015271 |
    | 18.01. |  -0.000412 |           0.0 |                   0.0 |                  5.0 |                 0.1 |          0.013785 | 0.021053 |              0.113577 |        0.113577 |        0.113577 |      0.008769 |            0.0 | 0.008769 |   -0.001649 |  0.05798 | 0.021053 | 0.008769 |
    | 19.01. |  -0.000506 |           0.0 |                   0.0 |                  5.0 |                 0.1 |          0.010522 | 0.010526 |               0.05798 |         0.05798 |         0.05798 |      0.004349 |            0.0 | 0.004349 |   -0.002024 | 0.026921 | 0.010526 | 0.004349 |
    | 20.01. |  -0.000662 |           0.0 |                   0.0 |                  5.0 |                 0.1 |          0.006257 |      0.0 |              0.026921 |        0.026921 |        0.013461 |      0.000957 |            0.0 | 0.000957 |   -0.002648 |      0.0 |      0.0 | 0.000957 |

>>> del model.pemodel

.. _dam_v002_flood_retention:

flood retention
_______________

This example repeats the :ref:`dam_v001_flood_retention` example of application model
|dam_v001|.  We use the same parameter and input time series configuration:

>>> neardischargeminimumthreshold(0.0)
>>> neardischargeminimumtolerance(0.0)
>>> waterlevelminimumthreshold(0.0)
>>> waterlevelminimumtolerance(0.0)
>>> waterlevel2flooddischarge(PPoly.from_data(xs=[0.0, 1.0], ys=[0.0, 2.5]))
>>> neardischargeminimumthreshold(0.0)
>>> with model.add_precipmodel_v2("meteo_precip_io") as precipmodel:
...     precipitationfactor(1.0)
>>> precipmodel.prepare_inputseries()
>>> precipmodel.sequences.inputs.precipitation.series = [
...     0.0, 50.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
>>> inflow.sequences.sim.series = [0.0, 0.0, 5.0, 9.0, 8.0, 5.0, 3.0, 2.0, 1.0, 0.0,
...                                0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
>>> demand.sequences.sim.series = 0.0
>>> test.inits.loggedrequiredremoterelease = 0.0

The recalculation results confirm the equality of both models for high-flow conditions:

.. integration-test::

    >>> test("dam_v002_flood_retention")
    |   date | waterlevel | precipitation | adjustedprecipitation | potentialevaporation | adjustedevaporation | actualevaporation | inflow | requiredremoterelease | requiredrelease | targetedrelease | actualrelease | flooddischarge |  outflow | watervolume | demand | inflow |  outflow |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. |        0.0 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |            0.0 |      0.0 |         0.0 |    0.0 |    0.0 |      0.0 |
    | 02.01. |   0.021027 |          50.0 |                   1.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.026521 | 0.026521 |    0.084109 |    0.0 |    0.0 | 0.026521 |
    | 03.01. |   0.125058 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    5.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.183777 | 0.183777 |     0.50023 |    0.0 |    5.0 | 0.183777 |
    | 04.01. |   0.307728 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    9.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.543038 | 0.543038 |    1.230912 |    0.0 |    9.0 | 0.543038 |
    | 05.01. |   0.459769 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    8.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.961082 | 0.961082 |    1.839074 |    0.0 |    8.0 | 0.961082 |
    | 06.01. |   0.540735 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    5.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.251541 | 1.251541 |    2.162941 |    0.0 |    5.0 | 1.251541 |
    | 07.01. |   0.575391 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    3.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.395548 | 1.395548 |    2.301566 |    0.0 |    3.0 | 1.395548 |
    | 08.01. |   0.587199 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    2.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.453371 | 1.453371 |    2.348795 |    0.0 |    2.0 | 1.453371 |
    | 09.01. |   0.577358 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    1.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.455585 | 1.455585 |    2.309432 |    0.0 |    1.0 | 1.455585 |
    | 10.01. |   0.547008 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.405115 | 1.405115 |     2.18803 |    0.0 |    0.0 | 1.405115 |
    | 11.01. |   0.518253 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.331251 | 1.331251 |     2.07301 |    0.0 |    0.0 | 1.331251 |
    | 12.01. |   0.491009 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |        1.26127 |  1.26127 |    1.964036 |    0.0 |    0.0 |  1.26127 |
    | 13.01. |   0.465198 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.194968 | 1.194968 |    1.860791 |    0.0 |    0.0 | 1.194968 |
    | 14.01. |   0.440743 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.132151 | 1.132151 |    1.762973 |    0.0 |    0.0 | 1.132151 |
    | 15.01. |   0.417574 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       1.072636 | 1.072636 |    1.670297 |    0.0 |    0.0 | 1.072636 |
    | 16.01. |   0.395623 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |        1.01625 |  1.01625 |    1.582493 |    0.0 |    0.0 |  1.01625 |
    | 17.01. |   0.374826 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.962828 | 0.962828 |    1.499305 |    0.0 |    0.0 | 0.962828 |
    | 18.01. |   0.355122 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.912214 | 0.912214 |     1.42049 |    0.0 |    0.0 | 0.912214 |
    | 19.01. |   0.336454 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.864261 | 0.864261 |    1.345818 |    0.0 |    0.0 | 0.864261 |
    | 20.01. |   0.318768 |           0.0 |                   0.0 |                  0.0 |                 0.0 |               0.0 |    0.0 |                   0.0 |             0.0 |             0.0 |           0.0 |       0.818829 | 0.818829 |    1.275071 |    0.0 |    0.0 | 0.818829 |
"""

# import...
# ...from HydPy
from hydpy.auxs.anntools import ANN  # pylint: disable=unused-import
from hydpy.auxs.ppolytools import Poly, PPoly  # pylint: disable=unused-import
from hydpy.core import modeltools
from hydpy.exe.modelimports import *
from hydpy.interfaces import petinterfaces
from hydpy.interfaces import precipinterfaces

# ...from dam
from hydpy.models.dam import dam_model
from hydpy.models.dam import dam_solver


class Model(
    modeltools.ELSModel, dam_model.Main_PrecipModel_V2, dam_model.Main_PEModel_V1
):
    """|dam_v002.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(short="Dam-V2", description="dam model, version 2")
    __HYDPY_ROOTMODEL__ = True

    SOLVERPARAMETERS = (
        dam_solver.AbsErrorMax,
        dam_solver.RelErrorMax,
        dam_solver.RelDTMin,
        dam_solver.RelDTMax,
    )
    SOLVERSEQUENCES = ()
    INLET_METHODS = (
        dam_model.Calc_Precipitation_V1,
        dam_model.Calc_PotentialEvaporation_V1,
        dam_model.Calc_AdjustedEvaporation_V1,
        dam_model.Pick_Inflow_V1,
        dam_model.Calc_RequiredRemoteRelease_V2,
        dam_model.Calc_RequiredRelease_V1,
        dam_model.Calc_TargetedRelease_V1,
    )
    OBSERVER_METHODS = ()
    RECEIVER_METHODS = (dam_model.Pick_LoggedRequiredRemoteRelease_V1,)
    ADD_METHODS = ()
    PART_ODE_METHODS = (
        dam_model.Calc_AdjustedPrecipitation_V1,
        dam_model.Pick_Inflow_V1,
        dam_model.Calc_WaterLevel_V1,
        dam_model.Calc_ActualEvaporation_V1,
        dam_model.Calc_ActualRelease_V1,
        dam_model.Calc_FloodDischarge_V1,
        dam_model.Calc_Outflow_V1,
    )
    FULL_ODE_METHODS = (dam_model.Update_WaterVolume_V1,)
    OUTLET_METHODS = (dam_model.Calc_WaterLevel_V1, dam_model.Pass_Outflow_V1)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (precipinterfaces.PrecipModel_V2, petinterfaces.PETModel_V1)
    SUBMODELS = ()

    precipmodel = modeltools.SubmodelProperty(
        precipinterfaces.PrecipModel_V2, optional=True
    )
    pemodel = modeltools.SubmodelProperty(petinterfaces.PETModel_V1, optional=True)


tester = Tester()
cythonizer = Cythonizer()
